# OSDU Troubleshooting Guide

Common issues and solutions when managing OSDU entitlements.

## Authentication Issues

### 401 Unauthorized - Invalid Token

**Error:**
```
HTTP 401: Unauthorized
{"error": "invalid_token", "error_description": "Token has expired"}
```

**Causes:**
- Client secret expired
- Wrong tenant ID
- Invalid client ID/secret pair

**Solutions:**

1. Verify credentials:
   ```bash
   # Test token acquisition
   curl -s -X POST \
     "https://login.microsoftonline.com/${AI_OSDU_TENANT_ID}/oauth2/token" \
     -d "grant_type=client_credentials" \
     -d "client_id=${AI_OSDU_CLIENT}" \
     -d "client_secret=${AI_OSDU_SECRET}" \
     -d "resource=${AI_OSDU_CLIENT}" \
     | jq '.access_token // .error_description'
   ```

2. Check secret expiration in Azure Portal:
   - Navigate to App Registration > Certificates & secrets
   - Verify secret hasn't expired
   - Create new secret if needed

3. Verify environment variables match Azure configuration

### AADSTS700016 - Application Not Found

**Error:**
```
AADSTS700016: Application with identifier 'xxx' was not found in directory
```

**Solution:**
- Verify `AI_OSDU_CLIENT` matches the app registration client ID
- Confirm the app registration exists in the correct tenant (`AI_OSDU_TENANT_ID`)

### AADSTS7000215 - Invalid Client Secret

**Error:**
```
AADSTS7000215: Invalid client secret provided
```

**Solution:**
- Regenerate the client secret in Azure Portal
- Update `AI_OSDU_SECRET` environment variable
- Ensure no trailing whitespace in the secret value

## Permission Issues

### 403 Forbidden - Insufficient Permissions

**Error:**
```
HTTP 403: Forbidden
{"error": "access_denied", "message": "Insufficient permissions"}
```

**Causes:**
- App registration lacks required OSDU API permissions
- Data partition access not granted

**Solutions:**

1. Add API permissions in OSDU:
   - Navigate to OSDU instance > Settings > API permissions
   - Grant the app registration access to Entitlements API

2. Required permissions:

   | Operation | Required Permission |
   |-----------|---------------------|
   | List members | `Entitlements.Read` |
   | Add/remove members | `Entitlements.Write` |
   | Create groups | `Entitlements.Admin` |

3. Verify data partition access:
   ```bash
   # Check partition is accessible
   uv run .claude/skills/azure-osdu/scripts/osdu_manage.py check
   ```

### 403 on Specific Groups

**Error:**
```
HTTP 403 when accessing users.data.root@partition.domain
```

**Solution:**
- Root group requires elevated permissions
- Your app registration may not have admin access
- Use Admin role instead of Root when possible

## GUID Resolution Issues

### GUIDs Not Resolving to Names

**Symptoms:**
- Output shows raw GUIDs instead of display names
- `display_name` is `null` in JSON output

**Causes:**
1. Azure CLI not authenticated
2. Insufficient Azure AD permissions
3. GUID doesn't exist in Azure AD (external identity)

**Solutions:**

1. Authenticate Azure CLI:
   ```bash
   az login
   az account show
   ```

2. Check Azure AD permissions:
   ```bash
   # Test user lookup
   az ad user show --id "YOUR_GUID" --query "displayName" -o tsv
   ```

3. Skip resolution for performance:
   ```bash
   uv run .claude/skills/azure-osdu/scripts/osdu_users.py --no-resolve
   ```

### Slow GUID Resolution

**Symptoms:**
- Script takes long time with many users
- Console shows "Resolving identities..."

**Causes:**
- Many unique GUIDs to look up
- Network latency to Azure AD
- No caching between runs

**Solutions:**

1. Use `--no-resolve` for initial listing
2. Resolve only specific users:
   ```bash
   # Find specific user with resolution
   uv run .claude/skills/azure-osdu/scripts/osdu_users.py --user "GUID"
   ```

3. Export JSON and resolve later:
   ```bash
   uv run .claude/skills/azure-osdu/scripts/osdu_users.py --json --no-resolve > users.json
   ```

### Service Principal or App GUIDs

**Symptom:**
- Display name prefixed with `[SP]` or `[App]`

**Explanation:**
- `[SP]` = Service Principal (managed identity or app)
- `[App]` = Application registration

**Not an error** - these are valid OSDU users representing applications rather than human users.

## API Errors

### 404 Not Found

**Error:**
```
HTTP 404: Group not found
```

**Causes:**
- Wrong group email format
- Incorrect partition name
- Group doesn't exist

**Solutions:**

1. Verify group format:
   ```
   users.datalake.{role}@{partition}.{domain}
   ```

2. Check partition name:
   ```bash
   echo "Partition: ${AI_OSDU_DATA_PARTITION}"
   ```

3. List available groups:
   ```bash
   curl -s -X GET \
     "https://${AI_OSDU_HOST}/api/entitlements/v2/groups" \
     -H "Authorization: Bearer ${TOKEN}" \
     -H "data-partition-id: ${AI_OSDU_DATA_PARTITION}" \
     | jq '.groups[].email'
   ```

### 409 Conflict

**Error:**
```
HTTP 409: Member already exists
```

**Explanation:**
- User is already a member of the group
- This is informational, not an error

**Behavior:**
- Script reports user already exists
- No action needed

### 400 Bad Request

**Error:**
```
HTTP 400: Invalid email format
```

**Causes:**
- Malformed email address
- Invalid characters in email

**Solutions:**

1. Verify email format:
   - Must be valid email: `user@domain.com`
   - GUIDs are also valid: `xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx`

2. Check for trailing whitespace or special characters

## Configuration Issues

### Missing Environment Variables

**Error:**
```
Missing required environment variables: AI_OSDU_HOST, AI_OSDU_SECRET
```

**Solution:**

1. Set all required variables:
   ```bash
   export AI_OSDU_HOST="your-osdu.energy.azure.com"
   export AI_OSDU_DATA_PARTITION="opendes"
   export AI_OSDU_CLIENT="your-client-id"
   export AI_OSDU_SECRET="your-secret"
   export AI_OSDU_TENANT_ID="your-tenant-id"
   ```

2. Verify with check command:
   ```bash
   uv run .claude/skills/azure-osdu/scripts/osdu_manage.py check
   ```

### Wrong OSDU Host

**Symptoms:**
- Connection timeout
- SSL/TLS errors
- DNS resolution failures

**Solution:**
- Verify hostname format: `osdu-instance.energy.azure.com`
- Don't include `https://` prefix
- Check if OSDU instance is accessible from your network

## Debugging

### Enable Verbose Output

Scripts output progress to stderr, data to stdout:
```bash
# See progress messages
uv run .claude/skills/azure-osdu/scripts/osdu_users.py 2>&1

# Silent mode for automation
uv run .claude/skills/azure-osdu/scripts/osdu_users.py --json 2>/dev/null
```

### Test Token Manually

```bash
# Get token
TOKEN=$(curl -s -X POST \
  "https://login.microsoftonline.com/${AI_OSDU_TENANT_ID}/oauth2/token" \
  -d "grant_type=client_credentials" \
  -d "client_id=${AI_OSDU_CLIENT}" \
  -d "client_secret=${AI_OSDU_SECRET}" \
  -d "resource=${AI_OSDU_CLIENT}" \
  | jq -r '.access_token')

# Test API call
curl -s "https://${AI_OSDU_HOST}/api/entitlements/v2/groups" \
  -H "Authorization: Bearer ${TOKEN}" \
  -H "data-partition-id: ${AI_OSDU_DATA_PARTITION}" \
  | jq .
```

### Check Script Help

```bash
uv run .claude/skills/azure-osdu/scripts/osdu_users.py --help
uv run .claude/skills/azure-osdu/scripts/osdu_manage.py --help
uv run .claude/skills/azure-osdu/scripts/osdu_manage.py add --help
```

## Getting Help

### Documentation
- [SKILL.md](../SKILL.md) - Quick reference
- [entitlements.md](entitlements.md) - API reference

### Azure Resources
- [OSDU Documentation](https://learn.microsoft.com/en-us/azure/energy-data-services/)
- [OSDU Entitlements API](https://community.opengroup.org/osdu/platform/system/reference/crs-catalog-service)
