#!/usr/bin/env -S uv run --script
# /// script
# requires-python = ">=3.11"
# dependencies = [
#     "httpx",
#     "click",
#     "rich",
# ]
# ///
"""
OSDU User Management

Add, remove, or update users in OSDU entitlement groups.

Owner Management:
    Users can be added as OWNER (can manage group members) or MEMBER (access only).
    When removing OWNERs, the script protects against orphaning groups by blocking
    removal of the last OWNER unless --force is used.

Usage:
    uv run osdu_manage.py add --user EMAIL --role ROLE [--as-owner]
    uv run osdu_manage.py remove --user EMAIL --role ROLE [--force]
    uv run osdu_manage.py remove --user EMAIL --all-roles [--force]
    uv run osdu_manage.py update --user EMAIL --role ROLE --to OWNER|MEMBER

Environment variables required:
    AI_OSDU_HOST            - OSDU instance hostname
    AI_OSDU_DATA_PARTITION  - Data partition ID
    AI_OSDU_CLIENT          - App registration client ID
    AI_OSDU_SECRET          - App registration secret
    AI_OSDU_TENANT_ID       - Azure AD tenant ID

Optional:
    AI_OSDU_DOMAIN          - Entitlements domain (default: dataservices.energy)
"""

import json
import os
import sys

import click
import httpx
from rich.console import Console

console = Console(stderr=True)

VALID_ROLES = ["Viewer", "Editor", "Admin", "Ops", "Root"]


def get_config() -> dict:
    """Get configuration from environment variables."""
    return {
        "host": os.environ.get("AI_OSDU_HOST"),
        "partition": os.environ.get("AI_OSDU_DATA_PARTITION"),
        "client_id": os.environ.get("AI_OSDU_CLIENT"),
        "client_secret": os.environ.get("AI_OSDU_SECRET"),
        "tenant_id": os.environ.get("AI_OSDU_TENANT_ID"),
        "domain": os.environ.get("AI_OSDU_DOMAIN", "dataservices.energy"),
    }


def validate_config(config: dict) -> list[str]:
    """Validate required configuration. Returns list of missing vars."""
    missing = []
    if not config.get("host"):
        missing.append("AI_OSDU_HOST")
    if not config.get("partition"):
        missing.append("AI_OSDU_DATA_PARTITION")
    if not config.get("client_id"):
        missing.append("AI_OSDU_CLIENT")
    if not config.get("client_secret"):
        missing.append("AI_OSDU_SECRET")
    if not config.get("tenant_id"):
        missing.append("AI_OSDU_TENANT_ID")
    return missing


def get_group_email(role: str, partition: str, domain: str) -> str:
    """Get the group email for a role."""
    role_map = {
        "Viewer": f"users.datalake.viewers@{partition}.{domain}",
        "Editor": f"users.datalake.editors@{partition}.{domain}",
        "Admin": f"users.datalake.admins@{partition}.{domain}",
        "Ops": f"users.datalake.ops@{partition}.{domain}",
        "Root": f"users.data.root@{partition}.{domain}",
    }
    return role_map.get(role, "")


def get_access_token(config: dict) -> str:
    """Get access token using client credentials flow (v1 endpoint)."""
    token_url = f"https://login.microsoftonline.com/{config['tenant_id']}/oauth2/token"

    data = {
        "grant_type": "client_credentials",
        "client_id": config["client_id"],
        "client_secret": config["client_secret"],
        "resource": config["client_id"],
    }

    with httpx.Client() as client:
        response = client.post(token_url, data=data)
        response.raise_for_status()
        return response.json()["access_token"]


def add_member_to_group(config: dict, token: str, group_email: str, user_email: str, member_role: str = "MEMBER") -> dict:
    """Add a user to a group."""
    url = f"https://{config['host']}/api/entitlements/v2/groups/{group_email}/members"

    headers = {
        "Authorization": f"Bearer {token}",
        "Content-Type": "application/json",
        "data-partition-id": config["partition"],
    }

    body = {
        "email": user_email,
        "role": member_role,
    }

    with httpx.Client() as client:
        response = client.post(url, headers=headers, json=body)
        if response.status_code == 409:
            return {"success": False, "error": "already_exists", "message": "User already in group"}
        response.raise_for_status()
        return {"success": True, "message": f"Added {user_email} to group as {member_role}"}


def remove_member_from_group(config: dict, token: str, group_email: str, user_email: str) -> dict:
    """Remove a user from a group."""
    url = f"https://{config['host']}/api/entitlements/v2/groups/{group_email}/members/{user_email}"

    headers = {
        "Authorization": f"Bearer {token}",
        "data-partition-id": config["partition"],
    }

    with httpx.Client() as client:
        response = client.delete(url, headers=headers)
        if response.status_code == 404:
            return {"success": False, "error": "not_found", "message": "User not in group"}
        response.raise_for_status()
        return {"success": True, "message": f"Removed {user_email} from group"}


def check_member_in_group(config: dict, token: str, group_email: str, user_email: str) -> bool:
    """Check if a user is in a group."""
    url = f"https://{config['host']}/api/entitlements/v2/groups/{group_email}/members"

    headers = {
        "Authorization": f"Bearer {token}",
        "Accept": "application/json",
        "data-partition-id": config["partition"],
    }

    with httpx.Client() as client:
        response = client.get(url, headers=headers)
        if response.status_code == 404:
            return False
        response.raise_for_status()
        members = response.json().get("members", [])
        return any(m.get("email", "").lower() == user_email.lower() for m in members)


def get_member_role(config: dict, token: str, group_email: str, user_email: str) -> str | None:
    """Get a user's role in a group (OWNER, MEMBER, or None if not found)."""
    url = f"https://{config['host']}/api/entitlements/v2/groups/{group_email}/members"

    headers = {
        "Authorization": f"Bearer {token}",
        "Accept": "application/json",
        "data-partition-id": config["partition"],
    }

    with httpx.Client() as client:
        response = client.get(url, headers=headers)
        if response.status_code == 404:
            return None
        response.raise_for_status()
        members = response.json().get("members", [])
        for member in members:
            if member.get("email", "").lower() == user_email.lower():
                return member.get("role", "MEMBER")
        return None


def count_owners(config: dict, token: str, group_email: str) -> int:
    """Count the number of OWNERs in a group."""
    url = f"https://{config['host']}/api/entitlements/v2/groups/{group_email}/members"

    headers = {
        "Authorization": f"Bearer {token}",
        "Accept": "application/json",
        "data-partition-id": config["partition"],
    }

    with httpx.Client() as client:
        response = client.get(url, headers=headers)
        if response.status_code == 404:
            return 0
        response.raise_for_status()
        members = response.json().get("members", [])
        return sum(1 for m in members if m.get("role") == "OWNER")


@click.group()
def cli():
    """OSDU User Management - Add or remove users from entitlement groups."""
    pass


@cli.command()
@click.option("--user", required=True, help="User email address")
@click.option("--role", required=True, type=click.Choice(VALID_ROLES), help="Role to assign")
@click.option("--as-owner", is_flag=True, help="Add as OWNER instead of MEMBER")
@click.option("--json", "output_json", is_flag=True, help="Output as JSON")
def add(user: str, role: str, as_owner: bool, output_json: bool):
    """Add a user to an OSDU entitlement group."""
    config = get_config()
    missing = validate_config(config)

    if missing:
        result = {"success": False, "error": "missing_config", "message": f"Missing: {', '.join(missing)}"}
        if output_json:
            print(json.dumps(result))
        else:
            console.print(f"[red]{result['message']}[/red]")
        sys.exit(1)

    group_email = get_group_email(role, config["partition"], config["domain"])
    member_role = "OWNER" if as_owner else "MEMBER"

    if not output_json:
        console.print(f"\n[bold blue]Adding user to OSDU[/bold blue]")
        console.print(f"User: {user}")
        console.print(f"Role: {role} ({member_role})")
        console.print(f"Group: {group_email}\n")

    try:
        if not output_json:
            with console.status("[bold green]Authenticating..."):
                token = get_access_token(config)
        else:
            token = get_access_token(config)

        # Check if already member
        if check_member_in_group(config, token, group_email, user):
            result = {"success": False, "error": "already_exists", "message": f"{user} is already a {role}"}
            if output_json:
                print(json.dumps(result))
            else:
                console.print(f"[yellow]⚠[/yellow] {result['message']}")
            return

        # Add member
        result = add_member_to_group(config, token, group_email, user, member_role)

        if output_json:
            print(json.dumps(result))
        else:
            if result["success"]:
                console.print(f"[green]✓[/green] {result['message']}")
            else:
                console.print(f"[red]✗[/red] {result['message']}")

    except httpx.HTTPStatusError as e:
        result = {"success": False, "error": "api_error", "message": str(e), "status_code": e.response.status_code}
        if output_json:
            print(json.dumps(result))
        else:
            console.print(f"[red]✗[/red] API Error: {e.response.status_code} - {e.response.text}")
        sys.exit(1)


@cli.command()
@click.option("--user", required=True, help="User email address")
@click.option("--role", type=click.Choice(VALID_ROLES), help="Role to remove from")
@click.option("--all-roles", is_flag=True, help="Remove from all roles")
@click.option("--force", is_flag=True, help="Force removal even if last OWNER")
@click.option("--json", "output_json", is_flag=True, help="Output as JSON")
def remove(user: str, role: str | None, all_roles: bool, force: bool, output_json: bool):
    """Remove a user from an OSDU entitlement group.

    By default, prevents removing the last OWNER of a group to avoid orphaning.
    Use --force to override this protection.
    """
    if not role and not all_roles:
        if output_json:
            print(json.dumps({"success": False, "error": "invalid_args", "message": "Specify --role or --all-roles"}))
        else:
            console.print("[red]Specify --role ROLE or --all-roles[/red]")
        sys.exit(1)

    config = get_config()
    missing = validate_config(config)

    if missing:
        result = {"success": False, "error": "missing_config", "message": f"Missing: {', '.join(missing)}"}
        if output_json:
            print(json.dumps(result))
        else:
            console.print(f"[red]{result['message']}[/red]")
        sys.exit(1)

    roles_to_remove = VALID_ROLES if all_roles else [role]

    if not output_json:
        console.print(f"\n[bold blue]Removing user from OSDU[/bold blue]")
        console.print(f"User: {user}")
        console.print(f"Roles: {', '.join(roles_to_remove)}\n")

    try:
        if not output_json:
            with console.status("[bold green]Authenticating..."):
                token = get_access_token(config)
        else:
            token = get_access_token(config)

        results = []
        for r in roles_to_remove:
            group_email = get_group_email(r, config["partition"], config["domain"])

            # Get user's role in group
            member_role = get_member_role(config, token, group_email, user)

            if member_role is None:
                results.append({"role": r, "success": True, "skipped": True, "message": f"Not a member of {r}"})
                if not output_json:
                    console.print(f"[dim]-[/dim] {r}: Not a member, skipped")
                continue

            # Check if removing OWNER and protect against orphaning
            if member_role == "OWNER" and not force:
                owner_count = count_owners(config, token, group_email)
                if owner_count <= 1:
                    results.append({
                        "role": r,
                        "success": False,
                        "error": "last_owner",
                        "was_owner": True,
                        "message": f"Cannot remove last OWNER from {r}. Use --force to override."
                    })
                    if not output_json:
                        console.print(f"[yellow]![/yellow] {r}: Cannot remove last OWNER. Use --force to override.")
                    continue

            # Remove member
            result = remove_member_from_group(config, token, group_email, user)
            result["role"] = r
            result["was_owner"] = member_role == "OWNER"
            results.append(result)

            if not output_json:
                role_label = "OWNER" if member_role == "OWNER" else "MEMBER"
                if result["success"]:
                    console.print(f"[green]✓[/green] Removed from {r} (was {role_label})")
                else:
                    console.print(f"[red]✗[/red] {r}: {result['message']}")

        if output_json:
            print(json.dumps({"success": True, "user": user, "results": results}))

    except httpx.HTTPStatusError as e:
        result = {"success": False, "error": "api_error", "message": str(e), "status_code": e.response.status_code}
        if output_json:
            print(json.dumps(result))
        else:
            console.print(f"[red]✗[/red] API Error: {e.response.status_code} - {e.response.text}")
        sys.exit(1)


@cli.command()
@click.option("--json", "output_json", is_flag=True, help="Output as JSON")
def check(output_json: bool):
    """Check OSDU configuration and connectivity."""
    config = get_config()
    missing = validate_config(config)

    if missing:
        result = {"success": False, "configured": False, "missing": missing}
        if output_json:
            print(json.dumps(result))
        else:
            console.print("[red]Missing environment variables:[/red]")
            for var in missing:
                console.print(f"  - {var}")
        sys.exit(1)

    # Try to authenticate
    try:
        if not output_json:
            with console.status("[bold green]Testing authentication..."):
                token = get_access_token(config)
        else:
            token = get_access_token(config)

        result = {
            "success": True,
            "configured": True,
            "host": config["host"],
            "partition": config["partition"],
            "authenticated": True,
        }

        if output_json:
            print(json.dumps(result))
        else:
            console.print("[green]✓[/green] Configuration valid")
            console.print(f"  Host: {config['host']}")
            console.print(f"  Partition: {config['partition']}")
            console.print("[green]✓[/green] Authentication successful")

    except httpx.HTTPStatusError as e:
        result = {
            "success": False,
            "configured": True,
            "authenticated": False,
            "error": str(e),
        }
        if output_json:
            print(json.dumps(result))
        else:
            console.print("[green]✓[/green] Configuration present")
            console.print(f"[red]✗[/red] Authentication failed: {e}")
        sys.exit(1)


@cli.command()
@click.option("--user", required=True, help="User email address")
@click.option("--role", required=True, type=click.Choice(VALID_ROLES), help="Role group to update")
@click.option("--to", "new_role", required=True, type=click.Choice(["OWNER", "MEMBER"]),
              help="New role (OWNER or MEMBER)")
@click.option("--force", is_flag=True, help="Force change even if last OWNER")
@click.option("--json", "output_json", is_flag=True, help="Output as JSON")
def update(user: str, role: str, new_role: str, force: bool, output_json: bool):
    """Change a user's role in an OSDU entitlement group (OWNER/MEMBER).

    Updates a user's membership type without removing and re-adding them manually.
    Protects against downgrading the last OWNER unless --force is used.
    """
    config = get_config()
    missing = validate_config(config)

    if missing:
        result = {"success": False, "error": "missing_config", "message": f"Missing: {', '.join(missing)}"}
        if output_json:
            print(json.dumps(result))
        else:
            console.print(f"[red]{result['message']}[/red]")
        sys.exit(1)

    group_email = get_group_email(role, config["partition"], config["domain"])

    if not output_json:
        console.print(f"\n[bold blue]Updating user role in OSDU[/bold blue]")
        console.print(f"User: {user}")
        console.print(f"Role group: {role}")
        console.print(f"New membership: {new_role}\n")

    try:
        if not output_json:
            with console.status("[bold green]Authenticating..."):
                token = get_access_token(config)
        else:
            token = get_access_token(config)

        # Get user's current role
        current_role = get_member_role(config, token, group_email, user)

        if current_role is None:
            result = {"success": False, "error": "not_found",
                     "message": f"{user} is not a member of {role}"}
            if output_json:
                print(json.dumps(result))
            else:
                console.print(f"[red]✗[/red] {result['message']}")
            sys.exit(1)

        if current_role == new_role:
            result = {"success": True, "message": f"{user} is already {new_role} in {role}"}
            if output_json:
                print(json.dumps(result))
            else:
                console.print(f"[yellow]⚠[/yellow] {result['message']}")
            return

        # Check if downgrading owner and protect against orphaning
        if current_role == "OWNER" and new_role == "MEMBER" and not force:
            owner_count = count_owners(config, token, group_email)
            if owner_count <= 1:
                result = {"success": False, "error": "last_owner",
                         "message": f"Cannot downgrade last OWNER in {role}. Use --force to override."}
                if output_json:
                    print(json.dumps(result))
                else:
                    console.print(f"[yellow]![/yellow] {result['message']}")
                sys.exit(1)

        # Perform the role change (remove + add with new role)
        if not output_json:
            with console.status("[bold green]Updating role..."):
                remove_member_from_group(config, token, group_email, user)
                add_member_to_group(config, token, group_email, user, new_role)
        else:
            remove_member_from_group(config, token, group_email, user)
            add_member_to_group(config, token, group_email, user, new_role)

        result = {
            "success": True,
            "user": user,
            "role": role,
            "from": current_role,
            "to": new_role,
            "message": f"Changed {user} from {current_role} to {new_role} in {role}"
        }

        if output_json:
            print(json.dumps(result))
        else:
            console.print(f"[green]✓[/green] {result['message']}")

    except httpx.HTTPStatusError as e:
        result = {"success": False, "error": "api_error", "message": str(e), "status_code": e.response.status_code}
        if output_json:
            print(json.dumps(result))
        else:
            console.print(f"[red]✗[/red] API Error: {e.response.status_code} - {e.response.text}")
        sys.exit(1)


if __name__ == "__main__":
    cli()
