#!/bin/bash

# backup-restore - Backup and restore database and configurations
# Usage: backup-restore <command> [options]

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

# Configuration
BACKUP_DIR="${BACKUP_DIR:-$HOME/dev-environment/backups}"
DB_CONTAINER="${DB_CONTAINER:-supabase-db}"
DB_NAME="${DB_NAME:-postgres}"
DB_USER="${DB_USER:-postgres}"

# Helper functions
print_success() { echo -e "${GREEN}✓${NC} $1"; }
print_error() { echo -e "${RED}✗${NC} $1"; }
print_info() { echo -e "${BLUE}ℹ${NC} $1"; }
print_warning() { echo -e "${YELLOW}⚠${NC} $1"; }

show_usage() {
    cat << EOF
${BLUE}backup-restore${NC} - Backup and restore database and configurations

${YELLOW}USAGE:${NC}
    backup-restore <command> [options]

${YELLOW}COMMANDS:${NC}
    backup [name]          Create a new backup (default: timestamped)
    restore <name>         Restore from a backup
    list                   List all available backups
    info <name>            Show backup information
    delete <name>          Delete a backup
    clean [days]           Remove backups older than N days (default: 30)
    auto                   Setup automatic daily backups
    export <name> <path>   Export backup to specific path
    import <path>          Import backup from path
    help                   Show this help message

${YELLOW}BACKUP TYPES:${NC}
    database              PostgreSQL database dump
    configs               Environment files and configs
    full                  Database + configs (default)

${YELLOW}OPTIONS:${NC}
    --type TYPE           Backup type: database, configs, or full
    --compress            Compress backup (gzip)
    --encrypt             Encrypt backup (gpg)
    --include-storage     Include uploaded files (storage)

${YELLOW}EXAMPLES:${NC}
    # Create a full backup
    backup-restore backup my-backup

    # Create database-only backup
    backup-restore backup db-only --type database

    # List all backups
    backup-restore list

    # Restore from backup
    backup-restore restore my-backup

    # Clean old backups
    backup-restore clean 7

    # Export backup for transfer
    backup-restore export my-backup /tmp/backup.tar.gz

${YELLOW}BACKUP LOCATION:${NC}
    $BACKUP_DIR

${YELLOW}AUTO-BACKUP:${NC}
    Use 'backup-restore auto' to setup daily automatic backups via cron
EOF
}

# Ensure backup directory exists
init_backup_dir() {
    mkdir -p "$BACKUP_DIR"
    mkdir -p "$BACKUP_DIR/database"
    mkdir -p "$BACKUP_DIR/configs"
    mkdir -p "$BACKUP_DIR/full"
}

# Generate backup name with timestamp
generate_backup_name() {
    local prefix=${1:-backup}
    echo "${prefix}_$(date +%Y%m%d_%H%M%S)"
}

# Backup database
backup_database() {
    local name=$1
    local compress=${2:-false}
    
    init_backup_dir
    
    local backup_file="$BACKUP_DIR/database/${name}.sql"
    
    print_info "Creating database backup: $name"
    
    # Check if container is running
    if ! docker ps --format '{{.Names}}' | grep -q "^${DB_CONTAINER}$"; then
        print_error "Database container '$DB_CONTAINER' is not running"
        exit 1
    fi
    
    # Create backup
    docker exec "$DB_CONTAINER" pg_dump -U "$DB_USER" "$DB_NAME" > "$backup_file"
    
    if [ "$compress" = true ]; then
        print_info "Compressing backup..."
        gzip "$backup_file"
        backup_file="${backup_file}.gz"
    fi
    
    local size=$(du -h "$backup_file" | cut -f1)
    print_success "Database backup created: $backup_file ($size)"
    
    # Create metadata
    cat > "${backup_file}.meta" << META
type=database
created=$(date -Iseconds)
size=$size
container=$DB_CONTAINER
database=$DB_NAME
META
}

# Backup configurations
backup_configs() {
    local name=$1
    
    init_backup_dir
    
    local backup_file="$BACKUP_DIR/configs/${name}.tar.gz"
    
    print_info "Creating config backup: $name"
    
    # Files to backup
    local files_to_backup=(
        "$HOME/.env"
        "$HOME/.env.local"
        "$HOME/dev-environment/.env"
        "$HOME/dev-environment/docker-compose.yml"
        "$HOME/dev-environment/supabase/kong.yml"
    )
    
    # Create tar archive of existing files
    local temp_list="/tmp/backup_files_$$"
    > "$temp_list"
    
    for file in "${files_to_backup[@]}"; do
        if [ -f "$file" ]; then
            echo "$file" >> "$temp_list"
        fi
    done
    
    if [ -s "$temp_list" ]; then
        tar -czf "$backup_file" -T "$temp_list" 2>/dev/null || true
        rm "$temp_list"
        
        local size=$(du -h "$backup_file" | cut -f1)
        print_success "Config backup created: $backup_file ($size)"
        
        # Create metadata
        cat > "${backup_file}.meta" << META
type=configs
created=$(date -Iseconds)
size=$size
META
    else
        print_warning "No config files found to backup"
        rm "$temp_list"
    fi
}

# Full backup (database + configs)
backup_full() {
    local name=$1
    local compress=${2:-true}
    
    init_backup_dir
    
    print_info "Creating full backup: $name"
    
    # Backup database
    backup_database "${name}_db" "$compress"
    
    # Backup configs
    backup_configs "${name}_cfg"
    
    # Create full backup directory
    local full_backup_dir="$BACKUP_DIR/full/$name"
    mkdir -p "$full_backup_dir"
    
    # Move backups to full backup directory
    if [ "$compress" = true ]; then
        mv "$BACKUP_DIR/database/${name}_db.sql.gz"* "$full_backup_dir/" 2>/dev/null || true
    else
        mv "$BACKUP_DIR/database/${name}_db.sql"* "$full_backup_dir/" 2>/dev/null || true
    fi
    mv "$BACKUP_DIR/configs/${name}_cfg.tar.gz"* "$full_backup_dir/" 2>/dev/null || true
    
    # Create full backup metadata
    cat > "$full_backup_dir/backup.meta" << META
type=full
created=$(date -Iseconds)
database_backup=${name}_db
config_backup=${name}_cfg
META
    
    print_success "Full backup created: $full_backup_dir"
}

# List backups
list_backups() {
    init_backup_dir
    
    print_info "Available backups:"
    echo ""
    
    # List full backups
    if [ -d "$BACKUP_DIR/full" ] && [ "$(ls -A "$BACKUP_DIR/full" 2>/dev/null)" ]; then
        echo -e "${CYAN}Full Backups:${NC}"
        for backup in "$BACKUP_DIR/full"/*; do
            if [ -d "$backup" ]; then
                local name=$(basename "$backup")
                local created=""
                if [ -f "$backup/backup.meta" ]; then
                    created=$(grep "^created=" "$backup/backup.meta" | cut -d= -f2)
                fi
                echo -e "  ${GREEN}●${NC} $name ${YELLOW}($created)${NC}"
            fi
        done
        echo ""
    fi
    
    # List database backups
    if [ -d "$BACKUP_DIR/database" ] && [ "$(ls -A "$BACKUP_DIR/database" 2>/dev/null | grep -v '.meta$')" ]; then
        echo -e "${CYAN}Database Backups:${NC}"
        for backup in "$BACKUP_DIR/database"/*.sql* 2>/dev/null; do
            if [ -f "$backup" ] && [[ ! "$backup" =~ \.meta$ ]]; then
                local name=$(basename "$backup" | sed 's/\.\(sql\|sql\.gz\)$//')
                local size=$(du -h "$backup" | cut -f1)
                echo -e "  ${GREEN}●${NC} $name ${YELLOW}($size)${NC}"
            fi
        done
        echo ""
    fi
    
    # List config backups
    if [ -d "$BACKUP_DIR/configs" ] && [ "$(ls -A "$BACKUP_DIR/configs" 2>/dev/null | grep -v '.meta$')" ]; then
        echo -e "${CYAN}Config Backups:${NC}"
        for backup in "$BACKUP_DIR/configs"/*.tar.gz 2>/dev/null; do
            if [ -f "$backup" ] && [[ ! "$backup" =~ \.meta$ ]]; then
                local name=$(basename "$backup" .tar.gz)
                local size=$(du -h "$backup" | cut -f1)
                echo -e "  ${GREEN}●${NC} $name ${YELLOW}($size)${NC}"
            fi
        done
    fi
}

# Restore database
restore_database() {
    local name=$1
    
    # Find backup file
    local backup_file=""
    if [ -f "$BACKUP_DIR/database/${name}.sql" ]; then
        backup_file="$BACKUP_DIR/database/${name}.sql"
    elif [ -f "$BACKUP_DIR/database/${name}.sql.gz" ]; then
        backup_file="$BACKUP_DIR/database/${name}.sql.gz"
    else
        print_error "Backup not found: $name"
        exit 1
    fi
    
    print_warning "This will restore database from: $name"
    print_warning "Current data will be overwritten! Continue? (y/N)"
    read -r confirm
    
    if [ "$confirm" != "y" ] && [ "$confirm" != "Y" ]; then
        print_info "Cancelled"
        exit 0
    fi
    
    print_info "Restoring database..."
    
    # Check if container is running
    if ! docker ps --format '{{.Names}}' | grep -q "^${DB_CONTAINER}$"; then
        print_error "Database container '$DB_CONTAINER' is not running"
        exit 1
    fi
    
    # Restore backup
    if [[ "$backup_file" =~ \.gz$ ]]; then
        gunzip -c "$backup_file" | docker exec -i "$DB_CONTAINER" psql -U "$DB_USER" "$DB_NAME"
    else
        docker exec -i "$DB_CONTAINER" psql -U "$DB_USER" "$DB_NAME" < "$backup_file"
    fi
    
    print_success "Database restored from: $name"
}

# Clean old backups
clean_backups() {
    local days=${1:-30}
    
    print_info "Cleaning backups older than $days days..."
    
    local count=0
    
    # Clean full backups
    if [ -d "$BACKUP_DIR/full" ]; then
        count=$((count + $(find "$BACKUP_DIR/full" -type d -mtime +"$days" | wc -l)))
        find "$BACKUP_DIR/full" -type d -mtime +"$days" -exec rm -rf {} + 2>/dev/null || true
    fi
    
    # Clean database backups
    if [ -d "$BACKUP_DIR/database" ]; then
        count=$((count + $(find "$BACKUP_DIR/database" -type f -mtime +"$days" | wc -l)))
        find "$BACKUP_DIR/database" -type f -mtime +"$days" -delete 2>/dev/null || true
    fi
    
    # Clean config backups
    if [ -d "$BACKUP_DIR/configs" ]; then
        count=$((count + $(find "$BACKUP_DIR/configs" -type f -mtime +"$days" | wc -l)))
        find "$BACKUP_DIR/configs" -type f -mtime +"$days" -delete 2>/dev/null || true
    fi
    
    print_success "Removed $count old backups"
}

# Delete specific backup
delete_backup() {
    local name=$1
    
    print_warning "Delete backup: $name? (y/N)"
    read -r confirm
    
    if [ "$confirm" != "y" ] && [ "$confirm" != "Y" ]; then
        print_info "Cancelled"
        exit 0
    fi
    
    # Try to delete from all locations
    rm -rf "$BACKUP_DIR/full/$name" 2>/dev/null || true
    rm -f "$BACKUP_DIR/database/${name}".* 2>/dev/null || true
    rm -f "$BACKUP_DIR/configs/${name}".* 2>/dev/null || true
    
    print_success "Backup deleted: $name"
}

# Main script logic
command="${1:-help}"
shift || true

case "$command" in
    backup)
        name=${1:-$(generate_backup_name)}
        backup_type=${2:-full}
        compress=true
        
        case "$backup_type" in
            database)
                backup_database "$name" "$compress"
                ;;
            configs)
                backup_configs "$name"
                ;;
            full|*)
                backup_full "$name" "$compress"
                ;;
        esac
        ;;
    restore)
        [ -z "$1" ] && { print_error "Backup name required"; show_usage; exit 1; }
        restore_database "$1"
        ;;
    list)
        list_backups
        ;;
    delete)
        [ -z "$1" ] && { print_error "Backup name required"; show_usage; exit 1; }
        delete_backup "$1"
        ;;
    clean)
        clean_backups "${1:-30}"
        ;;
    auto)
        print_info "Setting up automatic daily backups..."
        # Add cron job for daily backups at 2 AM
        (crontab -l 2>/dev/null | grep -v "backup-restore"; echo "0 2 * * * $HOME/.claude-skills/core/backup-restore backup auto_\$(date +\%Y\%m\%d)") | crontab -
        print_success "Daily backups configured for 2:00 AM"
        ;;
    help|--help|-h)
        show_usage
        ;;
    *)
        print_error "Unknown command: $command"
        show_usage
        exit 1
        ;;
esac
