#!/usr/bin/env python3
"""
Basecamp API client for BlackBelt Meeting Bot.

Handles authentication, client lookup, and comment posting.
Configuration loaded from ~/.config/blackbelt-basecamp.yaml
"""

import os
import re
import requests
import yaml
from pathlib import Path
from rapidfuzz import fuzz
from typing import Optional, Dict, List, Any


class BasecampClient:
    """Client for interacting with Basecamp API."""

    CONFIG_PATH = Path.home() / ".config" / "blackbelt-basecamp.yaml"
    BASE_URL = "https://3.basecampapi.com"
    USER_AGENT = "BlackBelt Meeting Bot (ed@eddale.com)"

    def __init__(self):
        self.config = self._load_config()
        self.account_id = self.config["credentials"]["account_id"]
        self.access_token = self.config["credentials"]["access_token"]
        self.project_id = self.config["onboarding"]["project_id"]
        self.groups = self.config["onboarding"]["groups"]

    def _load_config(self) -> dict:
        """Load configuration from YAML file."""
        if not self.CONFIG_PATH.exists():
            raise FileNotFoundError(
                f"Config not found at {self.CONFIG_PATH}. "
                "Run the OAuth setup first."
            )
        with open(self.CONFIG_PATH) as f:
            return yaml.safe_load(f)

    def _headers(self) -> dict:
        """Get headers for API requests."""
        return {
            "Authorization": f"Bearer {self.access_token}",
            "Content-Type": "application/json; charset=utf-8",
            "User-Agent": self.USER_AGENT,
        }

    def _api_url(self, path: str) -> str:
        """Build full API URL."""
        return f"{self.BASE_URL}/{self.account_id}{path}"

    def _get(self, path: str, paginate: bool = False) -> Any:
        """Make GET request to API.

        Args:
            path: API path to request
            paginate: If True, follow pagination links and return all results
        """
        url = self._api_url(path)
        response = requests.get(url, headers=self._headers())
        response.raise_for_status()

        if not paginate:
            return response.json()

        # Handle pagination - Basecamp uses Link headers
        all_results = response.json()

        while True:
            link_header = response.headers.get('Link', '')
            if 'rel="next"' not in link_header:
                break

            # Extract next URL from Link header
            match = re.search(r'<([^>]+)>; rel="next"', link_header)
            if not match:
                break

            next_url = match.group(1)
            response = requests.get(next_url, headers=self._headers())
            response.raise_for_status()
            all_results.extend(response.json())

        return all_results

    def _post(self, path: str, data: dict) -> Any:
        """Make POST request to API."""
        response = requests.post(
            self._api_url(path),
            headers=self._headers(),
            json=data
        )
        response.raise_for_status()
        return response.json()

    def refresh_access_token(self) -> str:
        """Use refresh token to get new access token."""
        refresh_token = self.config["credentials"]["refresh_token"]
        client_id = self.config["credentials"]["client_id"]
        client_secret = self.config["credentials"]["client_secret"]

        response = requests.post(
            "https://launchpad.37signals.com/authorization/token",
            data={
                "type": "refresh",
                "refresh_token": refresh_token,
                "client_id": client_id,
                "client_secret": client_secret,
            }
        )
        response.raise_for_status()
        data = response.json()

        # Update config with new token
        self.config["credentials"]["access_token"] = data["access_token"]
        self.access_token = data["access_token"]

        # Save updated config
        with open(self.CONFIG_PATH, "w") as f:
            yaml.dump(self.config, f, default_flow_style=False)

        return data["access_token"]

    def get_all_clients(self, include_completed: bool = True) -> List[Dict]:
        """Get all clients from all onboarding groups.

        Args:
            include_completed: If True, also fetch completed/checked-off todos.
                              Default True since clients may have completed onboarding
                              but still have active coaching calls.
        """
        all_clients = []

        for group_name, group_id in self.groups.items():
            try:
                # Fetch incomplete todos (with pagination)
                todos = self._get(
                    f"/buckets/{self.project_id}/todolists/{group_id}/todos.json",
                    paginate=True
                )
                for todo in todos:
                    client_info = self._parse_client_title(todo["title"])
                    client_info["todo_id"] = todo["id"]
                    client_info["group"] = group_name
                    client_info["full_title"] = todo["title"]
                    client_info["completed"] = False
                    all_clients.append(client_info)

                # Also fetch completed todos if requested
                if include_completed:
                    completed_todos = self._get(
                        f"/buckets/{self.project_id}/todolists/{group_id}/todos.json?completed=true",
                        paginate=True
                    )
                    for todo in completed_todos:
                        client_info = self._parse_client_title(todo["title"])
                        client_info["todo_id"] = todo["id"]
                        client_info["group"] = group_name
                        client_info["full_title"] = todo["title"]
                        client_info["completed"] = True
                        all_clients.append(client_info)
            except Exception as e:
                # Group might be empty or inaccessible
                continue

        return all_clients

    def _parse_client_title(self, title: str) -> Dict:
        """
        Parse client info from todo title.
        Format A: DD.MM.YYYY | Client Name | (email) | Location
        Format B: Client Name | (email) | Location (no date)
        """
        parts = [p.strip() for p in title.split("|")]
        result = {"name": title, "email": "", "location": "", "date": ""}

        if len(parts) < 2:
            return result

        # Check if first field looks like a date (DD.MM.YYYY or similar)
        first_field = parts[0]
        is_date = bool(re.match(r'^\d{1,2}[./]\d{1,2}[./]\d{2,4}$', first_field))

        if is_date:
            # Format A: DD.MM.YYYY | Name | (email) | Location
            result["date"] = first_field
            result["name"] = parts[1] if len(parts) > 1 else ""
            if len(parts) >= 3:
                email_part = parts[2]
                email_match = re.search(r'\(([^)]+)\)', email_part)
                result["email"] = email_match.group(1) if email_match else email_part
            if len(parts) >= 4:
                result["location"] = parts[3]
        else:
            # Format B: Name | (email) | Location (no date prefix)
            result["name"] = first_field
            if len(parts) >= 2:
                email_part = parts[1]
                email_match = re.search(r'\(([^)]+)\)', email_part)
                result["email"] = email_match.group(1) if email_match else email_part
            if len(parts) >= 3:
                result["location"] = parts[2]

        return result

    def find_client(self, search_name: str) -> Optional[Dict]:
        """Find client by name using fuzzy matching.

        Uses rapidfuzz token_set_ratio which handles word-based matching better.
        Returns best match if score >= 75%, otherwise None.
        """
        clients = self.get_all_clients()

        best_match = None
        best_score = 0

        for client in clients:
            # Use token_set_ratio for word-aware matching
            # This handles "Brad" matching "Brad Twynham" properly
            score = fuzz.token_set_ratio(
                search_name.lower(),
                client["name"].lower()
            ) / 100  # Convert to 0-1 range

            if score > best_score:
                best_score = score
                best_match = {**client, "score": score}

        # Require at least 75% match (raised from 50%)
        if best_match and best_score >= 0.75:
            return best_match

        return None

    def find_clients(self, search_name: str, limit: int = 5) -> List[Dict]:
        """Find multiple potential client matches."""
        clients = self.get_all_clients()
        scored = []

        for client in clients:
            # Use token_set_ratio for word-aware matching
            score = fuzz.token_set_ratio(
                search_name.lower(),
                client["name"].lower()
            ) / 100  # Convert to 0-1 range
            scored.append({**client, "score": score})

        # Sort by score descending
        scored.sort(key=lambda x: x["score"], reverse=True)
        return scored[:limit]

    def post_comment(self, todo_id: int, content: str) -> Dict:
        """
        Post a comment to a client's todo.

        Args:
            todo_id: The Basecamp todo ID
            content: HTML content for the comment

        Returns:
            API response with comment details
        """
        path = f"/buckets/{self.project_id}/recordings/{todo_id}/comments.json"
        return self._post(path, {"content": content})

    def get_client_comments(self, todo_id: int) -> List[Dict]:
        """Get all comments on a client's todo."""
        path = f"/buckets/{self.project_id}/recordings/{todo_id}/comments.json"
        return self._get(path)

    def get_project_people(self) -> List[Dict]:
        """Get all people in the project with their attachable_sgid for mentions."""
        path = f"/projects/{self.project_id}/people.json"
        return self._get(path, paginate=True)

    def get_default_mentions_html(self) -> str:
        """Generate HTML mention tags for default team members (Toni, Tamara, Vanessa)."""
        mentions = self.config.get("default_mentions", [])
        if not mentions:
            return ""

        tags = []
        for person in mentions:
            sgid = person.get("sgid")
            if sgid:
                tags.append(f'<bc-attachment sgid="{sgid}"></bc-attachment>')

        return " ".join(tags)

    def get_person_sgid(self, name: str) -> Optional[str]:
        """Find a team member's SGID by name for ad-hoc mentions."""
        team = self.config.get("team_cache", [])
        for person in team:
            if person["name"].lower() == name.lower():
                return person["sgid"]
            # Also check first name match
            if person["name"].split()[0].lower() == name.lower():
                return person["sgid"]
        return None


def markdown_to_basecamp_html(markdown_text: str) -> str:
    """
    Convert markdown to Basecamp-compatible HTML.

    Basecamp supports: div, h1, br, strong, em, a, pre, ol, ul, li, blockquote
    """
    html = markdown_text

    # Convert headers (only h1 supported)
    html = re.sub(r'^# (.+)$', r'<h1>\1</h1>', html, flags=re.MULTILINE)
    html = re.sub(r'^## (.+)$', r'<strong>\1</strong>', html, flags=re.MULTILINE)
    html = re.sub(r'^### (.+)$', r'<strong>\1</strong>', html, flags=re.MULTILINE)

    # Convert bold and italic
    html = re.sub(r'\*\*(.+?)\*\*', r'<strong>\1</strong>', html)
    html = re.sub(r'\*(.+?)\*', r'<em>\1</em>', html)

    # Convert bullet lists
    # Find consecutive lines starting with - and wrap in <ul>
    def replace_ul(match):
        items = match.group(0)
        items = re.sub(r'^- (.+)$', r'<li>\1</li>', items, flags=re.MULTILINE)
        return f'<ul>{items}</ul>'

    html = re.sub(r'(^- .+$\n?)+', replace_ul, html, flags=re.MULTILINE)

    # Convert numbered lists
    def replace_ol(match):
        items = match.group(0)
        items = re.sub(r'^\d+\. (.+)$', r'<li>\1</li>', items, flags=re.MULTILINE)
        return f'<ol>{items}</ol>'

    html = re.sub(r'(^\d+\. .+$\n?)+', replace_ol, html, flags=re.MULTILINE)

    # Convert line breaks (double newline = paragraph break, single = br)
    html = re.sub(r'\n\n', '<br><br>', html)
    html = re.sub(r'\n', '<br>', html)

    # Wrap in div
    return f'<div>{html}</div>'


if __name__ == "__main__":
    # Quick test
    client = BasecampClient()
    print("Config loaded successfully!")
    print(f"Account ID: {client.account_id}")
    print(f"Project ID: {client.project_id}")

    # Test finding a client
    result = client.find_client("Brad Twynham")
    if result:
        print(f"\nFound: {result['name']} (score: {result['score']:.2f})")
        print(f"Todo ID: {result['todo_id']}")
        print(f"Group: {result['group']}")
