#!/usr/bin/env python3
"""
Post a meeting summary to Basecamp.

Usage:
    python3 post_to_basecamp.py --client "Brad Twynham" --summary /path/to/summary.md
    python3 post_to_basecamp.py --client "Brad Twynham" --content "Direct content here"
    python3 post_to_basecamp.py --todo-id 9313565737 --summary /path/to/summary.md
    python3 post_to_basecamp.py --list-clients "Brad"  # Search for clients

    # With VA task (posts a short follow-up comment for VAs):
    python3 post_to_basecamp.py --client "Brad" --summary summary.md --va-task "Send recording"

Exit codes:
    0 - Success
    1 - Client not found
    2 - API error
    3 - Configuration error
"""

import argparse
import sys
import json
from pathlib import Path

# Add tools directory to path for imports
sys.path.insert(0, str(Path(__file__).parent))

from basecamp_client import BasecampClient, markdown_to_basecamp_html


def find_and_confirm_client(client: BasecampClient, name: str, auto_confirm: bool = False) -> dict:
    """Find client and confirm if match score is low."""
    result = client.find_client(name)

    if not result:
        print(f"ERROR: No client found matching '{name}'", file=sys.stderr)
        # Show top matches to help user find the right name
        top_matches = client.find_clients(name, limit=5)
        if top_matches:
            print("\nTop matches found:", file=sys.stderr)
            for m in top_matches[:5]:
                print(f"  {m['name']} ({m['score']:.0%})", file=sys.stderr)
        print("\nUse --list-clients to search, or --todo-id for direct posting.", file=sys.stderr)
        sys.exit(1)

    print(f"Found: {result['name']} (match: {result['score']:.0%})")
    print(f"  Group: {result['group'].replace('_', ' ').title()}")
    print(f"  Todo ID: {result['todo_id']}")

    # Safety floor: refuse to proceed below 80% even with --yes
    if result['score'] < 0.8:
        print(f"\nERROR: Match confidence {result['score']:.0%} is too low.", file=sys.stderr)
        print("Use --todo-id to post directly, or provide the full client name.", file=sys.stderr)
        # Show what we found
        top_matches = client.find_clients(name, limit=5)
        if top_matches:
            print("\nTop matches:", file=sys.stderr)
            for m in top_matches[:5]:
                print(f"  {m['name']} ({m['score']:.0%}) - ID: {m['todo_id']}", file=sys.stderr)
        sys.exit(1)

    # For scores between 80-90%, ask for confirmation unless --yes
    if result['score'] < 0.9 and not auto_confirm:
        print(f"\nMatch confidence is {result['score']:.0%}. Please confirm this is correct.")
        confirm = input(f"Post to '{result['name']}'? [y/N] ").strip().lower()
        if confirm != 'y':
            print("Cancelled.")
            sys.exit(0)

    return result


def post_summary(client: BasecampClient, todo_id: int, content: str,
                 is_markdown: bool = True, also_tag: list = None, no_mentions: bool = False):
    """Post content to a client's todo with team mentions.

    Args:
        client: BasecampClient instance
        todo_id: The Basecamp todo ID
        content: Content to post (markdown or HTML)
        is_markdown: If True, convert markdown to HTML
        also_tag: Additional team members to tag (names)
        no_mentions: If True, skip all mention tags
    """
    if is_markdown:
        html_content = markdown_to_basecamp_html(content)
    else:
        # Assume already HTML or plain text
        if not content.startswith('<'):
            html_content = f"<div>{content}</div>"
        else:
            html_content = content

    # Add team mentions unless disabled
    if not no_mentions:
        mentions_html = client.get_default_mentions_html()

        # Add ad-hoc mentions if requested
        if also_tag:
            extra_tags = []
            for name in also_tag:
                sgid = client.get_person_sgid(name)
                if sgid:
                    extra_tags.append(f'<bc-attachment sgid="{sgid}"></bc-attachment>')
                else:
                    print(f"  Warning: Could not find team member '{name}' for tagging")
            if extra_tags:
                mentions_html = mentions_html + " " + " ".join(extra_tags) if mentions_html else " ".join(extra_tags)

        if mentions_html:
            # Append mentions at the end of the content
            html_content = html_content.rstrip('</div>') + f"<br><br>{mentions_html}</div>"

    try:
        result = client.post_comment(todo_id, html_content)
        print(f"\n✓ Comment posted successfully!")
        print(f"  Comment ID: {result['id']}")
        print(f"  View at: {result.get('app_url', 'Check Basecamp')}")
        return result
    except Exception as e:
        print(f"\nERROR: Failed to post comment: {e}", file=sys.stderr)
        sys.exit(2)


def list_clients(client: BasecampClient, search: str = ""):
    """List clients matching search term."""
    if search:
        matches = client.find_clients(search, limit=10)
        print(f"Top matches for '{search}':\n")
    else:
        matches = client.get_all_clients()[:20]
        print("Recent clients (showing first 20):\n")

    for m in matches:
        score = f" ({m['score']:.0%})" if 'score' in m else ""
        group = m['group'].replace('_', ' ').title()
        print(f"  {m['name']}{score}")
        print(f"    ID: {m['todo_id']} | Group: {group}")


def main():
    parser = argparse.ArgumentParser(
        description="Post meeting summaries to Basecamp",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog=__doc__
    )

    # Client identification (one required)
    group = parser.add_mutually_exclusive_group()
    group.add_argument("--client", "-c", help="Client name to search for")
    group.add_argument("--todo-id", "-t", type=int, help="Direct Basecamp todo ID")
    group.add_argument("--list-clients", "-l", nargs="?", const="", metavar="SEARCH",
                       help="List clients (optionally filtered by search term)")

    # Content (one required for posting)
    parser.add_argument("--summary", "-s", type=Path, help="Path to summary markdown file")
    parser.add_argument("--content", help="Direct content to post")

    # Options
    parser.add_argument("--yes", "-y", action="store_true",
                        help="Auto-confirm client match without prompting")
    parser.add_argument("--dry-run", "-n", action="store_true",
                        help="Show what would be posted without posting")
    parser.add_argument("--json", "-j", action="store_true",
                        help="Output result as JSON")

    # Mention options
    parser.add_argument("--also-tag", "-a", action="append", metavar="NAME",
                        help="Additional team members to tag (can repeat: --also-tag Bianca --also-tag Zeder)")
    parser.add_argument("--no-mentions", action="store_true",
                        help="Skip all team mention tags")

    # VA task option - posts a short follow-up comment for VAs
    parser.add_argument("--va-task", metavar="TASK",
                        help="Post a short follow-up comment for VAs (e.g., --va-task 'Send recording')")

    args = parser.parse_args()

    # Initialize client
    try:
        bc = BasecampClient()
    except FileNotFoundError as e:
        print(f"ERROR: {e}", file=sys.stderr)
        sys.exit(3)

    # Handle --list-clients
    if args.list_clients is not None:
        list_clients(bc, args.list_clients)
        return

    # Need either --client or --todo-id for posting
    if not args.client and not args.todo_id:
        parser.error("Either --client or --todo-id is required for posting")

    # Need either --summary or --content
    if not args.summary and not args.content:
        parser.error("Either --summary or --content is required")

    # Get content
    if args.summary:
        if not args.summary.exists():
            print(f"ERROR: Summary file not found: {args.summary}", file=sys.stderr)
            sys.exit(1)
        content = args.summary.read_text()
    else:
        content = args.content

    # Get todo ID
    if args.client:
        client_info = find_and_confirm_client(bc, args.client, args.yes)
        todo_id = client_info['todo_id']
    else:
        todo_id = args.todo_id
        print(f"Posting to todo ID: {todo_id}")

    # Dry run
    if args.dry_run:
        print(f"\n[DRY RUN] Would post to todo {todo_id}:")
        print("-" * 40)
        html = markdown_to_basecamp_html(content)
        # Show mentions that would be added
        if not args.no_mentions:
            mentions = bc.get_default_mentions_html()
            if args.also_tag:
                for name in args.also_tag:
                    sgid = bc.get_person_sgid(name)
                    if sgid:
                        mentions += f' <bc-attachment sgid="{sgid}"></bc-attachment>'
            if mentions:
                print(f"Team mentions: Toni, Tamara, Vanessa" + (f" + {', '.join(args.also_tag)}" if args.also_tag else ""))
        print("-" * 40)
        print(html[:500] + "..." if len(html) > 500 else html)
        return

    # Post it
    result = post_summary(bc, todo_id, content, also_tag=args.also_tag, no_mentions=args.no_mentions)

    # Post VA follow-up comment if requested
    if args.va_task:
        # Build VA mention tags (Bianca and Zeder by default)
        va_names = ["Bianca", "Zeder"]
        va_tags = []
        for name in va_names:
            sgid = bc.get_person_sgid(name)
            if sgid:
                va_tags.append(f'<bc-attachment sgid="{sgid}"></bc-attachment>')

        if va_tags:
            va_html = f"<div>{' '.join(va_tags)} {args.va_task}</div>"
            try:
                va_result = bc.post_comment(todo_id, va_html)
                print(f"\n✓ VA task posted: {args.va_task}")
                print(f"  Comment ID: {va_result['id']}")
            except Exception as e:
                print(f"\nWarning: Failed to post VA task: {e}", file=sys.stderr)

    if args.json:
        print(json.dumps(result, indent=2))


if __name__ == "__main__":
    main()
