---
name: build-feature
description: "Autonomous task loop that picks ready tasks, implements them, updates progress.txt, commits, and repeats. Use when asked to 'build feature', 'run the loop', or 'implement these tasks'."
---

# Build Feature Loop

Autonomous task execution loop that implements tasks one by one until complete.

---

## Task Sizing

**Each task must be completable in ONE iteration (~one context window).**

Each iteration spawns a fresh subagent with no memory of previous work. If a task is too big, the agent runs out of context before finishing.

### Right-sized tasks:
- Add a database column + migration
- Create a single UI component
- Implement one server action
- Add a filter to an existing list
- Write tests for one module

### Too big (split these):
- "Build the entire dashboard" → Split into: schema, queries, components, filters
- "Add authentication" → Split into: schema, middleware, login UI, session handling
- "Refactor the API" → Split into one task per endpoint

**Rule of thumb:** If you can't describe the change in 2-3 sentences, it's too big.

---

## Workflow

### 0. Initialize or load progress.txt

Check if `scripts/build-feature-loop/progress.txt` exists:

**If it doesn't exist or user is starting a new feature:**

1. Ask the user to describe the feature and its tasks (or reference a PRD)
2. Create the progress file with the task list:

```markdown
# Build Progress Log
Started: [today's date]
Feature: [feature name]

## Codebase Patterns
(Patterns discovered during this feature build)

---

## Tasks

- [ ] Task 1 title
  Description: What to do
  Files: path/to/file.ts
  Depends: none

- [ ] Task 2 title
  Description: What to do
  Depends: Task 1

- [ ] Task 3 title
  Description: What to do
  Depends: Task 1

- [ ] Task 4 title
  Description: What to do
  Depends: Task 2, Task 3

---

## Completed Work
(Entries added as tasks complete)
```

**If it exists:** Read it to understand current state.

### 1. Sync TodoWrite with progress.txt

Parse the tasks from progress.txt and update TodoWrite so the user can see progress:
- `[ ]` tasks → `pending`
- `[x]` tasks → `completed`
- Currently executing → `in_progress`

### 2. Find the next ready task

A task is **ready** when:
- Status is `[ ]` (not completed)
- All tasks listed in `Depends:` are completed `[x]`

**Pick the next task:**
- Prefer tasks related to what was just completed (same module/area)
- If no prior context, pick the first ready task

### 3. If no ready tasks

Check if all tasks are completed:
- If ALL tasks are `[x]`:
  1. Archive progress.txt:
     ```bash
     DATE=$(date +%Y-%m-%d)
     FEATURE="feature-name-here"
     mkdir -p scripts/build-feature-loop/archive/$DATE-$FEATURE
     mv scripts/build-feature-loop/progress.txt scripts/build-feature-loop/archive/$DATE-$FEATURE/
     ```
  2. Commit: `git add scripts/build-feature-loop && git commit -m "chore: archive progress for [feature-name]"`
  3. Stop and report "Build complete - all tasks finished!"

- If some tasks are blocked (dependencies not met): Report which tasks are blocked and why

### 4. Execute the task

Use the `Task` tool to spawn a subagent with this prompt:

```
Implement and verify: [task-title]

[task-description]

**Files:** [files from task]

FIRST: Read scripts/build-feature-loop/progress.txt - check the "Codebase Patterns" section for important context from previous iterations.

When complete:

1. Run quality checks: `npm run typecheck` and `npm test`
   - If either fails, FIX THE ISSUES and re-run until both pass
   - Do NOT proceed until quality checks pass

2. Update AGENTS.md files if you learned something important:
   - Check for AGENTS.md in directories where you edited files
   - Add learnings that future developers/agents should know (patterns, gotchas, dependencies)
   - This is LONG-TERM memory - things anyone editing this code should know

3. Update progress.txt:
   a. Mark this task complete: change `- [ ] [task-title]` to `- [x] [task-title]`
   b. Add entry to "## Completed Work" section:
      ```
      ## [Date] - [Task Title]
      - What was implemented
      - Files changed
      - **Learnings for future iterations:**
        - Patterns discovered
        - Gotchas encountered
      ---
      ```
   c. If you discovered a reusable pattern, add it to "## Codebase Patterns" at the top

4. Commit all changes with message: `feat: [Task Title]`
```

### 5. After subagent returns

1. Read the updated progress.txt
2. Update TodoWrite to reflect completed task
3. Check for more ready tasks → go to step 2
4. If all done or blocked → report status

---

## Progress File Location

The progress file should be at `scripts/build-feature-loop/progress.txt`.

### Task Format in progress.txt

```markdown
- [ ] Short task title
  Description: Detailed description of what to do
  Files: path/to/file1.ts, path/to/file2.ts
  Depends: none

- [ ] Another task
  Description: What to implement
  Files: src/components/Thing.tsx
  Depends: Short task title

- [x] Completed task
  Description: What was done
  Files: src/lib/util.ts
  Depends: none
```

**Dependency rules:**
- `Depends: none` - Can start immediately
- `Depends: Task A` - Blocked until Task A is `[x]`
- `Depends: Task A, Task B` - Blocked until both are `[x]`

---

## Task Discovery

While working, **liberally add new tasks** to progress.txt when you discover:
- Failing tests or test gaps
- Code that needs refactoring
- Missing error handling
- Documentation gaps
- TODOs or FIXMEs in the code
- Build/lint warnings
- Performance issues

Add them to the `## Tasks` section with appropriate `Depends:` relationships.

---

## Task Description Guidelines

Write descriptions that a future iteration can pick up without context:

```
- [ ] Add priority field to tasks table
  Description: Add 'priority' column with values 'high'|'medium'|'low', default 'medium'. Create migration.
  Files: prisma/schema.prisma, prisma/migrations/
  Depends: none
  Acceptance: npm run typecheck passes, migration runs successfully
```

### Dependency ordering (typical):
1. Schema/database changes (migrations)
2. Server actions / backend logic
3. UI components that use the backend
4. Integration / E2E tests

---

## Browser Verification

For UI tasks, specify the right verification method in the task description:

**Functional testing** (checking behavior, not appearance):
- Use `browser_snapshot` tool to read page content as accessibility tree
- Use for: button exists, text appears, form works

**Visual testing** (checking appearance):
- Use `browser_take_screenshot` tool to capture visual appearance
- Use for: layout, colors, styling, animations

---

## Quality Requirements

Before marking any task complete:
- `npm run typecheck` must pass
- `npm test` must pass
- Changes must be committed
- Progress must be logged

---

## Stop Condition

When no ready tasks remain AND all tasks are completed:
1. Output: "Build complete - all tasks finished!"
2. Summarize what was accomplished

---

## Important Notes

- Each Task subagent runs with fresh context
- progress.txt is the memory between iterations - keep it updated
- Prefer tasks in the same area as just-completed work for better context continuity
- The Task prompt MUST include instructions to update progress.txt and commit

---

## Pre-Flight Checklist

Before starting the loop, verify:

- [ ] progress.txt exists with tasks defined
- [ ] At least one task has `Depends: none` (can start)
- [ ] Each task is small enough for one iteration
- [ ] Each task has clear acceptance criteria
- [ ] UI tasks specify snapshot vs screenshot verification
- [ ] Task descriptions have enough detail to implement without context
