#!/usr/bin/env python3
"""
Generate CEO briefings from vault data.
Usage: python generate_briefing.py /path/to/vault --type weekly
"""

import argparse
from pathlib import Path
from datetime import datetime, timedelta
import json
import yaml
import re


def parse_frontmatter(content: str) -> tuple[dict, str]:
    """Parse YAML frontmatter from markdown content."""
    if content.startswith('---'):
        parts = content.split('---', 2)
        if len(parts) >= 3:
            try:
                frontmatter = yaml.safe_load(parts[1])
                body = parts[2].strip()
                return frontmatter or {}, body
            except yaml.YAMLError:
                pass
    return {}, content


def collect_transactions(vault: Path, start: datetime, end: datetime) -> list:
    """Collect transactions from the accounting folder."""
    transactions = []
    accounting_dir = vault / 'Accounting' / 'transactions'

    if not accounting_dir.exists():
        # Try current month file
        current_month = vault / 'Accounting' / 'Current_Month.md'
        if current_month.exists():
            content = current_month.read_text()
            # Parse table format
            lines = content.split('\n')
            for line in lines:
                if '|' in line and '$' in line:
                    # Simple extraction
                    match = re.search(r'\$([0-9,]+\.?\d*)', line)
                    if match:
                        amount = float(match.group(1).replace(',', ''))
                        transactions.append({'amount': amount})
        return transactions

    for file in accounting_dir.glob('*.md'):
        frontmatter, body = parse_frontmatter(file.read_text())
        if frontmatter.get('type') == 'transaction':
            date_str = frontmatter.get('date', '')
            try:
                date = datetime.strptime(date_str, '%Y-%m-%d')
                if start <= date <= end:
                    transactions.append(frontmatter)
            except ValueError:
                continue

    return transactions


def collect_completed_tasks(vault: Path, start: datetime, end: datetime) -> list:
    """Collect completed tasks from Done folder."""
    tasks = []
    done_dir = vault / 'Done'

    if not done_dir.exists():
        return tasks

    for file in done_dir.glob('*.md'):
        frontmatter, body = parse_frontmatter(file.read_text())
        # Check if completed in period
        completed = frontmatter.get('completed') or frontmatter.get('created')
        if completed:
            try:
                date = datetime.fromisoformat(completed.replace('Z', '+00:00'))
                date = date.replace(tzinfo=None)
                if start <= date <= end:
                    tasks.append({
                        'title': file.stem.replace('_', ' ').title(),
                        'completed': date,
                        'type': frontmatter.get('type', 'task')
                    })
            except ValueError:
                continue

    return tasks


def analyze_subscriptions(vault: Path) -> list:
    """Analyze subscriptions for optimization opportunities."""
    suggestions = []
    subs_file = vault / 'Accounting' / 'Subscriptions.md'

    if not subs_file.exists():
        return suggestions

    content = subs_file.read_text()
    lines = content.split('\n')

    for line in lines:
        if '|' in line:
            parts = [p.strip() for p in line.split('|')]
            if len(parts) >= 4:
                service = parts[1] if len(parts) > 1 else ''
                # Look for low usage indicators
                if 'low' in line.lower() or 'unused' in line.lower():
                    suggestions.append({
                        'service': service,
                        'reason': 'Low usage detected',
                        'action': 'Review and consider cancellation'
                    })

    return suggestions


def read_goals(vault: Path) -> dict:
    """Read business goals."""
    goals_file = vault / 'Business_Goals.md'
    if not goals_file.exists():
        return {'monthly_target': 10000}

    content = goals_file.read_text()

    # Extract monthly target
    match = re.search(r'Monthly goal:\s*\$?([0-9,]+)', content)
    monthly_target = float(match.group(1).replace(',', '')) if match else 10000

    return {'monthly_target': monthly_target}


def generate_weekly_briefing(vault_path: str) -> str:
    """Generate a weekly CEO briefing."""
    vault = Path(vault_path)

    # Calculate period (last week)
    today = datetime.now()
    # Find last Monday
    days_since_monday = today.weekday()
    week_end = today - timedelta(days=days_since_monday + 1)  # Last Sunday
    week_start = week_end - timedelta(days=6)  # Previous Monday

    # Collect data
    transactions = collect_transactions(vault, week_start, week_end)
    tasks = collect_completed_tasks(vault, week_start, week_end)
    subscriptions = analyze_subscriptions(vault)
    goals = read_goals(vault)

    # Calculate financials
    revenue = sum(t.get('amount', 0) for t in transactions if t.get('amount', 0) > 0)
    expenses = abs(sum(t.get('amount', 0) for t in transactions if t.get('amount', 0) < 0))
    weekly_target = goals['monthly_target'] / 4

    # Calculate day of month for MTD
    mtd_target_pct = (today.day / 30) * 100

    # Determine trend
    variance = ((revenue / weekly_target) - 1) * 100 if weekly_target > 0 else 0
    if variance > 5:
        trend = "ahead of"
        trend_emoji = "↑"
        status = "🟢"
    elif variance < -5:
        trend = "behind"
        trend_emoji = "↓"
        status = "🔴"
    else:
        trend = "on track with"
        trend_emoji = "→"
        status = "🟢"

    # Generate executive summary
    if revenue > weekly_target:
        summary = f"Strong week with revenue {variance:.0f}% {trend} target. "
    else:
        summary = f"Revenue {abs(variance):.0f}% {trend} weekly target. "

    summary += f"{len(tasks)} tasks completed this week."

    if subscriptions:
        summary += f" {len(subscriptions)} subscription optimization opportunities identified."

    # Format subscription suggestions
    sub_suggestions = ""
    if subscriptions:
        for sub in subscriptions[:3]:
            sub_suggestions += f"""
**{sub['service']}**
- {sub['reason']}
- Action: {sub['action']}
"""

    # Format completed tasks
    tasks_list = ""
    for task in tasks[:5]:
        tasks_list += f"- [x] {task['title']}\n"

    if not tasks_list:
        tasks_list = "- No tasks completed in period (check Done folder)\n"

    # Generate briefing
    briefing = f"""---
type: ceo_briefing
period_start: {week_start.strftime('%Y-%m-%d')}
period_end: {week_end.strftime('%Y-%m-%d')}
generated: {datetime.now().isoformat()}
briefing_type: weekly
---

# Monday Morning CEO Briefing
**Week of {today.strftime('%B %d, %Y')}**

---

## Executive Summary

{summary}

---

## Financial Health

### Revenue
| Metric | Value | vs Target | Trend |
|--------|-------|-----------|-------|
| This Week | ${revenue:,.2f} | {variance:+.0f}% | {trend_emoji} |
| Weekly Target | ${weekly_target:,.2f} | - | - |
| Net | ${revenue - expenses:,.2f} | - | {trend_emoji} |

### Expenses This Week
| Category | Amount |
|----------|--------|
| Total | ${expenses:,.2f} |

### Cash Flow
- **Net this week**: ${revenue - expenses:+,.2f}
- **Status**: {status} {'Positive' if revenue > expenses else 'Negative'}

---

## Operations

### Completed This Week ✅
{tasks_list}
### Pending Items
Check `/Needs_Action/` for current pending items.

### Blocked Items
Check `/Plans/` for any blocked tasks.

---

## Proactive Suggestions

### 💰 Cost Optimization
{sub_suggestions if sub_suggestions else "No immediate optimization opportunities identified."}

### 📅 This Week's Priorities
1. Review pending approvals in `/Pending_Approval/`
2. Process any items in `/Needs_Action/`
3. Check subscription usage

---

## Key Metrics Dashboard

| Metric | Target | Actual | Status |
|--------|--------|--------|--------|
| Weekly Revenue | ${weekly_target:,.0f} | ${revenue:,.0f} | {status} |
| Expenses | <${weekly_target*0.3:,.0f} | ${expenses:,.0f} | {'🟢' if expenses < weekly_target*0.3 else '🟡'} |
| Tasks Completed | 5 | {len(tasks)} | {'🟢' if len(tasks) >= 5 else '🟡'} |

---

*Generated by AI Employee*
*Next briefing: {(today + timedelta(days=7)).strftime('%B %d, %Y')}*
"""

    # Save briefing
    briefings_dir = vault / 'Briefings'
    briefings_dir.mkdir(exist_ok=True)

    filename = f"{today.strftime('%Y-%m-%d')}_Monday_Briefing.md"
    briefing_file = briefings_dir / filename
    briefing_file.write_text(briefing)

    print(f"✅ Weekly briefing generated")
    print(f"   Period: {week_start.strftime('%Y-%m-%d')} to {week_end.strftime('%Y-%m-%d')}")
    print(f"   Revenue: ${revenue:,.2f} ({variance:+.0f}% vs target)")
    print(f"   Tasks completed: {len(tasks)}")
    print(f"   Saved to: {briefing_file}")

    return str(briefing_file)


def generate_monthly_briefing(vault_path: str) -> str:
    """Generate a monthly business review."""
    # Similar implementation for monthly briefing
    vault = Path(vault_path)
    today = datetime.now()

    # Calculate previous month
    first_of_month = today.replace(day=1)
    last_month_end = first_of_month - timedelta(days=1)
    last_month_start = last_month_end.replace(day=1)

    # For now, return placeholder
    print(f"Monthly briefing for {last_month_start.strftime('%B %Y')}")
    print("(Full implementation would aggregate weekly data)")

    return ""


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='Generate CEO briefing')
    parser.add_argument('vault_path', help='Path to the Obsidian vault')
    parser.add_argument('--type', choices=['weekly', 'monthly', 'quarterly'],
                        default='weekly', help='Type of briefing to generate')

    args = parser.parse_args()

    if args.type == 'weekly':
        generate_weekly_briefing(args.vault_path)
    elif args.type == 'monthly':
        generate_monthly_briefing(args.vault_path)
    else:
        print(f"Briefing type '{args.type}' not yet implemented")
