#!/usr/bin/env python3
"""
Content Opportunity Analyzer

Generates content opportunity reports by querying for trending topics,
knowledge gaps, and viral potential in a specific niche.

Usage:
    python content_opportunity_analyzer.py "cardiology" --platforms youtube,twitter,linkedin
    python content_opportunity_analyzer.py "heart health" --timeframe 7d
"""

import argparse
import json
from datetime import datetime

# Opportunity Scoring Framework
SCORING_WEIGHTS = {
    "demand": 0.30,
    "gap": 0.25,
    "virality_potential": 0.25,
    "alignment": 0.20
}

PLATFORM_QUERIES = {
    "youtube": {
        "trending": "What {niche} videos are trending on YouTube this week? Include titles, channels, view counts.",
        "gaps": "What {niche} topics do YouTube viewers request in comments but aren't covered well?",
    },
    "twitter": {
        "trending": "What {niche} topics are trending on Twitter/X this week? Include viral threads and engagement.",
        "gaps": "What {niche} questions are people asking on Twitter that don't get good answers?",
    },
    "linkedin": {
        "trending": "What {niche} topics are getting high engagement on LinkedIn from professionals?",
        "gaps": "What {niche} insights are professionals looking for but not finding on LinkedIn?",
    },
    "reddit": {
        "trending": "What's being discussed in {niche}-related subreddits this week?",
        "gaps": "What questions in {niche} subreddits don't get satisfactory answers?",
    },
    "news": {
        "trending": "What {niche} news stories are making headlines this week?",
        "gaps": "What {niche} topics are underreported in mainstream media?",
    }
}

VIRAL_TRIGGERS = [
    "Fear/urgency (health scares, deadlines)",
    "Hope/inspiration (success stories, breakthroughs)",
    "Surprise (counterintuitive findings, myth-busting)",
    "Anger/outrage (injustice, misinformation)",
    "Practical value (how-tos, tips, hacks)",
    "Social currency (insider knowledge, first-to-know)",
    "Nostalgia/emotion (personal stories, journeys)"
]


def generate_opportunity_report(niche: str, topic: str, scores: dict) -> str:
    """Generate a formatted opportunity report."""
    total_score = sum(scores[k] * SCORING_WEIGHTS[k] for k in SCORING_WEIGHTS)
    
    priority = "HIGH" if total_score >= 7 else "MEDIUM" if total_score >= 5 else "LOW"
    
    report = f"""
# Content Opportunity: {topic}

**Niche**: {niche}
**Date Analyzed**: {datetime.now().strftime("%Y-%m-%d")}
**Overall Score**: {total_score:.1f}/10
**Priority**: {priority}

## Scores Breakdown

| Factor | Score | Weight | Weighted |
|--------|-------|--------|----------|
| Demand | {scores['demand']}/10 | {SCORING_WEIGHTS['demand']*100:.0f}% | {scores['demand']*SCORING_WEIGHTS['demand']:.2f} |
| Gap | {scores['gap']}/10 | {SCORING_WEIGHTS['gap']*100:.0f}% | {scores['gap']*SCORING_WEIGHTS['gap']:.2f} |
| Virality Potential | {scores['virality_potential']}/10 | {SCORING_WEIGHTS['virality_potential']*100:.0f}% | {scores['virality_potential']*SCORING_WEIGHTS['virality_potential']:.2f} |
| Alignment | {scores['alignment']}/10 | {SCORING_WEIGHTS['alignment']*100:.0f}% | {scores['alignment']*SCORING_WEIGHTS['alignment']:.2f} |

## Recommended Actions

- **Primary Platform**: [Determine based on where demand is highest]
- **Content Format**: [Video/Article/Thread based on topic complexity]
- **Urgency**: {"Act within 48 hours - trending now" if priority == "HIGH" else "Plan for next week" if priority == "MEDIUM" else "Add to backlog"}

## Research Queries Needed

To fully analyze this opportunity, run these queries using perplexity-search:

1. Demand: "What is the search volume and question frequency for {topic} in {niche}?"
2. Gap: "How well does existing content cover {topic}? What's missing?"
3. Virality: "What viral content exists about {topic}? What made it spread?"
4. Competition: "Who are the main content creators covering {topic}?"

---
Generated by Content Opportunity Analyzer
"""
    return report


def get_platform_queries(niche: str, platforms: list) -> dict:
    """Get all relevant queries for specified platforms."""
    queries = {}
    for platform in platforms:
        if platform in PLATFORM_QUERIES:
            queries[platform] = {
                query_type: template.format(niche=niche)
                for query_type, template in PLATFORM_QUERIES[platform].items()
            }
    return queries


def main():
    parser = argparse.ArgumentParser(description="Analyze content opportunities in a niche")
    parser.add_argument("niche", help="The niche/topic area to analyze (e.g., 'cardiology')")
    parser.add_argument("--platforms", default="youtube,twitter,linkedin,reddit,news",
                        help="Comma-separated list of platforms to analyze")
    parser.add_argument("--timeframe", default="7d",
                        help="Timeframe for trend analysis (7d, 14d, 30d)")
    parser.add_argument("--output", default="opportunity_report.md",
                        help="Output filename for the report")
    
    args = parser.parse_args()
    platforms = [p.strip().lower() for p in args.platforms.split(",")]
    
    print(f"\n📊 Content Opportunity Analyzer")
    print(f"=" * 50)
    print(f"Niche: {args.niche}")
    print(f"Platforms: {', '.join(platforms)}")
    print(f"Timeframe: {args.timeframe}")
    print(f"=" * 50)
    
    # Get platform-specific queries
    queries = get_platform_queries(args.niche, platforms)
    
    print(f"\n📋 Queries to run with perplexity-search:\n")
    for platform, platform_queries in queries.items():
        print(f"\n### {platform.upper()}")
        for query_type, query in platform_queries.items():
            print(f"  [{query_type}]: {query}")
    
    print(f"\n\n🎯 Viral Triggers to Look For:")
    for trigger in VIRAL_TRIGGERS:
        print(f"  • {trigger}")
    
    print(f"\n\n📝 Next Steps:")
    print(f"  1. Run each query using perplexity-search")
    print(f"  2. Score each topic found (demand, gap, virality, alignment)")
    print(f"  3. Generate opportunity reports for top topics")
    print(f"  4. Prioritize and add to content calendar")
    
    # Example report generation (with placeholder scores)
    example_scores = {"demand": 8, "gap": 7, "virality_potential": 6, "alignment": 9}
    example_report = generate_opportunity_report(args.niche, "[Example Topic]", example_scores)
    
    with open(args.output, 'w') as f:
        f.write(f"# Content Opportunity Analysis: {args.niche}\n\n")
        f.write(f"Generated: {datetime.now().strftime('%Y-%m-%d %H:%M')}\n\n")
        f.write(f"## Queries for Research\n\n")
        f.write(json.dumps(queries, indent=2))
        f.write(f"\n\n## Example Opportunity Report Template\n")
        f.write(example_report)
    
    print(f"\n✅ Template saved to: {args.output}")


if __name__ == "__main__":
    main()
