# Snapshot Query Guide

Query and analyze snapshot log files generated by cursor-ide-browser to find element references for browser automation.

## Installation

Run without installation using `uvx`:
```bash
uvx snapshot-query <file_path> <command> [args]
```

Or install via pip:
```bash
pip install snapshot-query
```

## Core Concepts

### Snapshot Files

Snapshot files are YAML-formatted accessibility tree structures saved by cursor-ide-browser:
- **Location**: `C:\Users\{username}\.cursor\browser-logs\`
- **Format**: `snapshot-{ISO 8601 timestamp}.log`
- **Structure**: Tree of elements with `role`, `ref`, `name`, and `children`

### Element Properties

- **`role`**: Element type (button, link, textbox, etc.)
- **`ref`**: Unique reference identifier (used for browser interactions)
- **`name`**: Element text/label (optional)
- **`children`**: Child elements (optional)

## Quick Start

Find button reference for browser automation:
```bash
browser_snapshot()  # Get page snapshot
uvx snapshot-query snapshot.log find-name "search"  # Find element
# Extract ref from output, then use: browser_click(element="Search", ref="ref-xxxxx")
```

Find input field:
```bash
uvx snapshot-query snapshot.log find-role textbox
# Or: uvx snapshot-query snapshot.log find-name "username"
```

List all interactive elements:
```bash
uvx snapshot-query snapshot.log interactive
```

## Commands

### Basic Queries

- `find-name "text"` - Fuzzy search by name
- `find-name-exact "text"` - Exact match
- `find-name-bm25 "text" [top_k]` - Relevance-ranked (BM25)
- `find-role button` - Find by role
- `find-ref ref-xxxxx` - Find by reference ID

### Advanced Queries

- `find-grep "pattern" [field]` - Regular expression (field: name/role/ref)
- `find-selector "selector"` - CSS/jQuery selector (e.g., `button[name='search']`)
- `find-text "text"` - Text content search

### Utilities

- `interactive` - List interactive elements
- `count` - Count by type
- `path ref-xxxxx` - Show element path
- `all-refs` - List all refs

## Python API

```python
from snapshot_query import SnapshotQuery

query = SnapshotQuery("snapshot.log")
buttons = query.find_by_role("button")
element = query.find_by_ref("ref-xxxxx")

# Advanced: BM25, CSS selectors, regex
results = query.find_by_name_bm25("search", top_k=5)
results = query.find_by_selector("button[name='search']")
results = query.find_by_regex("^search$", field="name")
```

## MCP Integration

Configure in `~/.cursor/mcp.json`:
```json
{
  "mcpServers": {
    "snapshot-query": {
      "command": "snapshot-query-mcp",
      "args": []
    }
  }
}
```

### Available MCP Tools

- `mcp_snapshot-query_find_by_name` - Find elements by name (fuzzy match)
- `mcp_snapshot-query_find_by_name_bm25` - Find by name with BM25 relevance ranking
- `mcp_snapshot-query_find_by_role` - Find elements by role
- `mcp_snapshot-query_find_by_ref` - Find element by reference ID
- `mcp_snapshot-query_find_by_text` - Find elements containing text
- `mcp_snapshot-query_find_by_regex` - Find using regular expression
- `mcp_snapshot-query_find_by_selector` - Find using CSS/jQuery selector
- `mcp_snapshot-query_find_interactive_elements` - List all interactive elements
- `mcp_snapshot-query_count_elements` - Count elements by type
- `mcp_snapshot-query_get_element_path` - Get element path in tree
- `mcp_snapshot-query_extract_all_refs` - Extract all reference IDs

### MCP Usage Examples

```javascript
// Find element by name
mcp_snapshot-query_find_by_name(
  file_path="snapshot-2026-01-09T15-00-42-849Z.log",
  name="搜索"
)

// Find with relevance ranking
mcp_snapshot-query_find_by_name_bm25(
  file_path="snapshot.log",
  name="login",
  top_k=5
)

// Find by role
mcp_snapshot-query_find_by_role(
  file_path="snapshot.log",
  role="button"
)

// Find using CSS selector
mcp_snapshot-query_find_by_selector(
  file_path="snapshot.log",
  selector="button[name='search']"
)
```

## Integration with Browser Automation

### Complete Workflow

```javascript
// 1. Navigate and snapshot
browser_navigate(url="https://example.com")
browser_snapshot()
// Snapshot saved to: C:\Users\{username}\.cursor\browser-logs\snapshot-{timestamp}.log

// 2. Query snapshot to find element
const result = mcp_snapshot-query_find_by_name(
  file_path="snapshot-2026-01-09T15-00-42-849Z.log",
  name="搜索"
)

// 3. Use ref from query result
browser_click(element="搜索", ref=result.ref)
```

### Finding Multiple Elements

```javascript
browser_snapshot()

// Find username input
const username = mcp_snapshot-query_find_by_name(
  file_path="snapshot.log",
  name="username"
)

// Find password input
const password = mcp_snapshot-query_find_by_name(
  file_path="snapshot.log",
  name="password"
)

// Use refs
browser_type(element="Username", ref=username.ref, text="user")
browser_type(element="Password", ref=password.ref, text="pass")
```

### Using BM25 for Better Results

When element name is ambiguous or has variations:

```javascript
browser_snapshot()

// BM25 finds most relevant matches
const results = mcp_snapshot-query_find_by_name_bm25(
  file_path="snapshot.log",
  name="submit button",
  top_k=3
)

// Use top result
browser_click(element="Submit", ref=results[0].ref)
```

## Best Practices

1. **Prefer BM25 for relevance ranking** - `find-name-bm25` is better than `find-name` for ambiguous queries
2. **Use CSS selectors for flexibility** - `find-selector "button[name='search']"` for precise queries
3. **Refs change with each snapshot** - Get a new snapshot before using refs
4. **Use path command to understand hierarchy** - `path ref-xxxxx` shows element location in tree
5. **Query after snapshot** - Always query the most recent snapshot file

## Common Patterns

### Find and Click Button

```bash
browser_snapshot()
uvx snapshot-query snapshot.log find-name "submit"
browser_click(element="Submit", ref="ref-from-output")
```

Or with MCP:
```javascript
browser_snapshot()
const button = mcp_snapshot-query_find_by_name_bm25(
  file_path="snapshot.log",
  name="submit",
  top_k=1
)
browser_click(element="Submit", ref=button.ref)
```

### Fill Form

```bash
uvx snapshot-query snapshot.log find-name "username"  # Get ref-username
uvx snapshot-query snapshot.log find-name "password"  # Get ref-password
browser_type(element="Username", ref="ref-username", text="user")
browser_type(element="Password", ref="ref-password", text="pass")
```

Or with MCP:
```javascript
browser_snapshot()
const username = mcp_snapshot-query_find_by_name(file_path="snapshot.log", name="username")
const password = mcp_snapshot-query_find_by_name(file_path="snapshot.log", name="password")
browser_type(element="Username", ref=username.ref, text="user")
browser_type(element="Password", ref=password.ref, text="pass")
```

### Find All Buttons

```bash
uvx snapshot-query snapshot.log find-role button
```

Or with MCP:
```javascript
const buttons = mcp_snapshot-query_find_by_role(file_path="snapshot.log", role="button")
// buttons contains all button elements with their refs
```

### Find Interactive Elements

```bash
uvx snapshot-query snapshot.log interactive
```

Or with MCP:
```javascript
const interactive = mcp_snapshot-query_find_interactive_elements(file_path="snapshot.log")
// Returns all clickable, typeable, selectable elements
```

## Troubleshooting

- **Can't find element**: Try `find-name-bm25` or check spelling
- **Ref not working**: Refs change with each snapshot - get a new snapshot
- **Too many results**: Use `find-name-bm25` with `top_k` or more specific selectors
- **Element not found**: Verify snapshot file path is correct and file exists
- **Multiple matches**: Use BM25 with `top_k=1` to get best match, or use CSS selector for precision

## Command Line vs MCP

**Command line** (`uvx snapshot-query`):
- Good for quick one-off queries
- Outputs text that needs parsing
- Requires file path as argument

**MCP tools** (`mcp_snapshot-query_*`):
- Better for programmatic integration
- Returns structured data
- Can be used directly in automation workflows
- Recommended for browser automation integration
