# イミュータブルデータモデルの原則

このドキュメントは、データモデリングにおける基本原則を定義します。

## 参考資料

- [イミュータブルデータモデル - Scrapbox](https://scrapbox.io/kawasima/イミュータブルデータモデル)
- 川島義隆氏の提唱するデータモデリング手法

## 基本概念

### 1. エンティティの分類

データモデルにおけるエンティティは、以下の2種類に分類されます。

#### リソース（Resource）

**定義:**
- 継続的に存在するもの
- 時間経過で状態が変化しうるもの
- ビジネス上の「もの」「人」「組織」

**特徴:**
- 更新操作が発生する
- 現在の状態を保持する
- 履歴は別途イベントとして管理

**例:**
- 顧客（Customer）
- 商品（Product）
- 社員（Employee）
- 契約（Contract）
- プロジェクト（Project）

#### イベント（Event）

**定義:**
- 特定時点で発生した事実
- 一度発生したら変更されないもの（不変）
- ビジネス上の「できごと」「行為」

**特徴:**
- **必ず1つの日時属性を持つ**（最重要）
- 更新操作は発生しない（追加と参照のみ）
- 削除も基本的に行わない（論理削除も不要）

**例:**
- 注文（Order）
- 入金（Payment）
- 出荷（Shipment）
- 請求書送付（InvoiceSend）
- 社員異動（EmployeeTransfer）

### 2. イベントの日時属性に関するルール

**重要な原則:**
- イベントエンティティには**1つの日時属性のみ**を持たせる
- 複数の日時を持ちたい場合は、イベントを分割する

**理由:**
- 1つのイベントは1つの時点を表すべき
- 複数の日時 = 複数のイベントが混在している証拠

**悪い例:**
```
注文エンティティ {
  注文日時
  出荷日時  ← これは別のイベント
  配達日時  ← これも別のイベント
}
```

**良い例:**
```
注文エンティティ {
  注文日時
}

出荷エンティティ {
  出荷日時
}

配達エンティティ {
  配達日時
}
```

### 3. 日時属性の管理

**推奨事項:**
- 日時はUTC（協定世界時）で管理
- タイムゾーン情報が必要な場合は別属性で保持
- 日時型は`TIMESTAMP`または`DATETIME`を使用

**例:**
```sql
CREATE TABLE invoice_send (
  event_id INT PRIMARY KEY,
  customer_id INT,
  send_datetime TIMESTAMP NOT NULL,  -- UTC
  send_timezone VARCHAR(50),          -- 'Asia/Tokyo'等
  ...
);
```

## 高度なテクニック

### 4. 隠れたイベントの抽出

リソースに「更新日時」などがある場合、その背後にイベントが隠れている可能性があります。

**検出方法:**
1. リソースの属性に「〜日時」「〜日」がないか確認
2. 「更新」「変更」を表す属性がないか確認
3. それらが業務上意味のある時点なら、イベント化を検討

**例1: 社員情報**

**悪い設計:**
```
社員 {
  社員ID
  氏名
  部署
  役職
  更新日時  ← これは何を表す？
}
```

**改善:**
```
社員（リソース）{
  社員ID
  氏名
}

社員異動（イベント）{
  異動日時
  社員ID
  異動前部署
  異動後部署
  異動前役職
  異動後役職
}
```

**例2: 商品情報**

**悪い設計:**
```
商品 {
  商品ID
  商品名
  価格
  価格変更日  ← イベントが隠れている
}
```

**改善:**
```
商品（リソース）{
  商品ID
  商品名
  現在価格  ← 最新の価格
}

価格変更（イベント）{
  価格変更日時
  商品ID
  変更前価格
  変更後価格
}
```

### 5. 交差エンティティの導入

以下の場合に交差エンティティを導入します。

#### ケース1: 多対多の関係

**例: 学生と講義**

**悪い設計:**
```
学生 ←→ 講義  （多対多）
```

**改善:**
```
学生 → 履修 → 講義

履修（交差エンティティ）{
  学生ID
  講義ID
  履修日時
  成績
}
```

#### ケース2: 関連自体が重要な属性を持つ

**例: 社員と部署**

**単純な設計:**
```
社員 → 部署  （多対一）
```

**改善（交差エンティティ導入）:**
```
社員 → 所属 → 部署

所属（交差エンティティ）{
  社員ID
  部署ID
  配属日
  役職
  上司ID
}
```

#### 判断基準

以下の質問に1つでも「はい」なら交差エンティティを検討：
- [ ] 関連に日時情報が必要か？
- [ ] 関連に数量・金額などの属性が必要か？
- [ ] 関連自体が業務上の意味を持つか？
- [ ] 多対多の関係か？

### 6. カーディナリティの決定

**1:1（一対一）**
- 例: 国 ←→ 首都
- 例: 社員 ←→ 社員証

**1:N（一対多）**
- 例: 顧客 → 注文
- 例: 部署 → 社員

**M:N（多対多）**
- 必ず交差エンティティで解消する
- 例: 学生 ← 履修 → 講義

## モデリングの手順

### ステップ1: エンティティ候補の抽出

ユースケースから名詞と動詞を抽出

### ステップ2: リソースとイベントに分類

- 「〜がある」「〜を管理する」→ リソース
- 「〜した」「〜が発生した」→ イベント

### ステップ3: イベントの日時属性を確認

- 各イベントに1つだけ日時属性があることを確認
- 複数ある場合はイベントを分割

### ステップ4: 隠れたイベントを抽出

- リソースの「更新日時」などをチェック

### ステップ5: 関連を分析

- カーディナリティを決定
- 多対多なら交差エンティティを導入

### ステップ6: 属性を洗い出し

- 各エンティティに必要な属性を定義

### ステップ7: 検証

- イベントの日時属性が1つか確認
- 外部キーが適切か確認
- 正規化ルールに違反していないか確認

## よくある間違い

### ❌ イベントに複数の日時

```
注文 {
  注文日時
  支払日時
  出荷日時
}
```

### ✅ イベントを分割

```
注文 { 注文日時 }
支払 { 支払日時 }
出荷 { 出荷日時 }
```

### ❌ リソースに履歴を持たせる

```
社員 {
  現在の部署
  前回の部署
  前々回の部署
}
```

### ✅ イベントで履歴管理

```
社員 { 社員ID }
異動 { 異動日時, 部署 }
```

### ❌ 多対多を直接表現

```
学生 ←→ 講義
```

### ✅ 交差エンティティで解消

```
学生 → 履修 → 講義
```

## チェックリスト

モデリング完了時に以下を確認：

- [ ] すべてのイベントに1つだけ日時属性がある
- [ ] リソースに不要な日時属性がない
- [ ] 隠れたイベントを抽出した
- [ ] 多対多の関係が交差エンティティで解消されている
- [ ] 外部キー制約が適切に定義されている
- [ ] 各エンティティの役割が明確

## 参考: RDBMSでの実装

### リソースのテーブル

```sql
CREATE TABLE customer (
  customer_id INT PRIMARY KEY AUTO_INCREMENT,
  name VARCHAR(100) NOT NULL,
  address VARCHAR(255),
  phone VARCHAR(20),
  email VARCHAR(100),
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  INDEX idx_name (name)
);
```

### イベントのテーブル

```sql
CREATE TABLE invoice_send (
  event_id INT PRIMARY KEY AUTO_INCREMENT,
  customer_id INT NOT NULL,
  send_datetime TIMESTAMP NOT NULL,  -- 日時属性（1つのみ）
  send_method VARCHAR(50),
  FOREIGN KEY (customer_id) REFERENCES customer(customer_id),
  INDEX idx_send_datetime (send_datetime),
  INDEX idx_customer (customer_id)
);
```

### 注意点

- イベントテーブルには`UPDATE`文を発行しない
- 削除も基本的に行わない（必要なら論理削除フラグではなく削除イベントを記録）
