# /// script
# requires-python = ">=3.13"
# dependencies = [
#   "ddgs>=9.10.0",
#   "html2text",
#   "requests",
# ]
# ///
"""
DDGS Text Search Command Line Interface

Search the web using the DDGS library.
"""

import argparse
import sys
import io
import requests
import html2text
from ddgs import DDGS

# Set stdout and stderr to use UTF-8 encoding
sys.stdout = io.TextIOWrapper(sys.stdout.buffer, encoding="utf-8", errors="replace")
sys.stderr = io.TextIOWrapper(sys.stderr.buffer, encoding="utf-8", errors="replace")


def fetch_webpage_content(url, max_length, proxy=None):
    """Fetch webpage content and convert to markdown text."""
    try:
        # Set up proxy if provided
        proxies = None
        if proxy:
            proxies = {
                'http': proxy,
                'https': proxy,
            }
        
        # Add headers to mimic a real browser and avoid 403 errors
        headers = {
            'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
            'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language': 'en-US,en;q=0.5',
            'Accept-Encoding': 'gzip, deflate',
            'DNT': '1',
            'Connection': 'keep-alive',
            'Upgrade-Insecure-Requests': '1',
        }
        
        # Fetch the webpage with headers
        response = requests.get(url, proxies=proxies, headers=headers, timeout=10)
        response.raise_for_status()
        
        # Convert HTML to markdown text
        h = html2text.HTML2Text()
        h.ignore_links = True
        h.ignore_images = True
        h.body_width = 0  # No line wrapping
        
        markdown_text = h.handle(response.text)
        
        # Limit the output to a reasonable size
        if len(markdown_text) > max_length:
            markdown_text = markdown_text[:max_length] + "\n[... truncated ...]"
        
        return markdown_text.strip()
    
    except requests.exceptions.RequestException as e:
        return f"[Error fetching webpage: {e}]"
    except Exception as e:
        return f"[Error processing webpage: {e}]"


def main():
    parser = argparse.ArgumentParser(
        description="Search the web using the DDGS library.",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  %(prog)s "python programming"                     # Basic search (5 results)
  %(prog)s "python programming" -n 20               # More results
  %(prog)s "python programming" -r uk-en            # UK English results
  %(prog)s "python programming" -s off              # Disable safesearch
  %(prog)s "python programming" -t w                # Results from last week
  %(prog)s "python programming" -p 2                # Page 2 of results
  %(prog)s "python programming" -b "bing,google"    # Use specific backends
  %(prog)s "python programming" --brief             # Show only snippets (no full text)
        """,
    )

    # Required positional argument
    parser.add_argument("query", type=str, help="Search query")

    # Optional arguments matching the DDGS.text() parameters
    parser.add_argument(
        "-n",
        "--max-results",
        type=int,
        default=5,
        help="Maximum number of results (default: 5)",
    )

    parser.add_argument(
        "-r",
        "--region",
        type=str,
        default="us-en",
        help="Region/language code: us-en, uk-en, ru-ru, etc. (default: us-en)",
    )

    parser.add_argument(
        "-s",
        "--safesearch",
        type=str,
        choices=["on", "moderate", "off"],
        default="off",
        help="Safesearch level: on, moderate, off (default: off)",
    )

    parser.add_argument(
        "-t",
        "--timelimit",
        type=str,
        help="Time limit: d, w, m, y. (default: None)",
    )

    parser.add_argument(
        "-p", "--page", type=int, default=1, help="Page number of results (default: 1)"
    )

    parser.add_argument(
        "-b",
        "--backend",
        type=str,
        default="auto",
        help="Single or comma-delimited backends (default: auto)",
    )

    parser.add_argument(
        "--proxy",
        type=str,
        default=None,
        help="Proxy URL (default: None)",
    )

    parser.add_argument(
        "--brief",
        action="store_true",
        help="Suppress full text extraction from webpages (show only snippets)",
    )

    args = parser.parse_args()

    try:
        # Initialize DDGS with proxy if provided
        engine = DDGS(proxy=args.proxy) if args.proxy else DDGS()

        # Perform search with all specified parameters
        results = engine.text(
            query=args.query,
            region=args.region,
            safesearch=args.safesearch,
            timelimit=args.timelimit,
            max_results=args.max_results,
            page=args.page,
            backend=args.backend,
        )

        # Display results
        for i, r in enumerate(results, 1):
            print(f"--- Result {i} ---")

            # Safely print each field with encoding handling
            title = r.get("title", "N/A")
            href = r.get("href", "N/A")
            body = r.get("body", "N/A")

            print(f"Title: {title}")
            print(f"Href: {href}")
            print(f"Body: {body}")
            
            # Fetch and display full webpage content if not in brief mode
            if not args.brief and href != "N/A" and href.startswith(('http://', 'https://')):
                print("\nFull Text:")
                print("-" * 40)
                full_text = fetch_webpage_content(href, max_length=2000, proxy=args.proxy)
                print(full_text)
                print("-" * 40)
            
            print()

    except Exception as e:
        # Ensure error message is properly encoded
        error_msg = f"Error: {e}"
        print(error_msg, file=sys.stderr)
        sys.exit(1)


if __name__ == "__main__":
    main()
