# サンプルデータ設計パターン

## 基本原則

### リアルなシナリオ
実際のビジネスケースを再現：
- 3〜5件のリソースデータ
- 複数のシナリオパターン
- 時系列に沿ったイベント発生

### エッジケース
イミュータブルモデルの特徴を示す：
- 分割払い（複数回の入金）
- 担当者交代
- リスク変動
- 期日超過

## データ投入順序

必ず依存関係に従う：

1. マスターデータ（業界、役割、タイプ等）
2. リソースデータ（顧客、人、プロジェクト等）
3. ジャンクションデータ（タグ付け）
4. イベントデータ（時系列順）

## シナリオパターン

### パターン1: 正常フロー
最も一般的な流れを示す：

```sql
-- プロジェクト開始
INSERT INTO PROJECT_START (...) VALUES (...);

-- 担当者アサイン
INSERT INTO PERSON_ASSIGN (...) VALUES (...);

-- プロジェクト完了
INSERT INTO PROJECT_COMPLETE (...) VALUES (...);
```

### パターン2: エッジケース（交代）
担当者交代を示す：

```sql
-- 初回アサイン
INSERT INTO PERSON_ASSIGN (ProjectID, PersonID, RoleID, ...)
VALUES (1, 100, 1, ...);

-- 交代イベント
INSERT INTO PERSON_REPLACE (ProjectID, OldPersonID, NewPersonID, RoleID, ...)
VALUES (1, 100, 101, 1, ...);
```

### パターン3: エッジケース（分割）
分割払いを示す：

```sql
-- 1回目の入金
INSERT INTO PAYMENT (InvoiceID, PaymentAmount, ...)
VALUES (1, 50000.00, ...);

-- 2回目の入金
INSERT INTO PAYMENT (InvoiceID, PaymentAmount, ...)
VALUES (1, 50000.00, ...);
```

### パターン4: エッジケース（状態変化）
リスクランクの変動を示す：

```sql
-- 初回評価（低リスク）
INSERT INTO RISK_EVALUATE (ProjectID, RiskRank, ...)
VALUES (1, '低', ...);

-- 問題発生（高リスク）
INSERT INTO RISK_EVALUATE (ProjectID, RiskRank, ...)
VALUES (1, '高', ...);

-- 対策後（中リスク）
INSERT INTO RISK_EVALUATE (ProjectID, RiskRank, ...)
VALUES (1, '中', ...);
```

## データ量の目安

### リソース
- 主要エンティティ: 3〜5件
- マスター: 必要最小限（3〜10件）
- 関連エンティティ: 5〜10件

### イベント
- シナリオごとに5〜15イベント
- 合計20〜50イベント

## 日時の設定

### タイムゾーン付き
```sql
INSERT INTO ... VALUES (..., '2024-04-01 10:00:00+09');
```

### 時系列順
イベントは必ず時系列順に投入：

```sql
-- ✅ 正しい順序
INSERT INTO PROJECT_START (...) VALUES (..., '2024-04-01 10:00:00+09');
INSERT INTO PERSON_ASSIGN (...) VALUES (..., '2024-04-01 11:00:00+09');
INSERT INTO RISK_EVALUATE (...) VALUES (..., '2024-05-01 15:00:00+09');

-- ❌ 時系列がバラバラ
INSERT INTO RISK_EVALUATE (...) VALUES (..., '2024-05-01 15:00:00+09');
INSERT INTO PROJECT_START (...) VALUES (..., '2024-04-01 10:00:00+09');
```

## コメントによる説明

各シナリオにコメントを付与：

```sql
-- ================================================
-- シナリオ1: 顧客1（ABC商事）の請求書1 → 正常に支払い完了
-- ================================================
INSERT INTO INVOICE_SEND (...) VALUES (...);
INSERT INTO PAYMENT (...) VALUES (...);

-- ================================================
-- シナリオ2: 顧客1（ABC商事）の請求書2 → 期日超過 → 確認状送付 → 遅延入金
-- ================================================
INSERT INTO INVOICE_SEND (...) VALUES (...);
-- 期日（4/30）を過ぎても入金なし → 確認状送付（5/7）
INSERT INTO CONFIRMATION_SEND (...) VALUES (...);
-- 確認状後に入金（5/10）
INSERT INTO PAYMENT (...) VALUES (...);
```

## データサマリー

ファイル末尾に全体サマリーを記載：

```sql
-- ================================================
-- データサマリー
-- ================================================
-- リソース:
--   顧客: 3社
--   請求書: 5件
-- イベント:
--   請求書送付: 5件
--   入金: 4件
--   確認状送付: 2件
--
-- 状態パターン:
--   ✅ 正常支払い: 請求書1
--   ⚠️ 遅延支払い: 請求書2
--   💰 分割支払い: 請求書3
--   ❌ 未入金（督促済み）: 請求書4
--   ⏳ 未入金（期日前）: 請求書5
-- ================================================
```
