#!/usr/bin/env python3
"""
Track pharmaceutical deals, M&A, and partnerships.

Usage:
    python deal_tracker.py --feed --days 30
    python deal_tracker.py --target EGFR --type licensing
    python deal_tracker.py --summary --year 2024 --oncology
"""

import argparse
import json
import sys
from datetime import datetime, timedelta
from typing import Optional, Dict, Any, List

try:
    import requests
except ImportError:
    print("Error: requests required. Install with: pip install requests")
    sys.exit(1)


class DealTracker:
    """Track pharmaceutical industry deals."""

    def __init__(self):
        self.results = {
            "query_type": None,
            "deals": [],
            "summary": {},
            "timestamp": datetime.now().isoformat()
        }

        # Mock deal database (in production, connect to real sources)
        self.MOCK_DEALS = [
            {
                "date": "2024-12-15",
                "type": "acquisition",
                "acquirer": "Pfizer",
                "target": "Seagen",
                "value": 43000000000,
                "currency": "USD",
                "stage": "Approved",
                "therapeutic_area": "Oncology",
                "assets": ["ADC portfolio"],
                "status": "Completed"
            },
            {
                "date": "2024-12-10",
                "type": "acquisition",
                "acquirer": "BMS",
                "target": "Mirati Therapeutics",
                "value": 4800000000,
                "currency": "USD",
                "stage": "Phase 3",
                "therapeutic_area": "Oncology",
                "assets": ["Adagrasib (KRAS G12C)"],
                "status": "Completed"
            },
            {
                "date": "2024-12-08",
                "type": "licensing",
                "licensee": "AstraZeneca",
                "licensor": "Hengrui Medicine",
                "value": 900000000,
                "upfront": 100000000,
                "currency": "USD",
                "stage": "Phase 2",
                "therapeutic_area": "Oncology",
                "targets": ["KRAS G12D"],
                "status": "Announced"
            },
            {
                "date": "2024-12-05",
                "type": "licensing",
                "licensee": "Merck",
                "licensor": "Daiichi Sankyo",
                "value": 1800000000,
                "upfront": 300000000,
                "currency": "USD",
                "stage": "Phase 3",
                "therapeutic_area": "Oncology",
                "assets": ["ADC platform"],
                "status": "Announced"
            },
            {
                "date": "2024-12-01",
                "type": "partnership",
                "partner1": "Genentech",
                "partner2": "Roche",
                "value": 500000000,
                "currency": "USD",
                "stage": "Preclinical",
                "therapeutic_area": "Oncology",
                "targets": ["EGFR", "MET"],
                "status": "Announced"
            },
            {
                "date": "2024-11-28",
                "type": "licensing",
                "licensee": "Novartis",
                "licensor": "BridgeBio",
                "value": 790000000,
                "upfront": 150000000,
                "currency": "USD",
                "stage": "Phase 2",
                "therapeutic_area": "Oncology",
                "targets": ["SHP2"],
                "status": "Announced"
            },
            {
                "date": "2024-11-20",
                "type": "acquisition",
                "acquirer": "Eli Lilly",
                "target": "Morphic",
                "value": 1300000000,
                "currency": "USD",
                "stage": "Phase 2",
                "therapeutic_area": "Immunology",
                "targets": [" integrin"],
                "status": "Announced"
            },
            {
                "date": "2024-11-15",
                "type": "licensing",
                "licensee": "Gilead",
                "licensor": "Nurix",
                "value": 2300000000,
                "upfront": 450000000,
                "currency": "USD",
                "stage": "Phase 1",
                "therapeutic_area": "Oncology",
                "targets": ["BRD9"],
                "status": "Announced"
            }
        ]

    def get_feed(self, days: int = 30, deal_type: str = None) -> List[Dict]:
        """Get recent deals feed."""
        try:
            cutoff_date = datetime.now() - timedelta(days=days)

            filtered_deals = []
            for deal in self.MOCK_DEALS:
                deal_date = datetime.strptime(deal["date"], "%Y-%m-%d")
                if deal_date >= cutoff_date:
                    if not deal_type or deal["type"] == deal_type:
                        filtered_deals.append(deal)

            filtered_deals.sort(key=lambda x: x["date"], reverse=True)

            self.results["query_type"] = "feed"
            self.results["deals"] = filtered_deals
            self.results["summary"] = self._summarize_deals(filtered_deals)

            print(f"✓ Found {len(filtered_deals)} deals in last {days} days")
            return filtered_deals

        except Exception as e:
            print(f"✗ Feed error: {e}")
            return []

    def search_by_target(self, target: str, deal_type: str = None) -> List[Dict]:
        """Search deals by target."""
        try:
            target_lower = target.lower()

            filtered_deals = []
            for deal in self.MOCK_DEALS:
                # Check in targets list
                if "targets" in deal:
                    if any(target_lower in t.lower() for t in deal["targets"]):
                        if not deal_type or deal["type"] == deal_type:
                            filtered_deals.append(deal)

                # Check in assets
                if "assets" in deal:
                    if any(target_lower in a.lower() for a in deal["assets"]):
                        if not deal_type or deal["type"] == deal_type:
                            filtered_deals.append(deal)

            filtered_deals.sort(key=lambda x: x["date"], reverse=True)

            self.results["query_type"] = "target_search"
            self.results["deals"] = filtered_deals
            self.results["target"] = target

            print(f"✓ Found {len(filtered_deals)} deals for {target}")
            return filtered_deals

        except Exception as e:
            print(f"✗ Target search error: {e}")
            return []

    def get_summary(self, year: int = None, therapeutic_area: str = None) -> Dict:
        """Get deal summary statistics."""
        try:
            filtered_deals = self.MOCK_DEALS

            # Filter by year
            if year:
                filtered_deals = [
                    d for d in filtered_deals
                    if datetime.strptime(d["date"], "%Y-%m-%d").year == year
                ]

            # Filter by therapeutic area
            if therapeutic_area:
                area_lower = therapeutic_area.lower()
                filtered_deals = [
                    d for d in filtered_deals
                    if area_lower in d.get("therapeutic_area", "").lower()
                ]

            summary = {
                "year": year,
                "therapeutic_area": therapeutic_area,
                "total_deals": len(filtered_deals),
                "by_type": self._group_by_type(filtered_deals),
                "by_stage": self._group_by_stage(filtered_deals),
                "total_value": sum(d.get("value", 0) for d in filtered_deals),
                "top_deals": sorted(filtered_deals,
                                   key=lambda x: x.get("value", 0),
                                   reverse=True)[:10]
            }

            self.results["query_type"] = "summary"
            self.results["summary"] = summary

            print(f"✓ Summary: {len(filtered_deals)} deals")
            return summary

        except Exception as e:
            print(f"✗ Summary error: {e}")
            return {}

    def _summarize_deals(self, deals: List[Dict]) -> Dict:
        """Summarize deal list."""
        by_type = {}
        for deal in deals:
            deal_type = deal["type"]
            by_type[deal_type] = by_type.get(deal_type, 0) + 1

        return {
            "count": len(deals),
            "by_type": by_type,
            "total_value": sum(d.get("value", 0) for d in deals)
        }

    def _group_by_type(self, deals: List[Dict]) -> Dict:
        """Group deals by type."""
        grouped = {}
        for deal in deals:
            deal_type = deal["type"]
            if deal_type not in grouped:
                grouped[deal_type] = {"count": 0, "value": 0}
            grouped[deal_type]["count"] += 1
            grouped[deal_type]["value"] += deal.get("value", 0)

        return grouped

    def _group_by_stage(self, deals: List[Dict]) -> Dict:
        """Group deals by development stage."""
        staged = {
            "Preclinical": {"count": 0, "value": 0},
            "Phase 1": {"count": 0, "value": 0},
            "Phase 2": {"count": 0, "value": 0},
            "Phase 3": {"count": 0, "value": 0},
            "Approved": {"count": 0, "value": 0}
        }

        for deal in deals:
            stage = deal.get("stage", "Unknown")
            if stage in staged:
                staged[stage]["count"] += 1
                staged[stage]["value"] += deal.get("value", 0)

        # Remove empty stages
        return {k: v for k, v in staged.items() if v["count"] > 0}

    def save(self, output: str):
        """Save results to file."""
        with open(output, 'w') as f:
            json.dump(self.results, f, indent=2)
        print(f"✓ Results saved to {output}")


def main():
    parser = argparse.ArgumentParser(description="Deal tracker for pharma industry")
    parser.add_argument("--feed", action="store_true", help="Get recent deals feed")
    parser.add_argument("--days", type=int, default=30, help="Days for feed")
    parser.add_argument("--target", help="Search by target")
    parser.add_argument("--type", help="Filter by deal type")
    parser.add_argument("--summary", action="store_true", help="Get deal summary")
    parser.add_argument("--year", type=int, help="Filter by year")
    parser.add_argument("--oncology", action="store_true", help="Filter to oncology")
    parser.add_argument("--export", help="Export to file")
    parser.add_argument("--format", choices=["json", "summary", "csv"],
                        default="summary", help="Output format")

    args = parser.parse_args()

    tracker = DealTracker()

    if args.feed:
        # Recent deals feed
        deals = tracker.get_feed(days=args.days, deal_type=args.type)

        if args.format == "summary":
            print_feed_summary(deals, args.days)
        else:
            output = args.export or f"deals_feed_{args.days}d.json"
            tracker.save(output)

    elif args.target:
        # Target search
        deals = tracker.search_by_target(args.target, deal_type=args.type)

        if args.format == "summary":
            print_target_summary(args.target, deals)
        else:
            output = args.export or f"deals_{args.target}.json"
            tracker.save(output)

    elif args.summary:
        # Summary statistics
        area = "Oncology" if args.oncology else None
        summary = tracker.get_summary(year=args.year, therapeutic_area=area)

        if args.format == "summary":
            print_deal_summary(summary)
        else:
            output = args.export or "deals_summary.json"
            tracker.save(output)
    else:
        # Default: show recent feed
        deals = tracker.get_feed(days=args.days)
        print_feed_summary(deals, args.days)


def format_value(value: int) -> str:
    """Format deal value."""
    if value >= 1_000_000_000:
        return f"${value/1_000_000_000:.1f}B"
    elif value >= 1_000_000:
        return f"${value/1_000_000:.0f}M"
    else:
        return f"${value:,}"


def print_feed_summary(deals: List[Dict], days: int):
    """Print deals feed summary."""
    print("\n" + "="*70)
    print(f"PHARMA DEALS FEED (Last {days} days)")
    print("="*70 + "\n")

    if not deals:
        print("No deals found.")
        return

    by_type = {}
    for deal in deals:
        deal_type = deal["type"]
        by_type[deal_type] = by_type.get(deal_type, 0) + 1

    print("**Summary**")
    for deal_type, count in sorted(by_type.items()):
        print(f"  {deal_type.capitalize()}: {count}")
    print()

    print("**Recent Deals**")
    for deal in deals[:10]:
        print(f"**{deal['date']}** | {deal['type'].capitalize()}")

        if deal["type"] == "acquisition":
            acquirer = deal.get("acquirer", "Unknown")
            target = deal.get("target", "Unknown")
            value = format_value(deal.get("value", 0))
            print(f"  {acquirer} acquires {target} for {value}")
        elif deal["type"] == "licensing":
            licensee = deal.get("licensee", "Unknown")
            licensor = deal.get("licensor", "Unknown")
            upfront = format_value(deal.get("upfront", 0))
            total = format_value(deal.get("value", 0))
            print(f"  {licensee} in-licenses from {licensor}: {upfront} upfront, {total} total")
        elif deal["type"] == "partnership":
            p1 = deal.get("partner1", "Unknown")
            p2 = deal.get("partner2", "Unknown")
            value = format_value(deal.get("value", 0))
            print(f"  {p1} partners with {p2}: {value}")

        stage = deal.get("stage", "Unknown")
        area = deal.get("therapeutic_area", "Unknown")
        print(f"  Stage: {stage} | Area: {area}")
        print()

    print("="*70 + "\n")


def print_target_summary(target: str, deals: List[Dict]):
    """Print target deal summary."""
    print("\n" + "="*70)
    print(f"DEALS FOR: {target.upper()}")
    print("="*70 + "\n")

    if not deals:
        print(f"No deals found for {target}")
        return

    total_value = sum(d.get("value", 0) for d in deals)

    print(f"**Total Deals**: {len(deals)}")
    print(f"**Total Value**: {format_value(total_value)}\n")

    for deal in deals:
        print(f"**{deal['date']}** | {deal['type'].capitalize()}")
        print(f"  Value: {format_value(deal.get('value', 0))}")
        print(f"  Stage: {deal.get('stage', 'Unknown')}")
        print()

    print("="*70 + "\n")


def print_deal_summary(summary: Dict):
    """Print deal summary."""
    print("\n" + "="*70)
    title = f"DEAL SUMMARY"
    if summary.get("year"):
        title += f" {summary['year']}"
    if summary.get("therapeutic_area"):
        title += f" - {summary['therapeutic_area']}"
    print(title)
    print("="*70 + "\n")

    total = summary.get("total_deals", 0)
    value = summary.get("total_value", 0)

    print(f"**Total Deals**: {total}")
    print(f"**Total Value**: {format_value(value)}\n")

    print("**By Type**:")
    for deal_type, data in summary.get("by_type", {}).items():
        print(f"  {deal_type.capitalize()}: {data['count']} deals ({format_value(data['value'])})")

    print("\n**By Stage**:")
    for stage, data in summary.get("by_stage", {}).items():
        print(f"  {stage}: {data['count']} deals ({format_value(data['value'])})")

    print("\n**Top Deals**:")
    for deal in summary.get("top_deals", [])[:5]:
        print(f"  {format_value(deal.get('value', 0))} | {deal.get('type', 'Unknown')}")
        if deal["type"] == "acquisition":
            print(f"    {deal.get('acquirer', '-')} → {deal.get('target', '-')}")
        elif deal["type"] == "licensing":
            print(f"    {deal.get('licensor', '-')} → {deal.get('licensee', '-')}")

    print("\n" + "="*70 + "\n")


if __name__ == "__main__":
    main()
