"""
December Pipeline Priming - LinkedIn Campaign Slide Deck Generator
Creates PPTX with FirstMile green branding
"""
from pptx import Presentation
from pptx.util import Inches, Pt
from pptx.dml.color import RGBColor
from pptx.enum.text import PP_ALIGN, MSO_ANCHOR
from pptx.enum.shapes import MSO_SHAPE
import os

# FirstMile Brand Colors
FM_GREEN = RGBColor(0x28, 0xA7, 0x45)  # #28A745
FM_DARK_GREEN = RGBColor(0x1e, 0x7e, 0x34)  # #1e7e34
FM_RED = RGBColor(0xC4, 0x1E, 0x3A)  # #C41E3A
FM_CREAM = RGBColor(0xF5, 0xF5, 0xF0)  # #F5F5F0
FM_DARK = RGBColor(0x33, 0x33, 0x33)  # #333333
FM_GRAY = RGBColor(0x66, 0x66, 0x66)  # #666666
WHITE = RGBColor(0xFF, 0xFF, 0xFF)

# Slide dimensions (16:9)
SLIDE_WIDTH = Inches(13.33)
SLIDE_HEIGHT = Inches(7.5)

def create_presentation():
    prs = Presentation()
    prs.slide_width = SLIDE_WIDTH
    prs.slide_height = SLIDE_HEIGHT

    # Slide 1: Title
    create_title_slide(prs)

    # Slides 2-7: Post slides
    posts = [
        {
            "num": "POST 1: VOLUME REALITY CHECK",
            "date": "Mon Dec 16 | 8:00 AM",
            "title": "What volume do you actually control?",
            "copy": [
                "Your carrier should ask you this before quoting rates:",
                '"What volume do you actually control?"',
                "Here's what most shippers don't realize:",
                "• FBA volume? You don't control that.",
                "• 3PL-managed shipments? Often locked.",
                "• Marketplace-fulfilled? Out of your hands.",
                "I've seen brands claim 500K packages/year, but only 200K is actually addressable."
            ],
            "cta": 'CTA: "What % is truly addressable? Drop your number."',
            "footer": "38 Questions: Q1, Q3, Q4, Q6 | Score: 29/30",
            "header_color": FM_GREEN
        },
        {
            "num": "POST 2: THE CONTRACT QUESTION",
            "date": "Thu Dec 19 | 8:00 AM",
            "title": "January rate increases are coming.",
            "copy": [
                "Before you panic-switch carriers, answer this:",
                '"What minimum volume commitments do you have with your current carrier, and what penalties apply?"',
                "I've seen brands locked into contracts with $25K+ penalties they forgot about.",
                "Switching isn't always the answer. Sometimes it's renegotiating.",
                "But you can't strategize what you don't know."
            ],
            "cta": 'CTA: "Know your contract end date AND minimum? Reply with both."',
            "footer": "38 Questions: Q14, Q15 | Score: 26/30 | #GRI2025",
            "header_color": FM_GREEN,
            "cta_color": FM_RED
        },
        {
            "num": "POST 3: PEAK SEASON TRUTH SERUM",
            "date": "Mon Dec 23 | 8:00 AM",
            "title": "Peak season reveals everything.",
            "copy": [
                "One question I ask every brand in January:",
                '"How did your carrier perform during BFCM?"',
                "Not what they promised. What actually happened.",
                "• On-time delivery rate?",
                "• Claims processed in how many days?",
                "• Did they have capacity when you needed it?",
                "If you can't answer these questions, you're flying blind into 2025."
            ],
            "cta": 'CTA: "Grade your carrier: A, B, C, D, or F? Drop your grade + one word why."',
            "footer": "38 Questions: Q19, Q20, Q21 | Score: 27/30 | #BFCM",
            "header_color": FM_GREEN
        },
        {
            "num": "POST 4: IMPLEMENTATION BARRIERS",
            "date": "Mon Dec 30 | 8:00 AM",
            "title": "This question predicts success or failure.",
            "copy": [
                '"What would prevent you from ramping to full volume after a successful test?"',
                "Sounds simple. But the answers reveal everything:",
                '• "We\'ll start small and see" = No commitment',
                '• "Need to check with IT" = Wrong room',
                '• "Let\'s wait until after peak" = Indefinite delay',
                '• "Nothing - if you hit 95%, we\'re moving 100%" = Real partner'
            ],
            "cta": 'CTA: "Which response sounds like you? No judgment - awareness is step one."',
            "footer": "38 Questions: Q37, Q38 | Score: 29/30 | HIGH INTENT FILTER",
            "header_color": FM_GREEN
        },
        {
            "num": "POST 5: ZONE DISTRIBUTION",
            "date": "Mon Jan 6 | 8:00 AM",
            "title": "Simple question that changes everything.",
            "copy": [
                '"What are your top 5 destination states by volume?"',
                "Why it matters:",
                "Zone distribution determines your shipping costs more than anything.",
                "• Ship mostly to CA from TX? Zone 4-5.",
                "• Ship mostly to NY from TX? Zone 7-8.",
                "Different zones = different carrier strengths = different strategies.",
                "Most brands don't know their zone distribution. The ones who do save 20-30%."
            ],
            "cta": 'CTA: "Drop your top 5 states - I\'ll tell you which zones are eating your margins."',
            "footer": "38 Questions: Q17 | Score: 28/30 | FREE ZONE ANALYSIS OFFER",
            "header_color": FM_GREEN
        },
        {
            "num": "POST 6: GRI REACTION (HARVEST DAY)",
            "date": "Tue Jan 21 | ASAP After Announcement",
            "title": "GRI 2025 just dropped.",
            "copy": [
                "[CARRIER] rates going up [X]% effective [DATE].",
                "Before you panic:",
                "1. What's your ACTUAL cost per package today?",
                "2. What improvement would justify the pain of switching?",
                "3. Do you know your zone distribution well enough to compare?",
                "The brands who answer calmly will save money.",
                "The brands who panic will regret it."
            ],
            "cta": 'CTA: "DM me for help running the numbers. No pitch - just math."',
            "footer": "38 Questions: Q11, Q36 | Score: 30/30 PERFECT | BLOCK 2 HOURS",
            "header_color": FM_RED,
            "harvest": True
        }
    ]

    for post in posts:
        create_post_slide(prs, post)

    # Slide 8: Schedule
    create_schedule_slide(prs)

    return prs


def create_title_slide(prs):
    """Create title slide with green gradient background"""
    slide_layout = prs.slide_layouts[6]  # Blank
    slide = prs.slides.add_slide(slide_layout)

    # Green background
    background = slide.shapes.add_shape(
        MSO_SHAPE.RECTANGLE, 0, 0, SLIDE_WIDTH, SLIDE_HEIGHT
    )
    background.fill.solid()
    background.fill.fore_color.rgb = FM_GREEN
    background.line.fill.background()

    # Title
    title_box = slide.shapes.add_textbox(
        Inches(0.5), Inches(2), Inches(12.33), Inches(1.2)
    )
    tf = title_box.text_frame
    tf.word_wrap = True
    p = tf.paragraphs[0]
    p.text = "December Pipeline Priming"
    p.font.size = Pt(54)
    p.font.bold = True
    p.font.color.rgb = WHITE
    p.alignment = PP_ALIGN.CENTER

    # Subtitle
    sub_box = slide.shapes.add_textbox(
        Inches(0.5), Inches(3.3), Inches(12.33), Inches(0.8)
    )
    tf = sub_box.text_frame
    p = tf.paragraphs[0]
    p.text = "LinkedIn Thought Leadership Campaign"
    p.font.size = Pt(32)
    p.font.color.rgb = WHITE
    p.alignment = PP_ALIGN.CENTER

    # Date
    date_box = slide.shapes.add_textbox(
        Inches(0.5), Inches(4.8), Inches(12.33), Inches(0.5)
    )
    tf = date_box.text_frame
    p = tf.paragraphs[0]
    p.text = "December 16, 2024 - January 21, 2025"
    p.font.size = Pt(24)
    p.font.color.rgb = WHITE
    p.alignment = PP_ALIGN.CENTER

    # Tagline
    tag_box = slide.shapes.add_textbox(
        Inches(0.5), Inches(5.5), Inches(12.33), Inches(0.5)
    )
    tf = tag_box.text_frame
    p = tf.paragraphs[0]
    p.text = "6 Posts | 38 Questions Framework | GRI 2025 Harvest"
    p.font.size = Pt(18)
    p.font.color.rgb = WHITE
    p.alignment = PP_ALIGN.CENTER


def create_post_slide(prs, post):
    """Create a post slide with two-column layout"""
    slide_layout = prs.slide_layouts[6]  # Blank
    slide = prs.slides.add_slide(slide_layout)

    # Cream background
    background = slide.shapes.add_shape(
        MSO_SHAPE.RECTANGLE, 0, 0, SLIDE_WIDTH, SLIDE_HEIGHT
    )
    background.fill.solid()
    background.fill.fore_color.rgb = FM_CREAM
    background.line.fill.background()

    # Header bar
    header = slide.shapes.add_shape(
        MSO_SHAPE.RECTANGLE, 0, 0, SLIDE_WIDTH, Inches(0.8)
    )
    header.fill.solid()
    header.fill.fore_color.rgb = post.get("header_color", FM_GREEN)
    header.line.fill.background()

    # Post number
    num_box = slide.shapes.add_textbox(
        Inches(0.3), Inches(0.2), Inches(6), Inches(0.5)
    )
    tf = num_box.text_frame
    p = tf.paragraphs[0]
    p.text = post["num"]
    p.font.size = Pt(18)
    p.font.bold = True
    p.font.color.rgb = WHITE

    # Date
    date_box = slide.shapes.add_textbox(
        Inches(9), Inches(0.2), Inches(4), Inches(0.5)
    )
    tf = date_box.text_frame
    p = tf.paragraphs[0]
    p.text = post["date"]
    p.font.size = Pt(16)
    p.font.color.rgb = WHITE
    p.alignment = PP_ALIGN.RIGHT

    # White content card
    content_card = slide.shapes.add_shape(
        MSO_SHAPE.ROUNDED_RECTANGLE, Inches(0.3), Inches(1),
        Inches(12.73), Inches(6.2)
    )
    content_card.fill.solid()
    content_card.fill.fore_color.rgb = WHITE
    content_card.line.fill.background()

    # Harvest badge if applicable
    if post.get("harvest"):
        harvest_box = slide.shapes.add_textbox(
            Inches(0.5), Inches(1.2), Inches(5), Inches(0.4)
        )
        tf = harvest_box.text_frame
        p = tf.paragraphs[0]
        p.text = "HARVEST TIME - Every DM = Discovery Call"
        p.font.size = Pt(12)
        p.font.bold = True
        p.font.color.rgb = FM_GREEN

    # Title
    title_y = Inches(1.5) if post.get("harvest") else Inches(1.3)
    title_box = slide.shapes.add_textbox(
        Inches(0.5), title_y, Inches(5.5), Inches(0.8)
    )
    tf = title_box.text_frame
    p = tf.paragraphs[0]
    p.text = post["title"]
    p.font.size = Pt(24)
    p.font.bold = True
    p.font.color.rgb = post.get("header_color", FM_GREEN)

    # Copy text
    copy_y = title_y + Inches(0.8)
    copy_box = slide.shapes.add_textbox(
        Inches(0.5), copy_y, Inches(5.5), Inches(3.5)
    )
    tf = copy_box.text_frame
    tf.word_wrap = True

    for i, line in enumerate(post["copy"]):
        if i == 0:
            p = tf.paragraphs[0]
        else:
            p = tf.add_paragraph()
        p.text = line
        p.font.size = Pt(12)
        p.font.color.rgb = FM_DARK
        p.space_after = Pt(8)

    # CTA box
    cta_box = slide.shapes.add_shape(
        MSO_SHAPE.ROUNDED_RECTANGLE, Inches(0.5), Inches(5.8),
        Inches(5.5), Inches(0.6)
    )
    cta_color = post.get("cta_color", FM_GREEN)
    cta_box.fill.solid()
    cta_box.fill.fore_color.rgb = cta_color
    cta_box.line.fill.background()

    cta_text = slide.shapes.add_textbox(
        Inches(0.5), Inches(5.85), Inches(5.5), Inches(0.5)
    )
    tf = cta_text.text_frame
    p = tf.paragraphs[0]
    p.text = post["cta"]
    p.font.size = Pt(11)
    p.font.color.rgb = WHITE
    p.alignment = PP_ALIGN.CENTER

    # Image placeholder
    img_placeholder = slide.shapes.add_shape(
        MSO_SHAPE.ROUNDED_RECTANGLE, Inches(6.3), Inches(1.3),
        Inches(6.5), Inches(4.5)
    )
    img_placeholder.fill.solid()
    img_placeholder.fill.fore_color.rgb = RGBColor(0xF0, 0xF0, 0xF0)
    img_placeholder.line.color.rgb = post.get("header_color", FM_GREEN)
    img_placeholder.line.width = Pt(2)

    # Placeholder text
    placeholder_text = slide.shapes.add_textbox(
        Inches(6.3), Inches(3.3), Inches(6.5), Inches(0.5)
    )
    tf = placeholder_text.text_frame
    p = tf.paragraphs[0]
    p.text = "[IMAGE PLACEHOLDER]"
    p.font.size = Pt(14)
    p.font.color.rgb = FM_GRAY
    p.alignment = PP_ALIGN.CENTER

    # Footer
    footer_box = slide.shapes.add_textbox(
        Inches(6.3), Inches(6), Inches(6.5), Inches(0.4)
    )
    tf = footer_box.text_frame
    p = tf.paragraphs[0]
    p.text = post["footer"]
    p.font.size = Pt(10)
    p.font.color.rgb = FM_GRAY
    p.alignment = PP_ALIGN.CENTER


def create_schedule_slide(prs):
    """Create schedule summary slide"""
    slide_layout = prs.slide_layouts[6]  # Blank
    slide = prs.slides.add_slide(slide_layout)

    # Green background
    background = slide.shapes.add_shape(
        MSO_SHAPE.RECTANGLE, 0, 0, SLIDE_WIDTH, SLIDE_HEIGHT
    )
    background.fill.solid()
    background.fill.fore_color.rgb = FM_GREEN
    background.line.fill.background()

    # Title
    title_box = slide.shapes.add_textbox(
        Inches(0.5), Inches(0.3), Inches(12.33), Inches(0.8)
    )
    tf = title_box.text_frame
    p = tf.paragraphs[0]
    p.text = "Campaign Schedule"
    p.font.size = Pt(36)
    p.font.bold = True
    p.font.color.rgb = WHITE
    p.alignment = PP_ALIGN.CENTER

    # White content card
    content_card = slide.shapes.add_shape(
        MSO_SHAPE.ROUNDED_RECTANGLE, Inches(0.5), Inches(1.2),
        Inches(12.33), Inches(5.5)
    )
    content_card.fill.solid()
    content_card.fill.fore_color.rgb = WHITE
    content_card.line.fill.background()

    # Schedule data
    schedule = [
        ("Mon Dec 16", "Volume Reality", '"What do you actually control?"', "29/30"),
        ("Thu Dec 19", "Contract Question", '"What penalties apply?"', "26/30"),
        ("Mon Dec 23", "Peak Performance", '"How\'d your carrier actually perform?"', "27/30"),
        ("Mon Dec 30", "Implementation Barriers", '"What would prevent you from ramping?"', "29/30"),
        ("Mon Jan 6", "Zone Distribution", '"What are your top 5 states?"', "28/30"),
        ("Tue Jan 21", "GRI Reaction", '"Before you panic..." HARVEST DAY', "30/30")
    ]

    # Header row
    headers = ["Date", "Post", "Hook", "Score"]
    col_widths = [Inches(1.5), Inches(2.5), Inches(5.5), Inches(1.2)]
    col_x = [Inches(0.8), Inches(2.3), Inches(4.8), Inches(10.3)]

    header_y = Inches(1.5)
    for i, (header, width, x) in enumerate(zip(headers, col_widths, col_x)):
        header_box = slide.shapes.add_textbox(x, header_y, width, Inches(0.4))
        tf = header_box.text_frame
        p = tf.paragraphs[0]
        p.text = header
        p.font.size = Pt(14)
        p.font.bold = True
        p.font.color.rgb = FM_GREEN

    # Data rows
    row_y = Inches(2.1)
    for i, (date, post, hook, score) in enumerate(schedule):
        row_data = [date, post, hook, score]
        is_harvest = i == 5  # Last row is harvest

        for j, (data, width, x) in enumerate(zip(row_data, col_widths, col_x)):
            cell_box = slide.shapes.add_textbox(x, row_y, width, Inches(0.5))
            tf = cell_box.text_frame
            p = tf.paragraphs[0]
            p.text = data
            p.font.size = Pt(12)
            p.font.bold = is_harvest
            p.font.color.rgb = FM_RED if is_harvest else FM_DARK

        row_y += Inches(0.6)

    # Footer
    footer_box = slide.shapes.add_textbox(
        Inches(0.5), Inches(6.9), Inches(12.33), Inches(0.4)
    )
    tf = footer_box.text_frame
    p = tf.paragraphs[0]
    p.text = "38 Questions Framework | FirstMile Pipeline Priming | Dec 2024 - Jan 2025"
    p.font.size = Pt(14)
    p.font.color.rgb = WHITE
    p.alignment = PP_ALIGN.CENTER


if __name__ == "__main__":
    output_dir = os.path.dirname(os.path.abspath(__file__))
    output_path = os.path.join(output_dir, "..", "December_Pipeline_Priming_Campaign.pptx")

    prs = create_presentation()
    prs.save(output_path)
    print(f"Presentation saved to: {output_path}")
