---
name: deployment-checker
description: Verify all requirements before production deployment. Comprehensive checklist covering environment variables, database migrations, security, performance, monitoring, and payment configuration.
allowed-tools: Read, Grep, Glob, Bash
---

# Deployment Readiness Checker

## Purpose

Comprehensive pre-deployment validation to ensure production readiness:
- Environment variable verification
- Database migration status
- Security configuration
- Performance optimization
- Monitoring and logging
- Payment system configuration
- Testing and quality gates

## Auto-Invocation Triggers

This skill activates when:
- User asks to "deploy to production" or "deploy to prod"
- User mentions "deployment check" or "ready to deploy"
- User runs deployment commands (`gcloud run deploy`, `vercel deploy`)
- Before creating a release/tag
- User asks "are we ready to deploy"

## Deployment Checklist

### 1. Environment Variables (CRITICAL 🔴)

**Required Variables:**
```bash
# Database
DATABASE_URL                    # TiDB Cloud connection string

# Authentication
GOOGLE_CLIENT_ID               # OAuth (FREE)
GOOGLE_CLIENT_SECRET
GITHUB_CLIENT_ID               # OAuth (FREE)
GITHUB_CLIENT_SECRET
JWT_SECRET                     # 32+ character random string

# Payments
STRIPE_SECRET_KEY              # PRODUCTION key (sk_live_...)
VITE_STRIPE_PUBLISHABLE_KEY    # PRODUCTION key (pk_live_...)
STRIPE_WEBHOOK_SECRET          # Production webhook secret

# File Storage
AWS_ACCESS_KEY_ID              # S3 credentials
AWS_SECRET_ACCESS_KEY
AWS_REGION
AWS_S3_BUCKET

# Email
SMTP_HOST                      # Email service
SMTP_PORT
SMTP_USER
SMTP_PASS
EMAIL_FROM
OWNER_EMAIL

# Application
NODE_ENV=production            # MUST be production
```

**Optional Variables:**
```bash
# CRM Integration
HUBSPOT_ACCESS_TOKEN           # Optional but recommended

# Automation
N8N_WEBHOOK_URL               # Optional workflow triggers

# AI Features
LITELLM_API_KEY               # Optional AI features
LITELLM_BASE_URL
```

**Checks:**
- ✅ All required variables documented in DEPLOYMENT.md
- ✅ No `.env` files in git repository
- ✅ `NODE_ENV` set to `production`
- ✅ Secrets configured in hosting platform (not in code)
- ❌ No default/example values in production

**Validation:**
```bash
# Check for .env files in git
git ls-files | grep "\.env"
# Should return nothing

# Verify NODE_ENV
grep -r "NODE_ENV.*development" server/
# Should not hardcode development mode
```

### 2. Database Status (CRITICAL 🔴)

**Migration Checks:**
```bash
# Check for pending migrations
npx drizzle-kit check

# Verify all migrations applied
# Compare drizzle/meta/ with production database
```

**Checklist:**
- ✅ All migrations applied to production database
- ✅ No pending schema changes in `server/db.ts`
- ✅ Database backup completed recently
- ✅ Rollback plan documented
- ❌ No uncommitted schema changes

**Verification:**
```bash
# Check if schema.ts modified but no migration
git diff server/db.ts drizzle/schema.ts

# Ensure migrations are sequential
ls drizzle/*.sql | sort
```

### 3. Security Configuration (CRITICAL 🔴)

**HTTPS:**
- ✅ HTTPS enforced (no HTTP traffic)
- ✅ SSL certificate valid
- ✅ Redirects HTTP → HTTPS

**CORS:**
```typescript
// ✅ GOOD - Specific origins
app.use(cors({
  origin: ['https://studio535.com', 'https://app.studio535.com'],
  credentials: true
}));

// ❌ BAD - Allow all
app.use(cors({ origin: '*' }));
```

**Security Headers (Helmet.js):**
```typescript
app.use(helmet({
  contentSecurityPolicy: true,
  hsts: {
    maxAge: 31536000,
    includeSubDomains: true
  },
  noSniff: true,
  frameguard: { action: 'deny' }
}));
```

**Rate Limiting:**
```typescript
// On auth endpoints
rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 5 // 5 requests per window
})
```

**Checklist:**
- ✅ HTTPS enforced
- ✅ CORS configured for specific origins
- ✅ Helmet.js or equivalent configured
- ✅ Rate limiting on authentication endpoints
- ✅ Session security (httpOnly cookies, secure flag)
- ❌ No API keys or secrets in client-side code

**Grep Checks:**
```bash
# Check for allow-all CORS
grep -r "origin.*\*" server/

# Check for missing helmet
grep -r "helmet" server/_core/index.ts

# Check for console.log in production
grep -r "console\.log" server/ | grep -v test | grep -v debug
```

### 4. Performance (HIGH PRIORITY 🟠)

**Build Success:**
```bash
npm run build
# Must exit with code 0
```

**Bundle Size:**
```bash
# Check client bundle size
ls -lh dist/assets/*.js
# Main bundle should be < 500KB
# Total assets < 2MB
```

**Optimization Checks:**
- ✅ Code splitting implemented (`React.lazy`)
- ✅ Large components lazy-loaded
- ✅ Images optimized/compressed
- ✅ Dependencies minimized (no unused packages)
- ✅ Tree-shaking enabled

**Performance Audit:**
```bash
# Run Lighthouse or similar
# Target scores:
# Performance: 90+
# Accessibility: 95+
# Best Practices: 95+
# SEO: 90+
```

### 5. Monitoring & Logging (HIGH PRIORITY 🟠)

**Error Tracking:**
- ✅ Error tracking configured (Sentry, LogRocket, etc.)
- ✅ Error notifications enabled
- ✅ Source maps uploaded for production builds

**Health Check Endpoint:**
```typescript
// server/_core/index.ts
app.get('/api/health', (req, res) => {
  res.json({
    status: 'ok',
    timestamp: new Date().toISOString(),
    version: process.env.VERSION || 'unknown'
  });
});
```

**Logging:**
- ✅ Structured logging configured
- ✅ Log levels appropriate (no DEBUG in production)
- ✅ Sensitive data not logged (passwords, tokens)
- ✅ Request/response logging for debugging

**Verification:**
```bash
# Test health check
curl https://your-domain.com/api/health

# Check for debug logs
grep -r "console\.debug\|logger\.debug" server/
```

### 6. Payment Configuration (CRITICAL 🔴 - If Using Stripe)

**Stripe Keys:**
- ✅ Using PRODUCTION keys (sk_live_*, pk_live_*)
- ❌ NOT using test keys (sk_test_*, pk_test_*)
- ✅ Webhook endpoint configured in Stripe dashboard
- ✅ Webhook secret matches production environment

**Webhook Configuration:**
```bash
# Verify webhook endpoint
stripe listen --forward-to https://your-domain.com/api/stripe/webhook

# Test events
stripe trigger payment_intent.succeeded
```

**Checklist:**
- ✅ STRIPE_SECRET_KEY starts with `sk_live_`
- ✅ VITE_STRIPE_PUBLISHABLE_KEY starts with `pk_live_`
- ✅ Webhook endpoint: `https://your-domain.com/api/stripe/webhook`
- ✅ Webhook signature verification enabled
- ✅ Webhook events tested (checkout, payment, refund)

**Verification:**
```bash
# Check for test keys in code
grep -r "sk_test_\|pk_test_" .env* server/

# Verify webhook handler
grep -r "stripe.webhooks.constructEvent" server/
```

### 7. Testing & Quality (HIGH PRIORITY 🟠)

**Test Suite:**
```bash
npm run test
# All tests must pass
```

**Type Checking:**
```bash
npm run check
# No TypeScript errors
```

**Linting:**
```bash
npm run lint
# No critical linting errors
```

**Coverage:**
```bash
npm run test:coverage
# Aim for 80%+ coverage on critical paths
```

**Manual Testing:**
- ✅ User registration/login flow
- ✅ Critical user journeys tested
- ✅ Payment flow tested (test mode, then production)
- ✅ Email sending tested
- ✅ File upload tested

### 8. Infrastructure (MEDIUM PRIORITY 🟡)

**Hosting Platform:**
- ✅ Production environment configured (Render, Vercel, etc.)
- ✅ Auto-deploy from main branch enabled (or manual process documented)
- ✅ Environment variables set in platform
- ✅ Domain configured and SSL active

**Database:**
- ✅ Production database provisioned (TiDB Cloud)
- ✅ Connection pooling configured
- ✅ Backup strategy enabled
- ✅ Database credentials secured

**File Storage:**
- ✅ S3 bucket created (or R2)
- ✅ CORS configured for bucket
- ✅ Access policies set correctly
- ✅ CDN configured (if needed)

### 9. Documentation (MEDIUM PRIORITY 🟡)

**Deployment Docs:**
- ✅ DEPLOYMENT.md up to date
- ✅ Environment variables documented
- ✅ Deployment process documented
- ✅ Rollback procedure documented

**Runbooks:**
- ✅ Common issues and fixes documented
- ✅ Emergency contacts listed
- ✅ Incident response plan

**API Documentation:**
- ✅ API endpoints documented (if public API)
- ✅ Authentication flow documented
- ✅ Rate limits documented

## Output Formats

### All Checks Pass ✅
```
🚀 DEPLOYMENT READY

Environment Variables:
✅ All required variables configured
✅ No .env files in git
✅ Secrets in platform configuration

Database:
✅ All migrations applied
✅ No pending schema changes
✅ Backup completed (2024-01-09)

Security:
✅ HTTPS enforced
✅ CORS configured (2 allowed origins)
✅ Helmet.js enabled
✅ Rate limiting active

Performance:
✅ Build successful (2m 34s)
✅ Bundle size: 456KB (target: <500KB)
✅ Lazy loading implemented

Monitoring:
✅ Health check: /api/health responding
✅ Error tracking configured
✅ Logging enabled

Payments (Stripe):
✅ Production keys configured (sk_live_***)
✅ Webhook endpoint active
✅ Test transactions successful

Testing:
✅ All tests passing (127/127)
✅ TypeScript: No errors
✅ Coverage: 87% (target: 80%)

Status: READY FOR PRODUCTION DEPLOYMENT 🚀
```

### Critical Issues Found ❌
```
❌ DEPLOYMENT BLOCKED - CRITICAL ISSUES

Environment Variables:
❌ STRIPE_SECRET_KEY missing
❌ AWS_S3_BUCKET not configured
⚠️  JWT_SECRET only 16 characters (need 32+)

Database:
❌ 2 pending migrations not applied
❌ Schema changes in server/db.ts without migration

Security:
❌ CORS set to allow all origins (origin: '*')
❌ Helmet.js not configured
❌ No rate limiting on /api/auth/login

Testing:
❌ 5 tests failing
❌ TypeScript: 12 compilation errors

CANNOT DEPLOY - Fix critical issues first!

Priority Actions:
1. Add missing environment variables
2. Apply database migrations
3. Fix CORS configuration
4. Fix failing tests
5. Resolve TypeScript errors
```

### Warning Issues ⚠️
```
⚠️  DEPLOYMENT READY WITH WARNINGS

Environment Variables:
✅ All required variables configured
⚠️  HUBSPOT_ACCESS_TOKEN not set (CRM features disabled)

Database:
✅ All migrations applied
⚠️  Last backup: 7 days ago (consider manual backup)

Security:
✅ HTTPS enforced
✅ CORS configured
⚠️  Rate limiting not on /api/projects/create

Performance:
✅ Build successful
⚠️  Bundle size: 520KB (target: <500KB, consider code splitting)

Monitoring:
✅ Health check active
⚠️  Error tracking not configured (consider Sentry)

Testing:
✅ All tests passing
⚠️  Coverage: 72% (target: 80%)

You CAN deploy, but consider addressing warnings:
1. Setup error tracking
2. Add rate limiting to project creation
3. Optimize bundle size
4. Increase test coverage

Proceed with deployment? (Recommended: Yes, but monitor closely)
```

## Deployment Process

### Step 1: Pre-Deployment
```bash
# Run all checks
npm run check          # TypeScript
npm run lint           # Linting
npm run test           # Tests
npm run build          # Build

# Run deployment checker
# (invoke this skill)
```

### Step 2: Final Verification
- [ ] Review checklist above
- [ ] Fix any critical issues
- [ ] Consider warnings
- [ ] Tag release in git
- [ ] Notify team of deployment

### Step 3: Deploy
```bash
# Render (auto-deploys from GitHub)
git push origin main

# Or manual deploy
npm run deploy
```

### Step 4: Post-Deployment
```bash
# Verify health check
curl https://your-domain.com/api/health

# Monitor logs
npm run logs:prod

# Test critical flows
# - User login
# - Payment processing
# - File uploads
```

### Step 5: Rollback Plan
If issues occur:
```bash
# Render: Revert to previous deployment via dashboard
# Or redeploy previous git commit

# Database: Have migration rollback SQL ready
```

## Examples

### Example 1: First Production Deployment
```
📋 First Production Deployment Checklist

Critical Setup (Before First Deploy):
□ Domain purchased and configured
□ SSL certificate obtained
□ Production database provisioned
□ S3 bucket created
□ Stripe account setup (live mode enabled)
□ Email service configured
□ Error tracking setup

Environment Variables (27 required):
□ DATABASE_URL (TiDB Cloud)
□ STRIPE_SECRET_KEY (sk_live_...)
□ AWS credentials
□ SMTP credentials
... (full list in DEPLOYMENT.md)

Initial Testing:
□ Manual test all user flows
□ Test payment processing (live mode)
□ Verify email sending
□ Check file uploads

Documentation:
□ Update DNS records
□ Configure webhooks in Stripe dashboard
□ Setup monitoring alerts
□ Document rollback procedure

Estimated Time: 2-4 hours
Ready? Let's deploy! 🚀
```

### Example 2: Routine Update Deployment
```
✅ Routine Deployment Check

Changes Since Last Deploy:
- 23 commits (features, bug fixes)
- 5 new files
- 2 database migrations

Quick Verification:
✅ Tests passing (132/132)
✅ TypeScript clean
✅ Migrations applied to staging
✅ No environment variable changes
✅ Security scan clean

Risk Level: LOW
Estimated Downtime: < 30 seconds
Proceed: YES ✅

Deploy command ready:
  git push origin main
```

### Example 3: High-Risk Deployment
```
⚠️  HIGH-RISK DEPLOYMENT DETECTED

Changes:
- Payment processing logic modified
- Database schema changes (3 migrations)
- Authentication flow updated
- Stripe webhook handler modified

Additional Checks Required:
□ Payment flow tested in staging
□ Database migrations tested
□ Rollback SQL prepared
□ Team notified
□ Off-hours deployment scheduled
□ Monitoring alerts active

Recommended Actions:
1. Deploy to staging first
2. Run comprehensive tests
3. Schedule deployment for low-traffic time
4. Have rollback plan ready
5. Monitor closely for 1 hour post-deploy

Risk Level: HIGH ⚠️
Proceed with caution!
```

## Integration

### CI/CD Pipeline
Add to `.github/workflows/deploy.yml`:
```yaml
name: Deploy to Production

on:
  push:
    branches: [main]

jobs:
  deploy:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v3

      - name: Install dependencies
        run: npm ci

      - name: Run tests
        run: npm test

      - name: Type check
        run: npm run check

      - name: Build
        run: npm run build

      - name: Deploy
        run: npm run deploy
        env:
          # Environment variables from GitHub secrets
```

### Manual Deployment
```bash
# Create deployment script
#!/bin/bash
set -e

echo "🚀 Starting deployment..."

# Run checks
npm run check
npm run test
npm run build

# Deploy
npm run deploy

echo "✅ Deployment complete!"
```

## Maintenance

### Before Every Deployment
- Run this skill
- Fix critical issues
- Consider warnings
- Document changes

### Weekly
- Review deployment logs
- Update deployment docs
- Check for dependency updates
- Review security advisories

### Monthly
- Audit environment variables
- Review and test rollback procedure
- Update runbooks
- Performance audit

## Resources

- **Deployment Guide:** `/DEPLOYMENT.md`
- **Render Docs:** https://render.com/docs
- **Vercel Docs:** https://vercel.com/docs
- **Stripe Production:** https://stripe.com/docs/testing#production

---

This skill is your final safety check before production deployment.
Better to catch issues here than in production!
