# Page Templates for Book Design

Ready-to-use templates for LaTeX and CSS-based book production.

## LaTeX Templates

### Basic Memoir Class Setup

```latex
% memoir-template.tex
\documentclass[11pt,twoside,openright]{memoir}

% Page geometry for 6x9 trim
\setstocksize{9in}{6in}
\settrimmedsize{9in}{6in}{*}
\settrims{0pt}{0pt}

% Margins
\setlrmarginsandblock{0.875in}{0.625in}{*}  % inside, outside
\setulmarginsandblock{0.75in}{0.875in}{*}   % top, bottom
\checkandfixthelayout

% Typography
\usepackage{fontspec}
\setmainfont{EB Garamond}[
  Numbers=OldStyle,
  Ligatures=TeX
]
\setsansfont{Gill Sans}

% Line spacing
\setSingleSpace{1.15}
\SingleSpacing

% Paragraph settings
\setlength{\parindent}{1.5em}
\setlength{\parskip}{0pt}

% Widow/orphan control
\widowpenalty=10000
\clubpenalty=10000
\raggedbottom

% Chapter style
\makechapterstyle{elegant}{%
  \chapterstyle{default}
  \renewcommand*{\chapnumfont}{\normalfont\Large\scshape}
  \renewcommand*{\chaptitlefont}{\normalfont\huge\itshape}
  \renewcommand*{\printchaptername}{}
  \renewcommand*{\chapternamenum}{}
  \renewcommand*{\printchapternum}{%
    \centering\chapnumfont Chapter \thechapter\par\vskip 10pt}
  \renewcommand*{\afterchapternum}{}
  \renewcommand*{\printchaptertitle}[1]{%
    \centering\chaptitlefont ##1\par}
  \setlength{\beforechapskip}{2in}
  \setlength{\afterchapskip}{1in}
}
\chapterstyle{elegant}

% Running headers
\makepagestyle{memoir}
\makeevenhead{memoir}{\small\scshape\leftmark}{}{}
\makeoddhead{memoir}{}{}{\small\scshape\rightmark}
\makeevenfoot{memoir}{\thepage}{}{}
\makeoddfoot{memoir}{}{}{\thepage}
\pagestyle{memoir}

% Front matter style (no headers)
\makepagestyle{frontmatter}
\makeevenfoot{frontmatter}{\thepage}{}{}
\makeoddfoot{frontmatter}{}{}{\thepage}

% Drop caps
\usepackage{lettrine}
\setcounter{DefaultLines}{3}
\renewcommand{\DefaultLoversize}{0.1}
\renewcommand{\DefaultLhang}{0.2}

% Block quotes
\renewenvironment{quote}{%
  \list{}{%
    \leftmargin=0.5in
    \rightmargin=0.5in
  }%
  \item\relax\small\itshape
}{%
  \endlist
}

\begin{document}
\frontmatter
\pagestyle{frontmatter}

% Half title
\begin{titlingpage}
\vspace*{2.5in}
\begin{center}
{\Large\scshape Book Title}
\end{center}
\end{titlingpage}

% Title page
\begin{titlingpage}
\vspace*{2in}
\begin{center}
{\Huge\itshape Book Title}\\[0.5in]
{\Large Author Name}\\[2in]
{\small Publisher Name}\\
{\small City, Year}
\end{center}
\end{titlingpage}

% Copyright page
\thispagestyle{empty}
\vspace*{\fill}
\begin{flushleft}
\small
Copyright © 2024 Author Name\\
All rights reserved.\\[12pt]
ISBN: 978-X-XXXX-XXXX-X\\[12pt]
First Edition\\[12pt]
Printed in the United States of America
\end{flushleft}
\clearpage

% Dedication
\thispagestyle{empty}
\vspace*{2.5in}
\begin{center}
\textit{For my family}
\end{center}
\clearpage

\tableofcontents

\mainmatter
\pagestyle{memoir}

\chapter{The Beginning}

\lettrine{T}{his} is where the story begins. The first paragraph after a chapter heading uses a drop cap and has no indent.

Regular paragraphs that follow have a first-line indent and continue the narrative flow of the text. This demonstrates the proper paragraph styling.

\end{document}
```

### Chapter Opener Variations

**Style 1: Centered with ornament**
```latex
\makechapterstyle{ornament}{%
  \renewcommand*{\printchaptername}{}
  \renewcommand*{\chapternamenum}{}
  \renewcommand*{\printchapternum}{%
    \centering
    {\Large ❧}\\[10pt]
    {\chapnumfont\thechapter}\\[10pt]
    {\Large ❧}
  }
  \renewcommand*{\printchaptertitle}[1]{%
    \centering\chaptitlefont ##1
  }
  \setlength{\beforechapskip}{1.5in}
  \setlength{\afterchapskip}{0.75in}
}
```

**Style 2: Left-aligned modern**
```latex
\makechapterstyle{modern}{%
  \renewcommand*{\printchaptername}{}
  \renewcommand*{\printchapternum}{%
    {\chapnumfont\fontsize{60}{60}\selectfont\thechapter}
  }
  \renewcommand*{\afterchapternum}{\par\vskip 10pt}
  \renewcommand*{\printchaptertitle}[1]{%
    {\chaptitlefont ##1}
  }
  \setlength{\beforechapskip}{0.5in}
  \setlength{\afterchapskip}{0.5in}
}
```

**Style 3: Small caps elegant**
```latex
\makechapterstyle{smallcaps}{%
  \renewcommand*{\chapnumfont}{\normalfont\scshape\Large}
  \renewcommand*{\chaptitlefont}{\normalfont\huge}
  \renewcommand*{\printchaptername}{}
  \renewcommand*{\printchapternum}{%
    \centering\chapnumfont chapter \thechapter
  }
  \renewcommand*{\afterchapternum}{\par\vskip 20pt}
  \renewcommand*{\printchaptertitle}[1]{%
    \centering\chaptitlefont\MakeUppercase{##1}
  }
}
```

### Section Break Templates

```latex
% Three asterisks
\newcommand{\sectionbreak}{%
  \par\vspace{18pt}
  \centering * \quad * \quad *
  \par\vspace{18pt}
  \noindent
}

% Ornament
\newcommand{\ornamentbreak}{%
  \par\vspace{24pt}
  \centering ❧
  \par\vspace{24pt}
  \noindent
}

% Simple space
\newcommand{\simplebreak}{%
  \par\vspace{24pt}
  \noindent
}
```

### Photo Page Template

```latex
\newcommand{\photopage}[3]{%
  % #1 = image file, #2 = caption, #3 = credit
  \clearpage
  \thispagestyle{empty}
  \begin{figure}[p]
    \centering
    \includegraphics[width=\textwidth,height=0.8\textheight,keepaspectratio]{#1}
    \caption*{#2}
    \vspace{6pt}
    {\footnotesize\textit{#3}}
  \end{figure}
  \clearpage
}

% Usage: \photopage{grandma.jpg}{Grandma on her wedding day, 1952}{Photo courtesy of the Smith family}
```

### Block Quote with Attribution

```latex
\newenvironment{epigraph}[2]{%
  % #1 = author, #2 = source
  \vspace{12pt}
  \begin{quote}
  \itshape
}{%
  \end{quote}
  \hfill---#1, \textit{#2}
  \vspace{12pt}
}

% Usage:
% \begin{epigraph}{Oscar Wilde}{The Importance of Being Earnest}
% Memory is the diary that we all carry about with us.
% \end{epigraph}
```

## CSS/HTML Templates (for Prince or WeasyPrint)

### Basic Book Stylesheet

```css
/* book-styles.css */
@page {
  size: 6in 9in;
  margin-top: 0.75in;
  margin-bottom: 0.875in;
  margin-outside: 0.625in;
  margin-inside: 0.875in;

  @bottom-center {
    content: counter(page);
    font-size: 10pt;
  }
}

@page :left {
  @top-left {
    content: string(book-title);
    font-size: 9pt;
    font-variant: small-caps;
  }
}

@page :right {
  @top-right {
    content: string(chapter-title);
    font-size: 9pt;
    font-variant: small-caps;
  }
}

@page :first {
  @top-left { content: none; }
  @top-right { content: none; }
}

@page frontmatter {
  @top-left { content: none; }
  @top-right { content: none; }
  @bottom-center {
    content: counter(page, lower-roman);
  }
}

/* Typography */
body {
  font-family: "EB Garamond", Garamond, Georgia, serif;
  font-size: 11pt;
  line-height: 1.4;
  text-align: justify;
  hyphens: auto;
  orphans: 3;
  widows: 3;
}

h1.book-title {
  string-set: book-title content();
}

h1.chapter-title {
  string-set: chapter-title content();
  page-break-before: right;
  margin-top: 2in;
  margin-bottom: 1in;
  font-size: 24pt;
  font-weight: normal;
  font-style: italic;
  text-align: center;
}

.chapter-number {
  display: block;
  font-size: 14pt;
  font-variant: small-caps;
  text-align: center;
  margin-bottom: 0.5in;
}

p {
  margin: 0;
  text-indent: 1.5em;
}

p:first-of-type,
h1 + p,
h2 + p,
.section-break + p,
blockquote + p {
  text-indent: 0;
}

/* Drop cap */
p.has-dropcap::first-letter {
  float: left;
  font-size: 3.5em;
  line-height: 0.8;
  padding-right: 0.1em;
  margin-top: 0.1em;
}

/* Block quotes */
blockquote {
  margin: 1em 0.5in;
  font-style: italic;
}

blockquote cite {
  display: block;
  text-align: right;
  font-style: normal;
  margin-top: 0.5em;
}

blockquote cite::before {
  content: "—";
}

/* Section breaks */
.section-break {
  text-align: center;
  margin: 1.5em 0;
}

.section-break.asterisks::before {
  content: "* * *";
  letter-spacing: 0.5em;
}

.section-break.ornament::before {
  content: "❧";
  font-size: 14pt;
}

/* Front matter */
.frontmatter {
  page: frontmatter;
}

.title-page {
  page-break-before: right;
  text-align: center;
  padding-top: 2in;
}

.title-page h1 {
  font-size: 36pt;
  font-style: italic;
  margin-bottom: 0.5in;
}

.title-page .author {
  font-size: 18pt;
  margin-bottom: 2in;
}

.title-page .publisher {
  font-size: 12pt;
}

.copyright-page {
  page-break-before: left;
  font-size: 9pt;
  line-height: 1.6;
}

.dedication {
  page-break-before: right;
  text-align: center;
  padding-top: 2.5in;
  font-style: italic;
}

/* Photos */
figure {
  page-break-inside: avoid;
  text-align: center;
  margin: 1em 0;
}

figure img {
  max-width: 100%;
  max-height: 7in;
}

figure figcaption {
  font-size: 9pt;
  font-style: italic;
  margin-top: 0.5em;
}

/* Tables */
table {
  width: 100%;
  border-collapse: collapse;
  margin: 1em 0;
  font-size: 10pt;
}

th, td {
  padding: 0.5em;
  text-align: left;
  border-bottom: 0.5pt solid #ccc;
}

th {
  font-variant: small-caps;
  border-bottom: 1pt solid #000;
}
```

### HTML Book Structure

```html
<!DOCTYPE html>
<html>
<head>
  <meta charset="utf-8">
  <title>Book Title</title>
  <link rel="stylesheet" href="book-styles.css">
</head>
<body>

<!-- Front Matter -->
<section class="frontmatter">

  <!-- Half Title -->
  <div class="half-title">
    <h1>Book Title</h1>
  </div>

  <!-- Title Page -->
  <div class="title-page">
    <h1 class="book-title">Book Title</h1>
    <p class="author">Author Name</p>
    <p class="publisher">
      Publisher Name<br>
      City, Year
    </p>
  </div>

  <!-- Copyright Page -->
  <div class="copyright-page">
    <p>Copyright © 2024 Author Name</p>
    <p>All rights reserved.</p>
    <p>ISBN: 978-X-XXXX-XXXX-X</p>
    <p>First Edition</p>
  </div>

  <!-- Dedication -->
  <div class="dedication">
    <p>For my family</p>
  </div>

  <!-- Table of Contents -->
  <nav class="toc">
    <h2>Contents</h2>
    <ol>
      <li><a href="#chapter-1">The Beginning</a></li>
      <li><a href="#chapter-2">Growing Up</a></li>
    </ol>
  </nav>

</section>

<!-- Main Content -->
<section class="mainmatter">

  <article id="chapter-1">
    <span class="chapter-number">Chapter One</span>
    <h1 class="chapter-title">The Beginning</h1>

    <p class="has-dropcap">This is where the story begins. The first paragraph uses a drop cap and has no indent.</p>

    <p>Regular paragraphs have a first-line indent and continue the narrative.</p>

    <div class="section-break asterisks"></div>

    <p>After a section break, the paragraph has no indent.</p>

    <blockquote>
      Memory is the diary that we all carry about with us.
      <cite>Oscar Wilde</cite>
    </blockquote>

    <p>The narrative continues after the quote.</p>
  </article>

</section>

<!-- Back Matter -->
<section class="backmatter">

  <article class="acknowledgments">
    <h1>Acknowledgments</h1>
    <p>Thank you to everyone who made this book possible.</p>
  </article>

  <article class="about-author">
    <h1>About the Author</h1>
    <p>Author Name lives in City with their family.</p>
  </article>

</section>

</body>
</html>
```

## Pandoc Configuration

### YAML Metadata Block

```yaml
---
title: "Book Title"
subtitle: "A Memoir"
author: "Author Name"
date: "2024"
publisher: "Publisher Name"
rights: "Copyright © 2024 Author Name. All rights reserved."
isbn: "978-X-XXXX-XXXX-X"

# LaTeX settings
documentclass: memoir
classoption:
  - 11pt
  - twoside
  - openright
geometry:
  - paperwidth=6in
  - paperheight=9in
  - top=0.75in
  - bottom=0.875in
  - inner=0.875in
  - outer=0.625in
mainfont: "EB Garamond"
sansfont: "Gill Sans"
fontsize: 11pt
linestretch: 1.15

# Pandoc settings
toc: true
toc-depth: 1
numbersections: false

# Custom LaTeX
header-includes:
  - \usepackage{lettrine}
  - \widowpenalty=10000
  - \clubpenalty=10000
---
```

### Pandoc Command for PDF

```bash
pandoc manuscript.md \
  -o book.pdf \
  --pdf-engine=xelatex \
  --template=memoir-template.tex \
  --lua-filter=dropcaps.lua \
  --toc \
  --toc-depth=1 \
  -V documentclass=memoir \
  -V classoption=11pt,twoside,openright
```

### Lua Filter for Drop Caps

```lua
-- dropcaps.lua
-- Adds drop caps to first paragraph of each chapter

local chapter_start = false

function Header(el)
  if el.level == 1 then
    chapter_start = true
  end
  return el
end

function Para(el)
  if chapter_start then
    chapter_start = false
    local first_char = pandoc.utils.stringify(el.content[1]):sub(1,1)
    local rest = pandoc.utils.stringify(el.content):sub(2)

    local dropcap = pandoc.RawInline('latex',
      '\\lettrine{' .. first_char .. '}{' .. rest:sub(1,3) .. '}')
    local remaining = pandoc.Str(rest:sub(4))

    return pandoc.Para({dropcap, remaining})
  end
  return el
end
```

## Quick Reference: Page Counts

### Front Matter Template (typical)

| Page | Content | Notes |
|------|---------|-------|
| i | Half-title | Title only, recto |
| ii | Blank or "Also by" | verso |
| iii | Title page | Full title, recto |
| iv | Copyright | verso |
| v | Dedication | recto |
| vi | Blank | verso |
| vii | Epigraph | recto (optional) |
| viii | Blank | verso |
| ix-x | Table of Contents | |
| xi | Preface/Intro start | recto |

### Chapter Start Requirements

- Always start on recto (odd/right) page
- Insert blank verso if needed
- Chapter opener page: no running header, folio at bottom

### Divisibility

Total page count must be divisible by 4 (or 8/16 for signature printing).
Add blank pages at end if needed.
