# Pandoc Configuration for Book Production

Complete reference for using Pandoc to generate print-ready book PDFs.

## Basic Command Structure

```bash
pandoc input.md \
  -o output.pdf \
  --pdf-engine=xelatex \
  --template=template.tex \
  [options]
```

## Recommended PDF Engines

| Engine | Best For | Notes |
|--------|----------|-------|
| xelatex | OpenType fonts, Unicode | Recommended for most books |
| lualatex | Complex typography | Slower but more flexible |
| pdflatex | Basic ASCII text | Fastest, limited font support |

## Complete Book Command

```bash
pandoc manuscript.md \
  -o book.pdf \
  --pdf-engine=xelatex \
  --template=memoir-template.tex \
  --toc \
  --toc-depth=1 \
  --top-level-division=chapter \
  --number-sections=false \
  -V documentclass=memoir \
  -V classoption=11pt,twoside,openright \
  -V paperwidth=6in \
  -V paperheight=9in \
  -V mainfont="EB Garamond" \
  -V sansfont="Gill Sans" \
  -V fontsize=11pt \
  -V linestretch=1.15 \
  -V geometry="top=0.75in,bottom=0.875in,inner=0.875in,outer=0.625in"
```

## YAML Metadata Block

Include at the start of your markdown file:

```yaml
---
title: "Your Book Title"
subtitle: "A Memoir"
author: "Author Name"
date: "2024"

# Document settings
documentclass: memoir
classoption:
  - 11pt
  - twoside
  - openright

# Page geometry
geometry:
  - paperwidth=6in
  - paperheight=9in
  - top=0.75in
  - bottom=0.875in
  - inner=0.875in
  - outer=0.625in

# Typography
mainfont: "EB Garamond"
sansfont: "Gill Sans"
monofont: "Courier New"
fontsize: 11pt
linestretch: 1.15

# Structure
toc: true
toc-depth: 1
numbersections: false

# LaTeX customization
header-includes:
  - \usepackage{lettrine}
  - \widowpenalty=10000
  - \clubpenalty=10000
  - \raggedbottom
---
```

## Key Variables

### Document Class Options

```yaml
documentclass: memoir    # Best for books
# or: book, report, article

classoption:
  - 11pt                 # Font size (10pt, 11pt, 12pt)
  - twoside              # Different left/right margins
  - openright            # Chapters start on right page
  - onecolumn            # Single column (default)
```

### Page Geometry

```yaml
geometry:
  - paperwidth=6in
  - paperheight=9in
  - top=0.75in
  - bottom=0.875in
  - inner=0.875in        # Gutter (binding side)
  - outer=0.625in
  - bindingoffset=0in    # Extra gutter for binding
```

### Typography Variables

```yaml
mainfont: "EB Garamond"      # Body text
sansfont: "Gill Sans"        # Headings (if different)
monofont: "Courier New"      # Code blocks
mathfont: "Latin Modern"     # Math equations

fontsize: 11pt               # Base font size
linestretch: 1.15            # Line spacing multiplier
```

### Table of Contents

```yaml
toc: true                    # Include TOC
toc-depth: 1                 # Only chapters (1) or sections too (2)
toc-title: "Contents"        # Custom TOC title
```

## Common Options

| Option | Purpose | Example |
|--------|---------|---------|
| `--toc` | Add table of contents | |
| `--toc-depth=N` | TOC depth level | `--toc-depth=1` |
| `--number-sections` | Number chapters | `--number-sections=false` |
| `--top-level-division` | Top element type | `--top-level-division=chapter` |
| `--shift-heading-level-by=N` | Adjust heading levels | `--shift-heading-level-by=-1` |
| `--standalone` | Complete document | (default for PDF) |

## Filters and Extensions

### Lua Filters

```bash
pandoc input.md -o output.pdf \
  --lua-filter=dropcaps.lua \
  --lua-filter=smallcaps.lua
```

### Common Filters

**Drop caps filter (dropcaps.lua):**
```lua
local chapter_start = false

function Header(el)
  if el.level == 1 then
    chapter_start = true
  end
  return el
end

function Para(el)
  if chapter_start then
    chapter_start = false
    -- Add drop cap LaTeX command
    return pandoc.RawBlock('latex',
      '\\lettrine{' ..
      pandoc.utils.stringify(el):sub(1,1) ..
      '}{} ' ..
      pandoc.utils.stringify(el):sub(2))
  end
  return el
end
```

**Scene break filter (scenebreaks.lua):**
```lua
function HorizontalRule(el)
  return pandoc.RawBlock('latex',
    '\\bigskip\\centerline{* * *}\\bigskip\\noindent')
end
```

## Markdown Conventions for Books

### Chapter Headings

```markdown
# Chapter One: The Beginning

First paragraph text...

## Section Within Chapter

More text...
```

### Scene Breaks

Use horizontal rules:
```markdown
The scene ends here.

---

A new scene begins.
```

### Block Quotes

```markdown
> This is a block quote that will be
> indented and italicized.
>
> — Attribution
```

### Poetry/Verse

Use line blocks:
```markdown
| Roses are red,
| Violets are blue,
| This preserves
| Line breaks too.
```

### Images

```markdown
![Caption text](images/photo.jpg){width=80%}
```

### Front Matter Markers

```markdown
---
title: "Book Title"
---

\frontmatter

# Dedication {.unnumbered}

For my family.

# Preface {.unnumbered}

Preface text...

\mainmatter

# Chapter One

Story begins...

\backmatter

# Acknowledgments {.unnumbered}

Thanks to...
```

## Common Issues and Solutions

### Font Not Found

```bash
# List available fonts
fc-list | grep -i garamond

# Install fonts system-wide
sudo cp *.ttf /usr/share/fonts/truetype/
sudo fc-cache -fv
```

### Widows and Orphans

Add to header-includes:
```yaml
header-includes:
  - \widowpenalty=10000
  - \clubpenalty=10000
```

### Chapter Spacing

```yaml
header-includes:
  - \setlength{\beforechapskip}{2in}
  - \setlength{\afterchapskip}{1in}
```

### Running Headers

```yaml
header-includes:
  - |
    \makepagestyle{memoir}
    \makeevenhead{memoir}{\small\scshape\leftmark}{}{}
    \makeoddhead{memoir}{}{}{\small\scshape\rightmark}
    \makeevenfoot{memoir}{\thepage}{}{}
    \makeoddfoot{memoir}{}{}{\thepage}
    \pagestyle{memoir}
```

## Quick Reference Commands

**Simple conversion:**
```bash
pandoc book.md -o book.pdf
```

**With custom template:**
```bash
pandoc book.md -o book.pdf --template=memoir.tex
```

**From DOCX to PDF:**
```bash
pandoc book.docx -o book.pdf --pdf-engine=xelatex
```

**Generate LaTeX (for debugging):**
```bash
pandoc book.md -o book.tex
```

**Multiple input files:**
```bash
pandoc front.md chapter*.md back.md -o book.pdf
```
