#!/usr/bin/env node
/**
 * Book Cover Dimension Calculator
 *
 * Calculates full cover dimensions including spine width for print-on-demand.
 *
 * Usage:
 *   node calculate-cover.js --pages 200 --trim 6x9 --paper cream
 *   node calculate-cover.js -p 250 -t 5.5x8.5 --paper white
 */

import { parseArgs } from 'node:util';

// Paper thickness values (inches per page)
const PAPER_THICKNESS = {
  white: 0.002252,
  cream: 0.0025,
  natural: 0.0025,
  groundwood: 0.003,
  premium: 0.002
};

// Common trim sizes
const TRIM_SIZES = {
  '6x9': { width: 6, height: 9, name: 'Trade Paperback' },
  '5.5x8.5': { width: 5.5, height: 8.5, name: 'Digest' },
  '5x8': { width: 5, height: 8, name: 'Small Trade' },
  '8.5x11': { width: 8.5, height: 11, name: 'US Letter' },
  '7.5x7.5': { width: 7.5, height: 7.5, name: 'Square' },
  '6.14x9.21': { width: 6.14, height: 9.21, name: 'Royal' }
};

// Standard bleed
const BLEED = 0.125;

function parseArguments() {
  const options = {
    pages: { type: 'string', short: 'p' },
    trim: { type: 'string', short: 't', default: '6x9' },
    paper: { type: 'string', default: 'cream' },
    help: { type: 'boolean', short: 'h' }
  };

  try {
    const { values } = parseArgs({ options, allowPositionals: false });
    return values;
  } catch (error) {
    console.error('Error parsing arguments:', error.message);
    showHelp();
    process.exit(1);
  }
}

function showHelp() {
  console.log(`
Book Cover Dimension Calculator

Usage:
  node calculate-cover.js --pages <count> [options]

Options:
  -p, --pages <count>    Number of pages (required)
  -t, --trim <size>      Trim size (default: 6x9)
  --paper <type>         Paper type (default: cream)
  -h, --help             Show this help

Trim sizes: ${Object.keys(TRIM_SIZES).join(', ')}
Paper types: ${Object.keys(PAPER_THICKNESS).join(', ')}

Examples:
  node calculate-cover.js --pages 200
  node calculate-cover.js -p 300 -t 5.5x8.5 --paper white
`);
}

function calculateCover(pages, trimSize, paperType) {
  const trim = TRIM_SIZES[trimSize];
  const thickness = PAPER_THICKNESS[paperType];

  if (!trim) {
    throw new Error(`Unknown trim size: ${trimSize}. Available: ${Object.keys(TRIM_SIZES).join(', ')}`);
  }

  if (!thickness) {
    throw new Error(`Unknown paper type: ${paperType}. Available: ${Object.keys(PAPER_THICKNESS).join(', ')}`);
  }

  const spineWidth = pages * thickness;
  const fullWidth = trim.width + spineWidth + trim.width + (BLEED * 2);
  const fullHeight = trim.height + (BLEED * 2);

  // Calculate positions for design guides
  const guides = {
    leftBleed: 0,
    backTrim: BLEED,
    backSafe: BLEED + 0.25,
    spineLeft: BLEED + trim.width,
    spineCenter: BLEED + trim.width + (spineWidth / 2),
    spineRight: BLEED + trim.width + spineWidth,
    frontSafe: BLEED + trim.width + spineWidth + 0.25,
    frontTrim: fullWidth - BLEED,
    rightBleed: fullWidth,
    topBleed: 0,
    topTrim: BLEED,
    topSafe: BLEED + 0.25,
    bottomSafe: fullHeight - BLEED - 0.25,
    bottomTrim: fullHeight - BLEED,
    bottomBleed: fullHeight
  };

  // Spine text recommendation
  let spineTextRec;
  if (spineWidth < 0.25) {
    spineTextRec = 'No spine text (too thin)';
  } else if (spineWidth < 0.5) {
    spineTextRec = 'Horizontal text only, small font';
  } else if (spineWidth < 0.75) {
    spineTextRec = 'Rotated text possible';
  } else {
    spineTextRec = 'Full spine design with rotated text';
  }

  return {
    input: {
      pages,
      trimSize: `${trim.width}" × ${trim.height}" (${trim.name})`,
      paperType,
      paperThickness: `${thickness}" per page`
    },
    calculations: {
      spineWidth: `${spineWidth.toFixed(4)}" (${(spineWidth * 25.4).toFixed(2)}mm)`,
      fullWidth: `${fullWidth.toFixed(4)}" (${(fullWidth * 25.4).toFixed(2)}mm)`,
      fullHeight: `${fullHeight.toFixed(4)}" (${(fullHeight * 25.4).toFixed(2)}mm)`,
      bleed: `${BLEED}" on all sides`
    },
    pixels_300dpi: {
      width: Math.round(fullWidth * 300),
      height: Math.round(fullHeight * 300)
    },
    spineText: spineTextRec,
    guides: {
      description: 'Positions in inches from left/top edge',
      horizontal: guides
    }
  };
}

function main() {
  const args = parseArguments();

  if (args.help) {
    showHelp();
    process.exit(0);
  }

  if (!args.pages) {
    console.error('Error: --pages is required\n');
    showHelp();
    process.exit(1);
  }

  const pages = parseInt(args.pages, 10);
  if (isNaN(pages) || pages < 1) {
    console.error('Error: pages must be a positive number');
    process.exit(1);
  }

  try {
    const result = calculateCover(pages, args.trim, args.paper);
    console.log('\n📖 Book Cover Specifications\n');
    console.log('Input:');
    console.log(`  Pages: ${result.input.pages}`);
    console.log(`  Trim Size: ${result.input.trimSize}`);
    console.log(`  Paper: ${result.input.paperType} (${result.input.paperThickness})`);
    console.log('\nCalculated Dimensions:');
    console.log(`  Spine Width: ${result.calculations.spineWidth}`);
    console.log(`  Full Cover Width: ${result.calculations.fullWidth}`);
    console.log(`  Full Cover Height: ${result.calculations.fullHeight}`);
    console.log(`  Bleed: ${result.calculations.bleed}`);
    console.log('\nPixel Dimensions (300 DPI):');
    console.log(`  Width: ${result.pixels_300dpi.width}px`);
    console.log(`  Height: ${result.pixels_300dpi.height}px`);
    console.log(`\nSpine Text: ${result.spineText}`);
    console.log('\n✅ Ready for cover design\n');

    // Also output JSON for programmatic use
    if (process.env.JSON_OUTPUT) {
      console.log(JSON.stringify(result, null, 2));
    }
  } catch (error) {
    console.error(`Error: ${error.message}`);
    process.exit(1);
  }
}

main();
