#!/usr/bin/env node
/**
 * Print Preflight Checker
 *
 * Validates a PDF file for print production readiness.
 * Checks resolution, color space, fonts, and page count.
 *
 * Usage:
 *   node preflight-check.js book.pdf
 *   node preflight-check.js book.pdf --verbose
 *
 * Requires: poppler-utils (pdfinfo, pdffonts, pdfimages)
 */

import { execSync } from 'node:child_process';
import { existsSync } from 'node:fs';
import { parseArgs } from 'node:util';

const options = {
  verbose: { type: 'boolean', short: 'v', default: false },
  help: { type: 'boolean', short: 'h' }
};

function showHelp() {
  console.log(`
Print Preflight Checker

Usage:
  node preflight-check.js <pdf-file> [options]

Options:
  -v, --verbose    Show detailed output
  -h, --help       Show this help

Requirements:
  - poppler-utils (pdfinfo, pdffonts, pdfimages)

Install on Ubuntu/Debian:
  sudo apt-get install poppler-utils
`);
}

function runCommand(cmd) {
  try {
    return execSync(cmd, { encoding: 'utf8', stdio: ['pipe', 'pipe', 'pipe'] });
  } catch (error) {
    return error.stdout || '';
  }
}

function checkPdfInfo(pdfPath, verbose) {
  const issues = [];
  const warnings = [];
  const info = {};

  console.log('📋 Checking PDF information...');

  const output = runCommand(`pdfinfo "${pdfPath}"`);
  if (!output) {
    issues.push('Could not read PDF info. Is poppler-utils installed?');
    return { issues, warnings, info };
  }

  // Parse pdfinfo output
  output.split('\n').forEach(line => {
    const [key, ...valueParts] = line.split(':');
    if (key && valueParts.length) {
      info[key.trim()] = valueParts.join(':').trim();
    }
  });

  if (verbose) {
    console.log('\n  PDF Info:');
    console.log(`    Title: ${info['Title'] || '(not set)'}`);
    console.log(`    Author: ${info['Author'] || '(not set)'}`);
    console.log(`    Pages: ${info['Pages'] || 'unknown'}`);
    console.log(`    Page size: ${info['Page size'] || 'unknown'}`);
    console.log(`    PDF version: ${info['PDF version'] || 'unknown'}`);
  }

  // Check page count divisibility
  const pages = parseInt(info['Pages'], 10);
  if (pages && pages % 4 !== 0) {
    warnings.push(`Page count (${pages}) is not divisible by 4. Add ${4 - (pages % 4)} blank page(s).`);
  }

  return { issues, warnings, info };
}

function checkFonts(pdfPath, verbose) {
  const issues = [];
  const warnings = [];

  console.log('🔤 Checking fonts...');

  const output = runCommand(`pdffonts "${pdfPath}"`);
  if (!output) {
    warnings.push('Could not check fonts (pdffonts not available)');
    return { issues, warnings };
  }

  const lines = output.split('\n').filter(l => l.trim());
  const fonts = [];

  // Skip header lines
  for (let i = 2; i < lines.length; i++) {
    const line = lines[i];
    if (line.trim()) {
      // Parse font line (columns are fixed-width)
      const name = line.substring(0, 36).trim();
      const type = line.substring(36, 48).trim();
      const encoding = line.substring(48, 60).trim();
      const embedded = line.substring(60, 64).trim();
      const subset = line.substring(64, 68).trim();

      if (name) {
        fonts.push({ name, type, encoding, embedded, subset });
      }
    }
  }

  if (verbose && fonts.length > 0) {
    console.log('\n  Fonts found:');
    fonts.forEach(f => {
      const status = f.embedded === 'yes' ? '✓' : '✗';
      console.log(`    ${status} ${f.name} (${f.type}, embedded: ${f.embedded})`);
    });
  }

  // Check for non-embedded fonts
  const notEmbedded = fonts.filter(f => f.embedded !== 'yes');
  if (notEmbedded.length > 0) {
    issues.push(`${notEmbedded.length} font(s) not embedded: ${notEmbedded.map(f => f.name).join(', ')}`);
  }

  if (fonts.length === 0) {
    warnings.push('No fonts found in PDF (may be all images or outlines)');
  }

  return { issues, warnings, fonts };
}

function checkImages(pdfPath, verbose) {
  const issues = [];
  const warnings = [];

  console.log('🖼️  Checking images...');

  const output = runCommand(`pdfimages -list "${pdfPath}"`);
  if (!output) {
    warnings.push('Could not check images (pdfimages not available)');
    return { issues, warnings };
  }

  const lines = output.split('\n').filter(l => l.trim());
  const images = [];

  // Parse image list (skip header)
  for (let i = 2; i < lines.length; i++) {
    const parts = lines[i].split(/\s+/).filter(p => p);
    if (parts.length >= 10) {
      const width = parseInt(parts[3], 10);
      const height = parseInt(parts[4], 10);
      const xppi = parseInt(parts[12], 10) || parseInt(parts[11], 10);
      const yppi = parseInt(parts[13], 10) || parseInt(parts[12], 10);
      const colorSpace = parts[5];

      if (width && height) {
        images.push({ page: parts[0], width, height, xppi, yppi, colorSpace });
      }
    }
  }

  if (verbose && images.length > 0) {
    console.log(`\n  Images found: ${images.length}`);
    images.slice(0, 10).forEach(img => {
      const dpi = img.xppi || 'unknown';
      console.log(`    Page ${img.page}: ${img.width}×${img.height}, ${dpi} DPI, ${img.colorSpace}`);
    });
    if (images.length > 10) {
      console.log(`    ... and ${images.length - 10} more`);
    }
  }

  // Check for low resolution images
  const lowRes = images.filter(img => img.xppi && img.xppi < 300);
  if (lowRes.length > 0) {
    warnings.push(`${lowRes.length} image(s) below 300 DPI (may appear pixelated)`);
  }

  // Check for RGB in what should be CMYK (cover files)
  const rgbImages = images.filter(img => img.colorSpace === 'rgb');
  if (rgbImages.length > 0 && pdfPath.toLowerCase().includes('cover')) {
    warnings.push(`${rgbImages.length} RGB image(s) found. Cover should use CMYK.`);
  }

  return { issues, warnings, images };
}

function printReport(results) {
  console.log('\n' + '='.repeat(50));
  console.log('📊 PREFLIGHT REPORT');
  console.log('='.repeat(50));

  const allIssues = [
    ...results.pdfInfo.issues,
    ...results.fonts.issues,
    ...results.images.issues
  ];

  const allWarnings = [
    ...results.pdfInfo.warnings,
    ...results.fonts.warnings,
    ...results.images.warnings
  ];

  if (allIssues.length === 0 && allWarnings.length === 0) {
    console.log('\n✅ All checks passed! PDF is ready for print.\n');
    return 0;
  }

  if (allIssues.length > 0) {
    console.log('\n❌ ISSUES (must fix):');
    allIssues.forEach((issue, i) => {
      console.log(`   ${i + 1}. ${issue}`);
    });
  }

  if (allWarnings.length > 0) {
    console.log('\n⚠️  WARNINGS (review):');
    allWarnings.forEach((warning, i) => {
      console.log(`   ${i + 1}. ${warning}`);
    });
  }

  console.log('\n');

  return allIssues.length > 0 ? 1 : 0;
}

function main() {
  let args;
  try {
    const parsed = parseArgs({ options, allowPositionals: true });
    args = { ...parsed.values, positionals: parsed.positionals };
  } catch (error) {
    console.error('Error parsing arguments');
    showHelp();
    process.exit(1);
  }

  if (args.help) {
    showHelp();
    process.exit(0);
  }

  const pdfPath = args.positionals[0];
  if (!pdfPath) {
    console.error('Error: PDF file path required\n');
    showHelp();
    process.exit(1);
  }

  if (!existsSync(pdfPath)) {
    console.error(`Error: File not found: ${pdfPath}`);
    process.exit(1);
  }

  console.log(`\n🔍 Preflight check: ${pdfPath}\n`);

  const results = {
    pdfInfo: checkPdfInfo(pdfPath, args.verbose),
    fonts: checkFonts(pdfPath, args.verbose),
    images: checkImages(pdfPath, args.verbose)
  };

  const exitCode = printReport(results);
  process.exit(exitCode);
}

main();
