#!/usr/bin/env node
/**
 * Manuscript Preparation Script
 *
 * Cleans and formats a markdown manuscript for book production.
 * Standardizes headings, scene breaks, quotation marks, and spacing.
 *
 * Usage:
 *   node prepare-manuscript.js input.md -o output.md
 *   node prepare-manuscript.js input.md --check  (dry run)
 */

import { readFileSync, writeFileSync } from 'node:fs';
import { parseArgs } from 'node:util';

const options = {
  output: { type: 'string', short: 'o' },
  check: { type: 'boolean', short: 'c', default: false },
  verbose: { type: 'boolean', short: 'v', default: false },
  help: { type: 'boolean', short: 'h' }
};

function showHelp() {
  console.log(`
Manuscript Preparation Script

Cleans and formats markdown for book production.

Usage:
  node prepare-manuscript.js <input.md> [options]

Options:
  -o, --output <file>   Output file (default: overwrites input)
  -c, --check           Dry run - show changes without writing
  -v, --verbose         Show detailed changes
  -h, --help            Show this help

Transformations:
  - Standardizes chapter headings
  - Converts scene breaks to ---
  - Fixes quotation marks (straight → curly)
  - Normalizes dashes (-- → em dash)
  - Removes extra blank lines
  - Trims trailing whitespace
`);
}

function fixQuotationMarks(text) {
  // Opening double quote: after space, newline, or start
  text = text.replace(/(^|[\s\n(])"/g, '$1"');
  // Closing double quote: before space, newline, punctuation, or end
  text = text.replace(/"([\s\n.,!?;:)\-]|$)/g, '"$1');

  // Opening single quote/apostrophe at word start
  text = text.replace(/(^|[\s\n(])'/g, '$1'');
  // Apostrophe within words
  text = text.replace(/(\w)'(\w)/g, '$1'$2');
  // Closing single quote
  text = text.replace(/'([\s\n.,!?;:)\-]|$)/g, ''$1');

  return text;
}

function fixDashes(text) {
  // Convert -- to em dash
  text = text.replace(/--/g, '—');
  // Convert spaced hyphens to en dash (ranges)
  text = text.replace(/(\d)\s*-\s*(\d)/g, '$1–$2');
  return text;
}

function fixEllipsis(text) {
  // Convert three periods to proper ellipsis
  text = text.replace(/\.\.\./g, '…');
  return text;
}

function normalizeSceneBreaks(text) {
  // Convert various scene break patterns to standard ---
  const patterns = [
    /^\s*\*\s*\*\s*\*\s*$/gm,           // * * *
    /^\s*\*{3,}\s*$/gm,                  // ***
    /^\s*#\s*$/gm,                       // #
    /^\s*~{3,}\s*$/gm,                   // ~~~
    /^\s*-\s+-\s+-\s*$/gm,              // - - -
    /^\s*•\s*•\s*•\s*$/gm,              // • • •
  ];

  patterns.forEach(pattern => {
    text = text.replace(pattern, '\n---\n');
  });

  return text;
}

function normalizeChapterHeadings(text) {
  // Standardize chapter heading formats
  // "Chapter 1: Title" or "Chapter One: Title" → "# Chapter 1: Title"

  // Already markdown heading
  text = text.replace(/^#+\s*(Chapter\s+\w+)/gim, '# $1');

  // Plain text chapter heading (add # prefix)
  text = text.replace(/^(Chapter\s+\w+[:\s].*)$/gim, (match, p1) => {
    if (!match.startsWith('#')) {
      return `# ${p1}`;
    }
    return match;
  });

  return text;
}

function removeExtraBlankLines(text) {
  // Reduce multiple blank lines to maximum of two
  text = text.replace(/\n{4,}/g, '\n\n\n');
  return text;
}

function trimTrailingWhitespace(text) {
  return text.split('\n').map(line => line.trimEnd()).join('\n');
}

function countChanges(original, modified) {
  const changes = {
    quotationMarks: 0,
    dashes: 0,
    ellipses: 0,
    sceneBreaks: 0,
    chapterHeadings: 0,
    blankLines: 0,
    whitespace: 0
  };

  // Count straight quotes replaced
  changes.quotationMarks = (original.match(/"/g) || []).length -
                           (modified.match(/"/g) || []).length;

  // Count -- replaced
  changes.dashes = (original.match(/--/g) || []).length;

  // Count ... replaced
  changes.ellipses = (original.match(/\.\.\./g) || []).length;

  return changes;
}

function processManuscript(text, verbose) {
  const original = text;
  const log = [];

  // Apply transformations in order
  text = fixQuotationMarks(text);
  if (verbose) log.push('✓ Fixed quotation marks');

  text = fixDashes(text);
  if (verbose) log.push('✓ Fixed dashes');

  text = fixEllipsis(text);
  if (verbose) log.push('✓ Fixed ellipses');

  text = normalizeSceneBreaks(text);
  if (verbose) log.push('✓ Normalized scene breaks');

  text = normalizeChapterHeadings(text);
  if (verbose) log.push('✓ Standardized chapter headings');

  text = removeExtraBlankLines(text);
  if (verbose) log.push('✓ Removed extra blank lines');

  text = trimTrailingWhitespace(text);
  if (verbose) log.push('✓ Trimmed trailing whitespace');

  // Ensure file ends with newline
  if (!text.endsWith('\n')) {
    text += '\n';
  }

  return {
    text,
    log,
    changed: text !== original,
    changes: countChanges(original, text)
  };
}

function main() {
  let args;
  try {
    const parsed = parseArgs({ options, allowPositionals: true });
    args = { ...parsed.values, positionals: parsed.positionals };
  } catch (error) {
    console.error('Error parsing arguments');
    showHelp();
    process.exit(1);
  }

  if (args.help) {
    showHelp();
    process.exit(0);
  }

  const inputPath = args.positionals[0];
  if (!inputPath) {
    console.error('Error: Input file required\n');
    showHelp();
    process.exit(1);
  }

  let text;
  try {
    text = readFileSync(inputPath, 'utf8');
  } catch (error) {
    console.error(`Error reading file: ${error.message}`);
    process.exit(1);
  }

  console.log(`\n📝 Processing: ${inputPath}`);
  console.log(`   Original size: ${text.length.toLocaleString()} characters\n`);

  const result = processManuscript(text, args.verbose);

  if (args.verbose) {
    result.log.forEach(line => console.log(`   ${line}`));
    console.log('');
  }

  if (!result.changed) {
    console.log('✅ No changes needed - manuscript is clean!\n');
    process.exit(0);
  }

  // Show summary
  console.log('📊 Changes:');
  if (result.changes.quotationMarks > 0) {
    console.log(`   • ${result.changes.quotationMarks} quotation marks converted`);
  }
  if (result.changes.dashes > 0) {
    console.log(`   • ${result.changes.dashes} dashes converted to em dashes`);
  }
  if (result.changes.ellipses > 0) {
    console.log(`   • ${result.changes.ellipses} ellipses converted`);
  }

  if (args.check) {
    console.log('\n🔍 Dry run - no files modified\n');
    process.exit(0);
  }

  // Write output
  const outputPath = args.output || inputPath;
  try {
    writeFileSync(outputPath, result.text, 'utf8');
    console.log(`\n✅ Saved to: ${outputPath}`);
    console.log(`   New size: ${result.text.length.toLocaleString()} characters\n`);
  } catch (error) {
    console.error(`Error writing file: ${error.message}`);
    process.exit(1);
  }
}

main();
