# Documentation to Skill Conversion Methodology

Complete methodology for converting documentation into Agent Skills.

## Overview

Converting documentation to skills is not about copying content - it's about **reorganizing information** for AI agent consumption using progressive disclosure principles.

## Core Philosophy

### Progressive Disclosure

Information is loaded in three stages:

1. **Metadata** (always loaded)
   - `name`: Skill identifier
   - `description`: Trigger mechanism with use cases
   - `metadata.short-description`: Brief summary

2. **SKILL.md** (loaded when skill triggers)
   - Essential content only
   - Most common workflows
   - Quick reference
   - Links to detailed docs

3. **references/** (loaded only when needed)
   - Complete documentation
   - Detailed specifications
   - Advanced patterns
   - Edge cases

### Why This Matters

- **Context window efficiency**: Only load what's needed
- **Faster triggering**: Metadata is always available
- **Complete coverage**: References provide full details
- **Better organization**: Clear separation of concerns

## Conversion Process

### Phase 1: Analysis

**1.1 Document Structure Analysis**

Analyze the source document:
- Total length and section count
- Information hierarchy
- Content types (tutorial, reference, examples)
- Dependencies between sections

**Output:** Document structure map

**1.2 Usage Pattern Analysis**

Identify:
- Most frequently accessed content (80/20 rule)
- Common workflows vs. edge cases
- Quick reference needs vs. deep dives
- User journey through the content

**Output:** Usage priority list

**1.3 Trigger Scenario Analysis**

Determine when this skill should be used:
- What problems does it solve?
- What tasks does it enable?
- What contexts trigger its use?
- What are all the use cases?

**Output:** Complete list of trigger scenarios

### Phase 2: Extraction

**2.1 Core Content Extraction**

Extract for SKILL.md:
- Quick start / getting started
- Most common operations (top 20%)
- Essential workflows
- Basic configuration
- Common patterns
- Basic troubleshooting

**Criteria:**
- Used in 80% of cases
- Needed for quick reference
- Essential for understanding
- Frequently accessed

**2.2 Reference Content Extraction**

Extract for references/:
- Complete command/API reference
- Detailed specifications
- Advanced patterns
- Edge cases
- Comprehensive examples
- Deep dives

**Criteria:**
- Used in 20% of cases
- Complete reference needed
- Detailed specifications
- Advanced topics

### Phase 3: Organization

**3.1 SKILL.md Structure**

Organize core content:
```
# Title
Brief overview

## Quick Start
Most common operations

## Core Workflows
Organized by task

## Configuration
Essential settings

## Common Patterns
Best practices

## Troubleshooting
Common issues

## References
Links to references/
```

**Guidelines:**
- Keep under 500 lines (ideally 200-300)
- One concept per section
- Actionable content first
- Code examples for everything

**3.2 References Structure**

Organize detailed content:
```
references/
├── commands.md      # Complete command/API reference
├── workflows.md     # Detailed workflow scenarios
├── advanced.md      # Advanced patterns (optional)
└── examples.md      # Comprehensive examples (optional)
```

**Guidelines:**
- One file per major topic
- Complete coverage
- Detailed explanations
- All options/flags documented

### Phase 4: Writing

**4.1 Frontmatter**

Write comprehensive description:
```yaml
name: skill-name
description: [What it does]. Use when (1) [Scenario], (2) [Scenario], ...
metadata:
  short-description: Brief summary
```

**Description requirements:**
- Clear "what it does" statement
- Numbered list of ALL trigger scenarios
- No colons in description field
- Comprehensive use case coverage

**4.2 SKILL.md Body**

Write with these principles:
- **Actionability**: Prioritize executable code
- **Clarity**: One concept per section
- **Completeness**: Cover all core workflows
- **Conciseness**: No unnecessary words
- **Examples**: Real, runnable code

**4.3 References**

Write detailed documentation:
- Complete coverage
- All options/flags
- Detailed explanations
- Comprehensive examples
- Edge cases

### Phase 5: Validation

**5.1 Format Validation**

Check:
- Frontmatter format (YAML)
- No colons in description
- All required fields present
- References links valid

**5.2 Content Validation**

Check:
- SKILL.md under 500 lines
- No duplication between SKILL.md and references/
- All core workflows covered
- References properly linked

**5.3 Functional Validation**

Check:
- Code examples are runnable
- Commands are correct
- Examples match current versions
- Links work

## Quality Criteria

### SKILL.md Quality

**Excellent:**
- ✅ Under 300 lines
- ✅ All core workflows covered
- ✅ Actionable examples throughout
- ✅ Clear references to detailed docs
- ✅ No duplication

**Good:**
- ✅ Under 500 lines
- ✅ Most core workflows covered
- ✅ Some examples
- ✅ References mentioned

**Needs Improvement:**
- ❌ Over 500 lines
- ❌ Missing core workflows
- ❌ Few/no examples
- ❌ Duplication with references/

### Description Quality

**Excellent:**
- ✅ Clear "what it does"
- ✅ 5+ trigger scenarios
- ✅ Covers all use cases
- ✅ Specific and actionable

**Good:**
- ✅ Clear purpose
- ✅ 3-4 trigger scenarios
- ✅ Covers most use cases

**Needs Improvement:**
- ❌ Vague purpose
- ❌ 1-2 trigger scenarios
- ❌ Missing use cases

## Common Mistakes

### Mistake 1: Copy-Paste Conversion

**Wrong:** Copy entire document to SKILL.md

**Right:** Extract and reorganize core content

### Mistake 2: Missing Trigger Scenarios

**Wrong:** Vague description like "GitHub CLI guide"

**Right:** "Use when (1) Managing repos, (2) Creating PRs, (3) ..."

### Mistake 3: Duplication

**Wrong:** Same content in SKILL.md and references/

**Right:** SKILL.md = quick ref, references/ = complete docs

### Mistake 4: Too Much in SKILL.md

**Wrong:** 1000+ lines in SKILL.md

**Right:** <500 lines, details in references/

### Mistake 5: No References

**Wrong:** Everything in SKILL.md

**Right:** Core in SKILL.md, details in references/

## Best Practices

### 1. Start with Use Cases

Before writing, list all use cases:
- What problems does this solve?
- When would someone need this?
- What are the common workflows?

### 2. Apply 80/20 Rule

Identify the 20% of content used 80% of the time:
- Most common commands → SKILL.md
- All commands with all flags → references/

### 3. Prioritize Actionability

Prefer:
- Executable code
- Real examples
- Common patterns

Over:
- Theory
- Edge cases
- Historical context

### 4. Test Examples

All code examples should be:
- Runnable
- Current
- Tested
- Correct

### 5. Link, Don't Duplicate

In SKILL.md:
- Quick reference only
- Link to detailed docs
- Don't copy content

In references/:
- Complete documentation
- All details
- Comprehensive coverage

## Conversion Metrics

Track these metrics:

**Efficiency:**
- SKILL.md lines (target: <300)
- References files count
- Total content lines

**Coverage:**
- Core workflows covered (%)
- Trigger scenarios listed
- Use cases addressed

**Quality:**
- Code examples count
- Duplication instances
- Broken links count

## Iteration Process

**After initial conversion:**

1. **Test the skill** in real scenarios
2. **Identify gaps** (missing workflows, unclear triggers)
3. **Refine content** (add examples, clarify instructions)
4. **Optimize structure** (reorganize if needed)
5. **Update references** (add missing details)

**Iteration cycle:**
```
Convert → Test → Refine → Test → Package
```

## Tools and Automation

**Manual process:**
- Analyze document
- Extract content
- Write skill
- Validate

**Potential automation:**
- Document structure analysis
- Content extraction (80/20)
- Frontmatter generation
- Validation checks

**Current tools:**
- `init_skill.py` - Initialize skill structure
- `package_skill.py` - Package and validate
- `quick_validate.py` - Validate frontmatter

## Conclusion

Converting documentation to skills is a **design process**, not a mechanical conversion. The goal is to create a skill that:

- ✅ Triggers correctly (comprehensive description)
- ✅ Loads efficiently (progressive disclosure)
- ✅ Provides quick answers (SKILL.md)
- ✅ Offers complete details (references/)
- ✅ Follows best practices (no duplication, actionable)

Focus on **user needs** and **AI agent efficiency**, not document structure.
