# Docling スキル

Doclingを使用したドキュメント読み取り・変換スキルです。PDF、DOCX、PPTXなどの各種ドキュメントをMarkdown、JSON、HTMLに変換し、図・写真を個別のファイルに抽出できます。

## 主な機能

- **文書変換**: PDF, DOCX, PPTX, XLSX, HTML, Markdown などに対応
- **画像抽出**: 図はPNG、写真はJPEGとして`images/`ディレクトリに保存
- **表抽出**: Markdown形式で表を出力
- **OCR対応**: スキャンされたPDFからのテキスト抽出
- **大容量ファイル対応**: ページ範囲指定またはバッチ処理スクリプト生成

## スキルの起動条件

このスキルは以下の場合に自動的に起動します：

1. **文書ファイルを読み取る指示があった場合**
   - 「このPDFを読んで」「read this document」「ファイルの内容を確認して」など

2. **対応ファイル形式を処理する場合**
   - PDF, DOCX, PPTX, XLSX, HTML, Markdown, AsciiDoc
   - 画像ファイル: PNG, JPG, TIFF

3. **OCRでスキャン文書からテキスト抽出する場合**

4. **ドキュメントをMarkdown/JSON/HTMLに変換する場合**

5. **図・写真を抽出する場合**

## インストール

### 1. Doclingのインストール

```bash
pip install docling
```

### 2. ページ範囲処理用（オプション・推奨）

```bash
pip install pymupdf
```

### 3. OCRエンジン（オプション）

**EasyOCR**（推奨・自動インストール）:
```bash
pip install easyocr
```

**Tesseract**:
```bash
# macOS
brew install tesseract tesseract-lang

# Ubuntu/Debian
sudo apt-get install tesseract-ocr tesseract-ocr-jpn
```

## 使い方

### 基本的な変換

```bash
# PDF→Markdown変換（画像も抽出）
python scripts/convert_document.py document.pdf -o ./output

# すべての形式で出力
python scripts/convert_document.py document.pdf -o ./output -f all
```

### 出力ディレクトリ構造

```
output/
├── document.md       # Markdownファイル（画像リンク埋め込み）
├── document.json     # 構造化JSONデータ
├── document.html     # HTMLファイル（オプション）
└── images/
    ├── figure_001.png   # 図・グラフ（PNG形式）
    ├── figure_002.png
    ├── photo_001.jpg    # 写真（JPEG形式）
    └── photo_002.jpg
```

### OCR付き変換

```bash
# OCR有効化（スキャンPDF向け）
python scripts/convert_document.py scanned.pdf -o ./output --ocr

# 日本語OCR
python scripts/convert_document.py document.pdf -o ./output --ocr --languages ja en
```

### 大きなファイルの処理

#### ページ範囲指定

```bash
# 1-20ページを処理
python scripts/convert_document.py large.pdf -o ./output --pages 1-20

# 21-40ページを処理
python scripts/convert_document.py large.pdf -o ./output2 --pages 21-40
```

#### バッチ処理スクリプト生成

```bash
# バッチ処理スクリプトを生成
python scripts/convert_document.py large.pdf -o ./output --generate-script --batch-size 20

# 生成されたスクリプトを実行
python ./output/batch_process.py
```

### その他のオプション

```bash
# 高精度テーブル抽出
python scripts/convert_document.py document.pdf -o ./output --table-mode accurate

# JSON形式で出力
python scripts/convert_document.py document.pdf -o ./output -f json

# HTML形式で出力
python scripts/convert_document.py document.pdf -o ./output -f html
```

## スクリプトオプション一覧

| オプション | 説明 | デフォルト |
|-----------|------|-----------|
| `-o, --output` | 出力ディレクトリ（必須） | - |
| `-f, --format` | 出力形式 (markdown, json, html, all) | markdown |
| `--ocr` | OCRを有効化 | 無効 |
| `--ocr-engine` | OCRエンジン (easyocr, tesseract) | easyocr |
| `--languages` | OCR言語 | en ja |
| `--table-mode` | テーブル抽出モード (fast, accurate) | fast |
| `--pages` | ページ範囲 (例: 1-20) | 全ページ |
| `--generate-script` | バッチ処理スクリプトを生成 | - |
| `--batch-size` | バッチあたりのページ数 | 20 |

## Pythonからの使用

```python
from docling.document_converter import DocumentConverter

# シンプルな変換
converter = DocumentConverter()
result = converter.convert("document.pdf")

# Markdown出力
print(result.document.export_to_markdown())

# JSON出力
data = result.document.export_to_dict()

# ファイルに保存
result.document.save_as_markdown("output.md")
```

### 変換スクリプトを直接呼び出し

```python
import sys
sys.path.insert(0, "scripts")
from convert_document import convert_document

result = convert_document(
    source="document.pdf",
    output_dir="./output",
    output_format="all",  # markdown, json, html, all
    ocr=False,
    table_mode="fast",  # fast or accurate
)

print(f"Files created: {result['files']}")
print(f"Images extracted: {result['image_count']}")
print(f"Tables found: {result['table_count']}")
```

## 対応フォーマット

### 入力形式
- PDF（テキスト/スキャン両対応）
- Microsoft Office: DOCX, PPTX, XLSX
- Web: HTML, Markdown
- 画像: PNG, JPG, TIFF（OCR経由）

### 出力形式
- Markdown（画像リンク埋め込み）
- JSON（構造化データ）
- HTML

## OCR言語コード

よく使う言語コード:

| コード | 言語 |
|--------|------|
| `ja` | 日本語 |
| `en` | 英語 |
| `zh` | 中国語（簡体字） |
| `ko` | 韓国語 |
| `de` | ドイツ語 |
| `fr` | フランス語 |

## トラブルシューティング

### "docling is not installed" エラー
```bash
pip install docling
```

### PyMuPDFがない場合のページ範囲処理
```bash
pip install pymupdf
```

### OCRが遅い場合
- `--table-mode fast` を使用（デフォルト）
- GPU環境では自動的にGPUを使用

### 日本語が文字化けする場合
```bash
python scripts/convert_document.py document.pdf -o ./output --ocr --languages ja
```

### 大きなファイルでメモリ不足
```bash
# ページ範囲指定で分割処理
python scripts/convert_document.py large.pdf -o ./output --pages 1-20

# またはバッチスクリプト生成
python scripts/convert_document.py large.pdf -o ./output --generate-script --batch-size 10
```

## 詳細設定

詳細なオプションについては `references/advanced-options.md` を参照してください。

## 参考リンク

- [Docling GitHub](https://github.com/docling-project/docling)
- [Docling Documentation](https://docling-project.github.io/docling/)
- [Docling PyPI](https://pypi.org/project/docling/)
