# Playtest Procedure for Escape Room Games

Complete protocol for alpha/beta testing and metrics analysis to achieve 60-70% completion rate.

## Testing Phases

### Phase 1: Internal Alpha Test (5 people, Week 11)

**Participants**:
- 2 friends (game experience: casual)
- 2 colleagues (game experience: moderate)
- 1 family member (game experience: none)

**Purpose**: Find critical bugs, impossible puzzles

**Setup**:
```
Preparation:
- [ ] Clean test environment (fresh database)
- [ ] Create 5 test accounts
- [ ] Prepare observation sheet
- [ ] Set up screen recording (with permission)

During Test:
- [ ] No hints given (observe natural behavior)
- [ ] Take notes (when stuck, confusion points)
- [ ] Record time per scene
- [ ] Note all bugs immediately

After Test:
- [ ] Debrief interview (10 min each)
- [ ] Collect satisfaction rating
- [ ] Prioritize issues (critical/high/medium/low)
```

**Metrics to Track**:

| Metric | Target | Red Flag |
|--------|--------|----------|
| Completion rate | 60%+ (3/5) | <40% (too hard) |
| Average time | 90-150 min | >180 min (too long) |
| Critical bugs | 0 | Any |
| Stuck points | <2 scenes | >3 scenes |
| Satisfaction | 4+ / 5 | <3.5 |

**Data Collection Sheet**:

```
Tester: [Name]
Game Experience: [None/Casual/Moderate/Hardcore]
Started: [Time]
Ended: [Time]
Completed: Yes / No (which scene stopped?)

Scene Times:
- Scene 0: ___ min
- Scene 1: ___ min
- ...
- Scene 14: ___ min

Stuck Points:
- Scene X: [Description of confusion]
- Puzzle Y: [What they tried, why it didn't work]

Hints Used: ___ (tier-1), ___ (tier-2), ___ (tier-3)

Bugs Found:
1. [Critical/High/Medium/Low] [Description]
2. ...

Satisfaction: ☐ 1 ☐ 2 ☐ 3 ☐ 4 ☐ 5

Comments:
[Free-form feedback]
```

**Success Criteria**:
- ✅ 3/5 complete (60%)
- ✅ 0 critical bugs
- ✅ Average satisfaction 4+
- ✅ Clear fixes for stuck points

**If Failed**: Fix critical issues, repeat alpha with new 5 people

---

### Phase 2: External Beta Test (20 people, Week 12)

**Recruitment**:
- Discord community: 10 people
- Friends of friends: 5 people
- Reddit r/escaperooms: 5 people

**Incentive**: Free game key + credit in game (Special Thanks section)

**Setup**:

```
Pre-Test:
- [ ] Send welcome email with instructions
- [ ] Create beta Discord channel
- [ ] Enable telemetry (automatic event tracking)
- [ ] Prepare post-test survey (Google Forms)

During Test (7 days):
- [ ] Daily check-ins in Discord
- [ ] Answer questions within 2 hours
- [ ] Monitor telemetry dashboard
- [ ] Emergency hotfixes for critical bugs

Post-Test:
- [ ] Send survey (24 hours after completion)
- [ ] Analyze telemetry data
- [ ] Generate balance report
- [ ] Plan adjustments
```

**Telemetry Events**:

```typescript
// Automatic tracking
analytics.track('scene_entered', { sceneId, timestamp })
analytics.track('scene_completed', { sceneId, duration, hintsUsed })
analytics.track('evidence_discovered', { evidenceId, method })
analytics.track('puzzle_attempted', { puzzleId, answer, correct })
analytics.track('puzzle_solved', { puzzleId, attempts, timeSpent })
analytics.track('hint_requested', { puzzleId, tier })
analytics.track('npc_conversation', { characterId, messageCount })
analytics.track('game_completed', { endingType, totalTime, hintsUsed })
analytics.track('game_abandoned', { lastScene, reason })
```

**Post-Test Survey**:

```
1. Did you complete the game?
   ☐ Yes ☐ No (stopped at Scene ___)

2. Total play time: ___ hours ___ minutes

3. Difficulty rating: ☐ Too Easy ☐ Just Right ☐ Too Hard

4. Which scene was hardest? ___

5. How many hints did you use? ___

6. Did you enjoy the story? ☐ 1 ☐ 2 ☐ 3 ☐ 4 ☐ 5

7. Would you play another episode? ☐ Yes ☐ No ☐ Maybe

8. Would you recommend to friends? ☐ Yes ☐ No

9. What frustrated you most? [Open text]

10. What did you love most? [Open text]
```

**Analysis Targets**:

| Metric | Target | Action if Missed |
|--------|--------|------------------|
| **Completion rate** | 60-70% | <50%: Simplify hardest puzzles, add tier-0 hints |
| **Average time** | 120 min ± 20 | >150min: Reduce puzzle steps, clearer clues |
| **Stuck scenes** | <3 scenes with >30% stuck | Fix each stuck scene (clues, hints, affordances) |
| **Hint usage** | 3-5 tier-3 hints avg | >8: Puzzles too hard; <2: Puzzles too easy |
| **Satisfaction** | 4.5+ / 5 | <4: Identify pain points from comments |
| **Replay intent** | 40%+ | <30%: Enhance ending variations, hidden content |
| **Recommendation** | 80%+ | <70%: Address top frustrations |

---

## Stuck-Point Resolution Protocol

### Identification

**Stuck Point** = Scene where >30% of players spend >10 min without progress

**Example Analysis**:

```
Scene 7 - "피해자의 금고":
- 45% of players stuck (9/20)
- Average time stuck: 18 minutes
- Common behavior: Tried 10+ wrong passwords
- Root cause: Clue too subtle (hidden in background image detail)

Priority Score: 45% × 18 min = 810 points → CRITICAL
```

### Resolution Strategies

**Strategy 1: Visual Cue Enhancement**

```typescript
// Before: Clue hidden in static image
<img src="/evidence/desk-photo.jpg" />

// After: Pulsing glow on interactive element
<div className="relative">
  <img src="/evidence/desk-photo.jpg" />
  <div className="absolute top-20 left-40 w-16 h-16 animate-pulse ring-4 ring-yellow-400" />
</div>
```

**Strategy 2: Progressive Hint Unlocking**

```typescript
// Before: Hints after 5 failed attempts
// After: Graduated hints by time

if (timeInScene > 5 * 60) {
  showHint("Look at the desk carefully")
}

if (timeInScene > 8 * 60) {
  showHint("The calendar on the desk has a marked date")
}

if (timeInScene > 12 * 60) {
  showHint("The password is 0214 (February 14th)")
}
```

**Strategy 3: Simplify Puzzle**

```typescript
// Before: 4-step cipher
// 1. Find encrypted message in image
// 2. Identify cipher type (Caesar)
// 3. Determine shift amount (from another clue)
// 4. Decode to get password

// After: 2-step cipher
// 1. Find encrypted message (more obvious)
// 2. Decode using provided tool (built-in cipher decoder)
```

---

## Automated Balance Analysis

**Script**: `scripts/analyze_playtests.py`

```python
import json
import statistics

def analyze_playtests(telemetry_data):
    """Generate balance report from telemetry."""

    total_players = len(telemetry_data)
    completed = [p for p in telemetry_data if p['completed']]
    completion_rate = len(completed) / total_players

    # Scene times analysis
    scene_times = {}
    for scene_id in range(15):
        times = [p['scene_times'][scene_id] for p in telemetry_data if scene_id in p['scene_times']]
        scene_times[scene_id] = {
            'avg': statistics.mean(times),
            'median': statistics.median(times),
            'max': max(times),
            'stuck_rate': len([t for t in times if t > 10]) / len(times)  # >10min = stuck
        }

    # Identify stuck scenes
    stuck_scenes = [
        sid for sid, data in scene_times.items()
        if data['stuck_rate'] > 0.3  # >30% players stuck
    ]

    # Hint usage analysis
    avg_hints = statistics.mean([p['hint_count'] for p in telemetry_data])

    # Generate report
    return {
        'completion_rate': completion_rate,
        'avg_time': statistics.mean([p['total_time'] for p in completed]),
        'stuck_scenes': stuck_scenes,
        'avg_hints': avg_hints,
        'recommendations': generate_recommendations(completion_rate, stuck_scenes, avg_hints)
    }

def generate_recommendations(completion_rate, stuck_scenes, avg_hints):
    """Generate actionable recommendations."""

    recs = []

    if completion_rate < 0.5:
        recs.append("CRITICAL: Completion rate too low ({}%). Simplify hardest puzzles.".format(completion_rate))

    if len(stuck_scenes) > 3:
        recs.append("HIGH: {} stuck scenes detected: {}. Review each for clarity.".format(len(stuck_scenes), stuck_scenes))

    if avg_hints > 8:
        recs.append("MEDIUM: High hint usage (avg {}). Puzzles may be too obscure.".format(avg_hints))

    if completion_rate > 0.8:
        recs.append("LOW: Completion rate very high ({}%). Consider increasing difficulty.".format(completion_rate))

    return recs
```

**Output Example**:

```
=== BALANCE REPORT ===

Completion Rate: 65% (13/20) ✅ TARGET MET
Average Time: 125 min ✅ WITHIN RANGE (target: 120 ± 20)
Hint Usage: 4.2 avg ✅ OPTIMAL (target: 3-5)

Stuck Scenes:
- Scene 7: 45% stuck, avg 18 min → CRITICAL FIX NEEDED
- Scene 11: 35% stuck, avg 12 min → HIGH PRIORITY

Recommendations:
1. Scene 7: Add visual pulse to safe clue location
2. Scene 11: Simplify 4-step puzzle to 3-step
3. Overall: Excellent balance, minor tweaks only

READY FOR LAUNCH: ✅
```

---

## Difficulty Adjustment Formulas

### If Completion Rate Too Low (<50%)

**Immediate Fixes**:
```
1. Add Tier-0 Hints (auto-show after 2 min stuck)
2. Highlight interactive elements (pulsing glow)
3. Simplify top 3 hardest puzzles (reduce steps)
4. Make critical clues more obvious
```

**Example**:
```typescript
// Scene 7 stuck rate: 45%
// Fix: Add auto-hint after 5 min

useEffect(() => {
  if (sceneId === 'scene_7' && timeInScene > 300) {
    showHint("Look at the desk calendar for a marked date")
  }
}, [timeInScene])
```

### If Completion Rate Too High (>80%)

**Challenge Enhancements**:
```
1. Remove some Tier-1 hints
2. Add red herrings (misleading clues)
3. Increase puzzle step count
4. Make some clues more subtle
```

**Example**:
```typescript
// Add red herring to Scene 3
const redHerringClue = {
  title: "의심스러운 영수증",
  description: "박서준의 책상에서 발견한 영수증. 사건 당일 밤 편의점...",
  // Misleads to박서준, but he's innocent
  suspicion_impact: { "박서준": 15 }
}
```

---

## Post-Launch Monitoring

**Continuous Telemetry**:

```typescript
// Dashboard metrics (update every hour)
const liveMetrics = {
  totalPlayers: count(all_sessions),
  activeNow: count(sessions_last_hour),
  completionRate: completed / total,
  avgTime: average(completed_times),
  topStuckScenes: scenes_sorted_by_stuck_rate.slice(0, 3),
  crashRate: errors / total_sessions
}

// Alerts
if (liveMetrics.completionRate < 0.5) {
  sendSlackAlert("🚨 Completion rate dropped to 45%! Investigate.")
}

if (liveMetrics.crashRate > 0.05) {
  sendSlackAlert("🔥 Crash rate 5%! Critical bug likely.")
}
```

**Hotfix Protocol**:

```
If >10% crash rate OR >3 critical bugs reported:
1. Pause marketing (prevent new broken experiences)
2. Deploy hotfix within 4 hours
3. Notify affected players (email/Discord)
4. Offer compensation (free DLC)
```

---

## Balancing Checklist

```
Pre-Alpha:
- [ ] Difficulty curve designed (Easy → Peak → Moderate)
- [ ] 15 scenes assigned difficulty levels
- [ ] 16 puzzles calibrated
- [ ] 3-tier hints written for each puzzle

Alpha Test (5 people):
- [ ] Completion rate measured
- [ ] Stuck points identified
- [ ] Critical bugs fixed
- [ ] Satisfaction >4/5

Post-Alpha Adjustments:
- [ ] Stuck scenes fixed (visual cues, hints)
- [ ] Impossible puzzles simplified
- [ ] Critical bugs resolved

Beta Test (20 people):
- [ ] Telemetry enabled
- [ ] 60-70% completion rate validated
- [ ] No stuck scenes (>30% threshold)
- [ ] Average hints 3-5
- [ ] Satisfaction 4.5+

Pre-Launch:
- [ ] All metrics in target range
- [ ] Zero critical bugs
- [ ] Lighthouse 90+ (performance)
- [ ] Final polish complete
```

---

**Total Tests**: 25 players (5 alpha + 20 beta)
**Total Budget**: ~50 hours testing time
**Target Metrics**: 65% completion, 4.5 satisfaction, 120 min avg time
