# FastAPI Testing Patterns

## Overview

This document covers comprehensive testing patterns for FastAPI applications using pytest and TestClient. These patterns help ensure your APIs are reliable, maintainable, and properly validated.

## Basic Testing Patterns

### Simple Endpoint Testing
```python
from fastapi.testclient import TestClient
from app.main import app

client = TestClient(app)

def test_health_endpoint():
    response = client.get("/health")
    assert response.status_code == 200
    assert response.json() == {"status": "healthy"}
```

### Testing Different HTTP Methods
```python
def test_post_request():
    response = client.post("/items/", json={"name": "test", "price": 10.5})
    assert response.status_code == 200
    data = response.json()
    assert data["name"] == "test"
    assert data["price"] == 10.5

def test_put_request():
    response = client.put("/items/1", json={"name": "updated", "price": 15.0})
    assert response.status_code == 200

def test_delete_request():
    response = client.delete("/items/1")
    assert response.status_code == 200
```

### Testing Query Parameters
```python
def test_query_params():
    response = client.get("/items/?skip=0&limit=10")
    assert response.status_code == 200

def test_optional_query_params():
    response = client.get("/items/?q=search_term")
    assert response.status_code == 200
```

## Advanced Testing Patterns

### Testing with Database Integration
```python
import pytest
from sqlalchemy import create_engine
from sqlalchemy.orm import sessionmaker
from fastapi.testclient import TestClient

# Use an in-memory database for tests
TEST_DATABASE_URL = "sqlite:///./test.db"
engine = create_engine(TEST_DATABASE_URL, connect_args={"check_same_thread": False})
TestingSessionLocal = sessionmaker(autocommit=False, autoflush=False, bind=engine)

@pytest.fixture
def db():
    """Create a database session for testing."""
    Base.metadata.create_all(bind=engine)
    session = TestingSessionLocal()
    try:
        yield session
    finally:
        session.close()
        Base.metadata.drop_all(bind=engine)

def test_create_item_with_db(db):
    # Override the dependency
    def override_get_db():
        try:
            yield db
        finally:
            pass

    app.dependency_overrides[get_db] = override_get_db

    response = client.post("/items/", json={
        "name": "Database Test",
        "price": 25.99
    })

    assert response.status_code == 200
    data = response.json()
    assert data["name"] == "Database Test"

    # Clean up
    app.dependency_overrides.clear()
```

### Testing Authentication and Authorization
```python
def test_protected_endpoint_without_auth():
    """Test that protected endpoints return 401 without auth."""
    response = client.get("/protected/")
    assert response.status_code == 401

def test_protected_endpoint_with_auth():
    """Test that protected endpoints work with proper auth."""
    headers = {"Authorization": "Bearer valid_token"}
    response = client.get("/protected/", headers=headers)
    assert response.status_code == 200
```

### Testing Error Cases
```python
def test_invalid_input():
    """Test that invalid input returns appropriate error."""
    response = client.post("/items/", json={
        "name": "",  # Invalid - empty name
        "price": -5  # Invalid - negative price
    })
    assert response.status_code == 422  # Validation error

def test_not_found():
    """Test that non-existent items return 404."""
    response = client.get("/items/999999/")
    assert response.status_code == 404
```

## Pytest Best Practices

### Parametrized Tests
```python
import pytest

@pytest.mark.parametrize("item_data,expected_status", [
    ({"name": "valid", "price": 10.0}, 200),
    ({"name": "", "price": 10.0}, 422),
    ({"name": "valid", "price": -1}, 422),
])
def test_item_creation_validation(item_data, expected_status):
    response = client.post("/items/", json=item_data)
    assert response.status_code == expected_status
```

### Test Fixtures for Complex Scenarios
```python
@pytest.fixture
def sample_item(db):
    """Create a sample item for testing."""
    item = Item(name="Sample Item", price=15.99)
    db.add(item)
    db.commit()
    db.refresh(item)
    return item

def test_get_existing_item(sample_item):
    response = client.get(f"/items/{sample_item.id}")
    assert response.status_code == 200
    assert response.json()["name"] == "Sample Item"
```

## Testing with Different Scopes

### Module-level Fixture (for tests that share setup)
```python
@pytest.fixture(scope="module")
def test_client():
    """Create a test client for the module."""
    with TestClient(app) as client:
        yield client
```

### Function-level Fixture (for isolated tests)
```python
@pytest.fixture
def clean_database():
    """Provide a clean database for each test."""
    Base.metadata.create_all(bind=engine)
    yield
    Base.metadata.drop_all(bind=engine)
```

## Testing External Dependencies

### Mocking External Services
```python
from unittest.mock import patch

def test_external_api_call():
    with patch('app.services.external_service.call_api') as mock_call:
        mock_call.return_value = {"result": "success"}

        response = client.post("/external-call/")
        assert response.status_code == 200
        mock_call.assert_called_once()
```

## Performance and Load Testing

### Basic Performance Testing
```python
import time

def test_endpoint_performance():
    start_time = time.time()
    response = client.get("/heavy-operation/")
    end_time = time.time()

    assert response.status_code == 200
    assert (end_time - start_time) < 1.0  # Should complete in under 1 second
```

## Running Tests

### Basic Test Execution
```bash
# Run all tests
pytest

# Run tests with verbose output
pytest -v

# Run tests in a specific file
pytest tests/test_main.py

# Run tests matching a pattern
pytest -k "test_item"
```

### Coverage Testing
```bash
# Run tests with coverage
pytest --cov=app

# Generate coverage report
pytest --cov=app --cov-report=html
```

### Configuration Options
Create a `pytest.ini` file:
```ini
[tool:pytest]
testpaths = tests
python_files = test_*.py *_test.py
python_classes = Test*
python_functions = test_*
addopts =
    -ra
    -v
    --strict-markers
markers =
    slow: marks tests as slow
    integration: marks tests as integration tests
```

## Common Testing Anti-patterns to Avoid

1. Don't share state between tests
2. Don't rely on test execution order
3. Don't make network calls in unit tests
4. Don't use real databases in unit tests
5. Don't test implementation details instead of behavior