#!/usr/bin/env python3
"""
FastAPI Project Generator

This script helps create a new FastAPI project with proper structure and dependencies.
"""

import os
import sys
from pathlib import Path

def create_fastapi_project(project_name: str, include_auth: bool = False, include_db: bool = False):
    """
    Create a new FastAPI project with the specified features.

    Args:
        project_name: Name of the project
        include_auth: Whether to include authentication setup
        include_db: Whether to include database setup
    """
    project_path = Path(project_name)
    project_path.mkdir(exist_ok=True)

    # Create main app directory
    app_dir = project_path / "app"
    app_dir.mkdir(exist_ok=True)

    # Create basic app structure
    init_file = app_dir / "__init__.py"
    init_file.write_text("")

    # Create main application file
    main_file = app_dir / "main.py"
    main_content = '''from fastapi import FastAPI

app = FastAPI(title="My FastAPI App")

@app.get("/")
def read_root():
    return {"message": "Welcome to FastAPI!"}

@app.get("/health")
def health_check():
    return {"status": "healthy"}
'''
    main_file.write_text(main_content)

    # Create models file
    models_file = app_dir / "models.py"
    models_content = '''from pydantic import BaseModel


class Item(BaseModel):
    name: str
    description: str = None
    price: float
    tax: float = None
'''
    models_file.write_text(models_content)

    # Create requirements.txt
    requirements_file = project_path / "requirements.txt"
    requirements_content = "fastapi\nuvicorn[standard]\n"

    # Add additional dependencies based on options
    if include_db:
        requirements_content += "sqlalchemy\n"

    if include_auth:
        requirements_content += "python-jose[cryptography]\nbcrypt\n"

    requirements_file.write_text(requirements_content)

    # Create README
    readme_file = project_path / "README.md"
    readme_content = f"""# {project_name}

A FastAPI application created with FastAPI Skill Generator.

## Setup

1. Install dependencies with UV:
   ```bash
   uv sync
   # Or if starting fresh:
   uv init
   uv add fastapi uvicorn
   ```

2. Run the development server:
   ```bash
   uv run uvicorn app.main:app --reload
   ```

The application will be available at http://localhost:8000
"""
    readme_file.write_text(readme_content)

    # Create .gitignore
    gitignore_file = project_path / ".gitignore"
    gitignore_content = """__pycache__/
*.pyc
*.pyo
*.pyd
.Python
env/
venv/
.venv/
pip-log.txt
pip-delete-this-directory.txt
.tox
.coverage
.coverage.*
.cache
nosetests.xml
coverage.xml
*.cover
*.log
.mypy_cache/
.pytest_cache/
.hypothesis/
.DS_Store
"""
    gitignore_file.write_text(gitignore_content)

    print(f"FastAPI project '{project_name}' created successfully!")
    print(f"Directory structure:")
    print(f"  {project_path}/")
    print(f"  ├── app/")
    print(f"  │   ├── __init__.py")
    print(f"  │   ├── main.py")
    print(f"  │   └── models.py")
    print(f"  ├── requirements.txt")
    print(f"  ├── README.md")
    print(f"  └── .gitignore")
    print(f"\nTo run the application:")
    print(f"  cd {project_name}")
    print(f"  uv run uvicorn app.main:app --reload")


def main():
    if len(sys.argv) < 2:
        print("Usage: python create_fastapi_app.py <project_name> [--with-auth] [--with-db]")
        sys.exit(1)

    project_name = sys.argv[1]
    include_auth = "--with-auth" in sys.argv
    include_db = "--with-db" in sys.argv

    create_fastapi_project(project_name, include_auth, include_db)


if __name__ == "__main__":
    main()
