#!/usr/bin/env python3
"""
Test Scaffolding Generator for FastAPI Projects

This script helps create a comprehensive test structure for FastAPI projects
with proper fixtures, conftest.py, and example tests.
"""

import os
from pathlib import Path


def create_test_scaffolding(project_path: str = "."):
    """
    Create a comprehensive test scaffolding for FastAPI projects.

    Args:
        project_path: Path to the FastAPI project (defaults to current directory)
    """
    project_path = Path(project_path)

    # Create tests directory structure
    tests_dir = project_path / "tests"
    tests_dir.mkdir(exist_ok=True)

    # Create subdirectories for organized testing
    (tests_dir / "unit").mkdir(exist_ok=True)
    (tests_dir / "integration").mkdir(exist_ok=True)
    (tests_dir / "api").mkdir(exist_ok=True)

    # Create main conftest.py
    conftest_content = '''import pytest
from fastapi.testclient import TestClient
from sqlalchemy import create_engine
from sqlalchemy.orm import sessionmaker
from app.main import app
from app.database import Base, get_db

# Use an in-memory SQLite database for tests
TEST_DATABASE_URL = "sqlite:///./test.db"

@pytest.fixture(scope="session")
def test_engine():
    """Create a test database engine."""
    engine = create_engine(TEST_DATABASE_URL, connect_args={"check_same_thread": False})
    Base.metadata.create_all(bind=engine)
    yield engine
    engine.dispose()

@pytest.fixture(scope="function")
def test_db_session(test_engine):
    """Create a test database session."""
    connection = test_engine.connect()
    transaction = connection.begin()
    Session = sessionmaker(bind=connection)
    session = Session()

    yield session

    session.close()
    transaction.rollback()
    connection.close()

@pytest.fixture(scope="function")
def override_get_db(test_db_session):
    """Override the get_db dependency."""
    def _get_db():
        try:
            yield test_db_session
        finally:
            pass

    app.dependency_overrides[get_db] = _get_db
    yield
    app.dependency_overrides.clear()

@pytest.fixture(scope="module")
def client():
    """Create a test client for the application."""
    with TestClient(app) as test_client:
        yield test_client

# Additional fixtures for authentication, etc.
@pytest.fixture
def authenticated_client(client):
    """Return an authenticated test client."""
    # Example: Add authentication headers or cookies
    client.headers.update({"Authorization": "Bearer test_token"})
    return client
'''

    with open(tests_dir / "conftest.py", "w") as f:
        f.write(conftest_content)

    # Create basic test for main app
    main_test_content = '''from fastapi.testclient import TestClient
from app.main import app

client = TestClient(app)

def test_read_main():
    """Test the main endpoint."""
    response = client.get("/")
    assert response.status_code == 200
    assert "message" in response.json()

def test_health_check():
    """Test the health check endpoint."""
    response = client.get("/health")
    assert response.status_code == 200
    assert response.json() == {"status": "healthy"}
'''

    with open(tests_dir / "test_main.py", "w") as f:
        f.write(main_test_content)

    # Create example API tests
    api_test_content = '''import pytest
from fastapi.testclient import TestClient

def test_create_item(client):
    """Test creating an item."""
    response = client.post("/items/", json={
        "name": "Test Item",
        "price": 10.50,
        "description": "A test item"
    })
    assert response.status_code == 200
    data = response.json()
    assert data["name"] == "Test Item"
    assert data["price"] == 10.50

def test_get_item(client):
    """Test retrieving an item."""
    # First create an item
    create_response = client.post("/items/", json={
        "name": "Get Test",
        "price": 15.75
    })
    assert create_response.status_code == 200
    item_id = create_response.json()["id"]

    # Then retrieve it
    response = client.get(f"/items/{item_id}")
    assert response.status_code == 200
    assert response.json()["name"] == "Get Test"

@pytest.mark.parametrize("invalid_data,expected_status", [
    ({"name": "", "price": 10.0}, 422),  # Empty name
    ({"name": "Valid", "price": -1}, 422),  # Negative price
])
def test_invalid_item_creation(client, invalid_data, expected_status):
    """Test that invalid item data returns appropriate error."""
    response = client.post("/items/", json=invalid_data)
    assert response.status_code == expected_status
'''

    with open(tests_dir / "api" / "test_items.py", "w") as f:
        f.write(api_test_content)

    # Create requirements-test.txt
    requirements_content = '''pytest
pytest-cov
pytest-asyncio
httpx
'''

    with open(project_path / "requirements-test.txt", "w") as f:
        f.write(requirements_content)

    # Create pytest.ini configuration
    pytest_config = '''[tool:pytest]
testpaths = tests
python_files = test_*.py *_test.py
python_classes = Test*
python_functions = test_*
addopts =
    -ra
    -v
    --strict-markers
    --cov=app
    --cov-report=term-missing
markers =
    slow: marks tests as slow
    integration: marks tests as integration tests
'''

    with open(project_path / "pytest.ini", "w") as f:
        f.write(pytest_config)

    print(f"Test scaffolding created in {tests_dir}/")
    print("Directory structure:")
    print("  tests/")
    print("  ├── conftest.py")
    print("  ├── test_main.py")
    print("  ├── unit/")
    print("  ├── integration/")
    print("  └── api/")
    print("  └── api/test_items.py")
    print("Additional files created:")
    print("  - requirements-test.txt")
    print("  - pytest.ini")
    print("\nTo run tests:")
    print("  pip install -r requirements-test.txt")
    print("  pytest")
    print("  pytest --cov=app  # for coverage")


def main():
    import sys

    project_path = sys.argv[1] if len(sys.argv) > 1 else "."
    create_test_scaffolding(project_path)


if __name__ == "__main__":
    main()