# Finance Tracking Examples

## Example 1: Processing Bank Transactions

**Input:** CSV from bank export

```csv
Date,Description,Amount,Balance
2026-01-05,NOTION.SO,"-10.00",4990.00
2026-01-04,GOOGLE ADS,"-150.00",5000.00
2026-01-03,CLIENT PAYMENT - INV001,"1500.00",5150.00
2026-01-02,AMAZON.COM,"-45.99",3650.00
```

**Output:** Transaction files in `/Accounting/`

```markdown
# /Accounting/transactions/2026-01-05_notion.md
---
type: transaction
date: 2026-01-05
amount: -10.00
category: Software/SaaS
vendor: Notion
description: NOTION.SO
is_recurring: true
frequency: monthly
requires_review: false
---

Monthly Notion subscription payment.
```

```markdown
# /Accounting/transactions/2026-01-04_google_ads.md
---
type: transaction
date: 2026-01-04
amount: -150.00
category: Marketing
vendor: Google Ads
description: GOOGLE ADS
is_recurring: false
requires_review: true
---

Google Ads advertising spend.

**Review Note**: Single charge > $100, verify campaign performance.
```

## Example 2: Budget Status Report

```markdown
# Budget Status - January 2026

**Last Updated**: 2026-01-07T10:00:00Z

## Summary
- **Total Budget**: $2,500.00
- **Total Spent**: $1,205.99
- **Remaining**: $1,294.01
- **Days Left**: 24
- **Daily Budget**: $53.92

## By Category

| Category | Budget | Spent | Remaining | % Used | Status |
|----------|--------|-------|-----------|--------|--------|
| Software/SaaS | $500 | $210.00 | $290.00 | 42% | 🟢 |
| Marketing | $1,000 | $650.00 | $350.00 | 65% | 🟢 |
| Office Supplies | $200 | $145.99 | $54.01 | 73% | 🟡 |
| Utilities | $300 | $0.00 | $300.00 | 0% | 🟢 |
| Travel | $500 | $200.00 | $300.00 | 40% | 🟢 |

## Alerts
- ⚠️ Office Supplies at 73% - slow down non-essential purchases

## Projections
At current rate, by end of month:
- Software/SaaS: $630 (26% over budget)
- Marketing: On track
- Office Supplies: $438 (119% over budget) ⚠️

## Recent Transactions

| Date | Vendor | Amount | Category |
|------|--------|--------|----------|
| Jan 5 | Notion | $10.00 | Software/SaaS |
| Jan 4 | Google Ads | $150.00 | Marketing |
| Jan 2 | Amazon | $45.99 | Office Supplies |
```

## Example 3: Subscription Audit

```markdown
# Subscription Audit - January 2026

**Generated**: 2026-01-07

## Active Subscriptions

| Service | Monthly Cost | Last Charged | Last Used | Status |
|---------|--------------|--------------|-----------|--------|
| Notion | $10.00 | Jan 5 | Jan 7 | ✅ Active |
| GitHub | $4.00 | Jan 1 | Jan 7 | ✅ Active |
| Slack | $8.75 | Dec 28 | Jan 6 | ✅ Active |
| Adobe CC | $54.99 | Dec 15 | Nov 20 | ⚠️ Review |
| Zoom | $14.99 | Jan 2 | Jan 5 | ✅ Active |

## Total Monthly Subscription Cost
**$92.73**

## Recommendations

### Review Required
1. **Adobe Creative Cloud** - $54.99/month
   - Last used: 47 days ago
   - Recommendation: Consider cancellation or downgrade
   - Potential savings: $54.99/month ($659.88/year)

### Price Changes Detected
- None this month

### New Subscriptions
- None this month

## Action Items
- [ ] Review Adobe CC usage - cancel if not needed
- [ ] Check for annual billing options (typically 15-20% savings)
```

## Example 4: Transaction Categorization Script

```python
# categorize_transactions.py
from pathlib import Path
from datetime import datetime
import csv
import yaml

CATEGORIES = {
    'Software/SaaS': {
        'keywords': ['subscription', 'monthly', '.com', '.io', 'pro', 'premium'],
        'vendors': ['notion', 'slack', 'github', 'adobe', 'zoom', 'dropbox'],
        'budget': 500
    },
    'Marketing': {
        'keywords': ['ads', 'advertising', 'campaign', 'boost'],
        'vendors': ['facebook', 'google ads', 'linkedin', 'twitter'],
        'budget': 1000
    },
    'Office Supplies': {
        'keywords': ['office', 'supplies', 'paper', 'equipment'],
        'vendors': ['amazon', 'staples', 'office depot'],
        'budget': 200
    },
    'Utilities': {
        'keywords': ['electric', 'internet', 'phone', 'utility'],
        'vendors': [],
        'budget': 300
    },
    'Travel': {
        'keywords': ['airline', 'hotel', 'uber', 'lyft', 'airbnb'],
        'vendors': [],
        'budget': 500
    }
}

def categorize(description: str, amount: float) -> dict:
    """Categorize a transaction based on description."""
    desc_lower = description.lower()

    for category, rules in CATEGORIES.items():
        # Check vendors
        for vendor in rules['vendors']:
            if vendor in desc_lower:
                return {
                    'category': category,
                    'vendor': vendor.title(),
                    'confidence': 'high'
                }

        # Check keywords
        for keyword in rules['keywords']:
            if keyword in desc_lower:
                return {
                    'category': category,
                    'vendor': extract_vendor(description),
                    'confidence': 'medium'
                }

    # Unknown - flag for review
    return {
        'category': 'Uncategorized',
        'vendor': extract_vendor(description),
        'confidence': 'low',
        'requires_review': True
    }

def extract_vendor(description: str) -> str:
    """Extract vendor name from description."""
    # Remove common prefixes/suffixes
    cleaned = description.upper()
    for prefix in ['PAYMENT TO ', 'PURCHASE ', 'DEBIT ']:
        cleaned = cleaned.replace(prefix, '')
    return cleaned.split()[0] if cleaned else 'Unknown'

def process_csv(csv_path: str, output_dir: str):
    """Process bank CSV and create transaction files."""
    output = Path(output_dir)
    output.mkdir(parents=True, exist_ok=True)

    with open(csv_path, 'r') as f:
        reader = csv.DictReader(f)
        for row in reader:
            date = row['Date']
            description = row['Description']
            amount = float(row['Amount'])

            cat_result = categorize(description, amount)

            # Create transaction file
            filename = f"{date}_{cat_result['vendor'].lower().replace(' ', '_')}.md"
            filepath = output / filename

            content = f"""---
type: transaction
date: {date}
amount: {amount}
category: {cat_result['category']}
vendor: {cat_result['vendor']}
description: {description}
is_recurring: false
requires_review: {cat_result.get('requires_review', False)}
---

{description}
"""
            filepath.write_text(content)
            print(f"Created: {filename}")
```

## Example 5: CEO Financial Summary

```markdown
## Financial Health - Week of Jan 1-7, 2026

### Revenue
- **This Week**: $1,500.00
- **MTD**: $1,500.00
- **Target**: $10,000.00 (15% achieved)
- **Projection**: $6,428 (at current pace)

### Expenses
- **This Week**: $205.99
- **MTD**: $205.99
- **Budget**: $2,500.00 (8% used)

### Cash Flow
- **Net This Week**: +$1,294.01
- **Runway**: Healthy

### Alerts
- 📈 Revenue on track for first week
- ⚠️ Office supplies trending over budget
- ✅ All subscriptions active and utilized

### Recommendations
1. Review Adobe CC subscription (unused 47 days)
2. Consider reducing Google Ads if CAC too high
```
