---
name: finishing-a-development-branch
description: Use when implementation is complete, all tests pass, and you need to decide how to integrate the work - guides completion of development work by presenting structured options for merge, PR, or cleanup
---

# Finishing a Development Branch

## Overview

Guide completion of development work by presenting clear options and handling chosen workflow.

**Core principle:** Verify tests → Present options → Execute choice → Clean up.

**Announce at start:** "I'm using the finishing-a-development-branch skill to complete this work."

## The Process

### Step 1: Verify Tests

**Before presenting options, verify tests pass:**

```bash
# .NET / C# (ECTSystem)
dotnet test AF.ECT.Tests --no-build --logger "console;verbosity=minimal"

# Node.js
npm test

# Python
pytest

# Go
go test ./...
```

**For ECTSystem specifically:**
```bash
dotnet test AF.ECT.Tests --no-build --logger "console;verbosity=minimal"
```

**If tests fail:**
```
Tests failing (5 failures). Must fix before completing:

FAIL: GetWorkflow_WithValidId_ReturnsWorkflow
FAIL: CreateWorkflow_WithValidData_PersistsToDatabase
...

Cannot proceed with merge/PR until tests pass.
```

Stop. Don't proceed to Step 2.

**If tests pass:** Continue to Step 2.

**Build verification (ECTSystem):**
```bash
dotnet build ECTSystem.sln --no-restore
# Expected: 0 errors, 0 warnings
```

### Step 2: Determine Base Branch

```bash
# Try common base branches
git merge-base HEAD main 2>/dev/null || git merge-base HEAD master 2>/dev/null
```

Or ask: "This branch split from main - is that correct?"

### Step 3: Present Options

Present exactly these 4 options:

```
Implementation complete. What would you like to do?

1. Merge back to <base-branch> locally
2. Push and create a Pull Request
3. Keep the branch as-is (I'll handle it later)
4. Discard this work

Which option?
```

**Don't add explanation** - keep options concise.

### Step 4: Execute Choice

#### Option 1: Merge Locally

```bash
# Switch to base branch
git checkout <base-branch>

# Pull latest
git pull

# Merge feature branch
git merge <feature-branch>

# Verify tests on merged result
# .NET / C# (ECTSystem)
dotnet test AF.ECT.Tests --no-build --logger "console;verbosity=minimal"
dotnet build ECTSystem.sln --no-restore

# Node.js
npm test

# If tests pass
git branch -d <feature-branch>
```

Then: Cleanup worktree (Step 5)

#### Option 2: Push and Create PR

```bash
# Push branch
git push -u origin <feature-branch>

# Create PR with verification checklist
# .NET / C# (ECTSystem) template
gh pr create --title "feat: [feature description]" --body "$(cat <<'EOF'
## Summary
- Added/fixed/changed [X]
- Implementation in [AF.ECT.Server/AF.ECT.Shared/AF.ECT.WebClient]
- Tests added: [list test names]

## Verification
- [x] `dotnet build ECTSystem.sln` succeeds (0 errors, 0 warnings)
- [x] `dotnet test AF.ECT.Tests` passes (X tests, 0 failures)
- [x] gRPC contracts updated (AF.ECT.Shared/Protos)
- [x] EF Core migrations applied (if DB changes)
- [x] Audit logging included (if applicable)

## Test Plan
- [ ] Manual test: [steps to verify]
- [ ] Run: `dotnet test AF.ECT.Tests --filter "[TestName]"` to verify
EOF
)"
```

Then: Cleanup worktree (Step 5)

#### Option 3: Keep As-Is

Report: "Keeping branch <name>. Worktree preserved at <path>."

**Don't cleanup worktree.**

#### Option 4: Discard

**Confirm first:**
```
This will permanently delete:
- Branch <name>
- All commits: <commit-list>
- Worktree at <path>

Type 'discard' to confirm.
```

Wait for exact confirmation.

If confirmed:
```bash
git checkout <base-branch>
git branch -D <feature-branch>
```

Then: Cleanup worktree (Step 5)

### Step 5: Cleanup Worktree

**For Options 1, 2, 4:**

Check if in worktree:
```bash
git worktree list | grep $(git branch --show-current)
```

If yes:
```bash
git worktree remove <worktree-path>
```

**For Option 3:** Keep worktree.

## Quick Reference

| Option | Merge | Push | Keep Worktree | Cleanup Branch |
|--------|-------|------|---------------|----------------|
| 1. Merge locally | ✓ | - | - | ✓ |
| 2. Create PR | - | ✓ | ✓ | - |
| 3. Keep as-is | - | - | ✓ | - |
| 4. Discard | - | - | - | ✓ (force) |

## Common Mistakes

**Skipping test verification**
- **Problem:** Merge broken code, create failing PR
- **Fix:** Always verify tests before offering options

**Open-ended questions**
- **Problem:** "What should I do next?" → ambiguous
- **Fix:** Present exactly 4 structured options

**Automatic worktree cleanup**
- **Problem:** Remove worktree when might need it (Option 2, 3)
- **Fix:** Only cleanup for Options 1 and 4

**No confirmation for discard**
- **Problem:** Accidentally delete work
- **Fix:** Require typed "discard" confirmation

## Red Flags

**Never:**
- Proceed with failing tests
- Merge without verifying tests on result
- Delete work without confirmation
- Force-push without explicit request

**Always:**
- Verify tests before offering options
- Present exactly 4 options
- Get typed confirmation for Option 4
- Clean up worktree for Options 1 & 4 only

## Integration

**Called by:**
- **subagent-driven-development** (Step 7) - After all tasks complete
- **executing-plans** (Step 5) - After all batches complete

**Pairs with:**
- **using-git-worktrees** - Cleans up worktree created by that skill
