# i18n Patterns

Teetsh internationalization conventions using react-i18next.

## Translation Hook Usage

Always specify namespace array in `useTranslation`:

```tsx
// Correct
const { t } = useTranslation(["common"]);
const { t } = useTranslation(["dashboard", "common"]);

// Wrong - no namespace
const { t } = useTranslation();
```

## Translation Key Format

Use full dot-notation paths with namespace prefix:

```tsx
// Correct
t("common:buttons.save");
t("dashboard:header.title");
t("cahierJournal:activities.empty");

// Wrong - missing namespace
t("buttons.save");

// Wrong - incorrect format
t("common.buttons.save");
```

## Locale Files Structure

Translations are in `/public/locales/{locale}/{namespace}.json`:

```
public/locales/
├── fr/
│   ├── common.json
│   ├── dashboard.json
│   └── cahierJournal.json
├── fr-BE/
├── en/
└── en-AU/
```

Supported locales:

- `fr` - French (France)
- `fr-BE` - French (Belgium)
- `en` - English
- `en-AU` - English (Australia)

## Date Formatting

Use the `format` utility function, NOT date-fns directly:

```tsx
import { format } from "@teetsh/app/src/utils/time/format";

// Correct
format(date, "fr-FR");
format(date, user.locale);

// Wrong - using date-fns directly
import { format } from "date-fns";
format(date, "PPP");
```

Common format patterns:

- `'fr-FR'` - French format
- `'en-US'` - US English format
- `'en-AU'` - Australian format

## Interpolation

Pass variables to translations:

```tsx
// In JSON
{
  "welcome": "Welcome, {{name}}!",
  "items": "You have {{count}} items"
}

// In code
t('common:welcome', { name: user.name })
t('common:items', { count: items.length })
```

## Pluralization

Use count for pluralization:

```tsx
// In JSON
{
  "student": "{{count}} student",
  "student_plural": "{{count}} students"
}

// In code
t('common:student', { count: studentCount })
```

## Flattened Keys

Translation keys use flattened dot-notation (not nested objects):

```tsx
// In JSON - flattened format
{
  "form.email": "Email",
  "form.password": "Password",
  "form.errors.required": "This field is required",
  "form.errors.invalid": "Invalid format"
}

// In code
t('common:form.email')
t('common:form.errors.required')
```

## Trans Component

For complex translations with JSX:

```tsx
import { Trans } from 'react-i18next';

// In JSON
{
  "terms": "By signing up, you agree to our <link>Terms of Service</link>"
}

// In code
<Trans
  i18nKey="common:terms"
  components={{
    link: <a href="/terms" tw="text-brand-intense underline" />,
  }}
/>
```

## Common Mistakes

### Hardcoded User-Facing Strings

```tsx
// Wrong
<Button>Delete</Button>
<p>No items found</p>
<span>Loading...</span>

// Correct
<Button>{t('common:buttons.delete')}</Button>
<p>{t('common:empty.noItems')}</p>
<span>{t('common:loading')}</span>
```

### Missing Namespace

```tsx
// Wrong
const { t } = useTranslation();
t("buttons.save");

// Correct
const { t } = useTranslation(["common"]);
t("common:buttons.save");
```

### Wrong Key Format

```tsx
// Wrong - dot instead of colon for namespace
t("common.buttons.save");

// Correct
t("common:buttons.save");
```

### Using date-fns Directly

```tsx
// Wrong
import { format } from "date-fns";
import { fr } from "date-fns/locale";
format(date, "PPP", { locale: fr });

// Correct
import { format } from "@teetsh/app/src/utils/time/format";
format(date, "fr-FR");
```

### Forgetting Locale Variants

```tsx
// Check: Does the translation exist in all locales?
// - fr
// - fr-BE
// - en
// - en-AU
```

### Concatenating Translated Strings

```tsx
// Wrong - word order varies by language
t("common:hello") + " " + name + "!";

// Correct - use interpolation
t("common:greeting", { name });
```
