# Gemini Deep Research API Reference

## API Endpoints

### Base URL
```
https://generativelanguage.googleapis.com/v1beta/interactions
```

### Authentication
Include API key via header:
```
x-goog-api-key: YOUR_GEMINI_API_KEY
```

Or set environment variable:
```bash
export GEMINI_API_KEY="your-api-key"
```

---

## Create Interaction

**POST** `/interactions`

Create and start a new research task.

### Request Body

```json
{
  "input": "string",
  "agent": "deep-research-pro-preview-12-2025",
  "background": true,
  "stream": false,
  "tools": [],
  "agent_config": {
    "type": "deep-research",
    "thinking_summaries": "auto"
  },
  "previous_interaction_id": "string"
}
```

### Parameters

| Field | Type | Required | Description |
|-------|------|----------|-------------|
| `input` | string | Yes | The research query or prompt |
| `agent` | string | Yes | Agent identifier. Must be `deep-research-pro-preview-12-2025` |
| `background` | boolean | Yes | Must be `true` for async execution |
| `stream` | boolean | No | Enable real-time event streaming. Default: `false` |
| `tools` | array | No | Tool configurations (e.g., `file_search`) |
| `agent_config` | object | No | Agent-specific configuration |
| `previous_interaction_id` | string | No | Continue conversation from prior interaction |

### Agent Config Options

```json
{
  "type": "deep-research",
  "thinking_summaries": "auto" | "none"
}
```

- `thinking_summaries: "auto"` - Receive intermediate reasoning steps during streaming
- `thinking_summaries: "none"` - Only receive final content

### Response

```json
{
  "id": "interaction-id-string",
  "status": "in_progress",
  "outputs": [],
  "created_time": "2025-01-01T00:00:00Z"
}
```

---

## Get Interaction

**GET** `/interactions/{INTERACTION_ID}`

Poll for research status and results.

### Parameters

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `INTERACTION_ID` | string | Yes | ID from create response |
| `stream` | boolean | No | Stream remaining events. Default: `false` |
| `last_event_id` | string | No | Resume streaming from specific event |

### Response

```json
{
  "id": "interaction-id-string",
  "status": "completed",
  "outputs": [
    {
      "text": "# Research Report\n\nComprehensive findings...",
      "citations": [...]
    }
  ],
  "error": null
}
```

### Status Values

| Status | Description |
|--------|-------------|
| `in_progress` | Research is actively running |
| `completed` | Research finished successfully |
| `failed` | Research encountered an error |

---

## Streaming Events

When `stream=true`, the API returns Server-Sent Events (SSE).

### Event Types

#### interaction.start
Emitted when research begins. Contains the interaction ID for tracking.

```json
{
  "event_type": "interaction.start",
  "event_id": "event-123",
  "interaction": {
    "id": "interaction-456"
  }
}
```

#### content.delta
Emitted for each content chunk. May be text or thinking summary.

```json
{
  "event_type": "content.delta",
  "event_id": "event-124",
  "delta": {
    "text": "The research findings indicate...",
    "type": "text"
  }
}
```

Delta types:
- `"text"` - Final report content
- `"thought_summary"` - Intermediate reasoning (when `thinking_summaries: "auto"`)

#### interaction.complete
Emitted when research finishes successfully.

```json
{
  "event_type": "interaction.complete",
  "event_id": "event-999",
  "interaction": {
    "id": "interaction-456",
    "status": "completed"
  }
}
```

#### error
Emitted when an error occurs.

```json
{
  "event_type": "error",
  "error": {
    "code": "INTERNAL_ERROR",
    "message": "Research failed due to..."
  }
}
```

---

## File Search Tool

Integrate uploaded documents into research.

### Configuration

```json
{
  "tools": [
    {
      "type": "file_search",
      "file_search_store_names": [
        "fileSearchStores/my-store-name"
      ]
    }
  ]
}
```

### Supported File Types

- PDF documents
- CSV spreadsheets
- Text documents (TXT, MD)
- Word documents (DOCX)

### Creating File Stores

Use the Files API to create and populate file stores before referencing them in Deep Research.

```python
from google import genai

client = genai.Client()

# Upload file
file = client.files.upload(path="research_data.pdf")

# Create file store
store = client.file_search_stores.create(
    name="my-research-store",
    files=[file.id]
)

# Use in research
interaction = client.interactions.create(
    input="Analyze the uploaded document",
    agent='deep-research-pro-preview-12-2025',
    background=True,
    tools=[{
        "type": "file_search",
        "file_search_store_names": [store.name]
    }]
)
```

---

## Constraints & Limits

| Constraint | Value |
|------------|-------|
| Maximum runtime | 60 minutes |
| Context limit | ~1M tokens |
| Poll interval (recommended) | 10 seconds |
| Background mode | Required (`background=True`) |

### Unsupported Features

- Audio input
- Custom function calling tools
- Human-in-the-loop planning approval
- Real-time video input

---

## Pricing

| Model | Input Cost | Output Cost |
|-------|------------|-------------|
| Deep Research Pro | $2.00 / 1M tokens | Included |

---

## Error Codes

| Code | Description | Resolution |
|------|-------------|------------|
| `INVALID_AGENT` | Agent ID not recognized | Use exact agent ID |
| `MISSING_BACKGROUND` | background=True not set | Add `background=True` |
| `RATE_LIMITED` | Too many requests | Implement exponential backoff |
| `CONTEXT_OVERFLOW` | Input too large | Reduce query/file size |
| `INTERNAL_ERROR` | Server-side failure | Retry with backoff |

---

## cURL Examples

### Create Research Task

```bash
curl -X POST \
  "https://generativelanguage.googleapis.com/v1beta/interactions" \
  -H "Content-Type: application/json" \
  -H "x-goog-api-key: $GEMINI_API_KEY" \
  -d '{
    "input": "Research the history of neural networks",
    "agent": "deep-research-pro-preview-12-2025",
    "background": true
  }'
```

### Poll for Results

```bash
curl -X GET \
  "https://generativelanguage.googleapis.com/v1beta/interactions/INTERACTION_ID" \
  -H "x-goog-api-key: $GEMINI_API_KEY"
```

### Stream Results

```bash
curl -X GET \
  "https://generativelanguage.googleapis.com/v1beta/interactions/INTERACTION_ID?stream=true" \
  -H "x-goog-api-key: $GEMINI_API_KEY" \
  -N
```
