#!/usr/bin/env python3
"""
Basic Gemini Deep Research Agent

Simple implementation demonstrating core polling-based research.
Install: pip install google-genai
Set: export GEMINI_API_KEY="your-api-key"
"""

import os
import time
from google import genai


def create_research_agent(api_key: str | None = None):
    """Create a Gemini client for deep research."""
    return genai.Client(api_key=api_key or os.getenv("GEMINI_API_KEY"))


def research(client: genai.Client, query: str, poll_interval: int = 10) -> str:
    """
    Execute a deep research task and return the final report.

    Args:
        client: Initialized Gemini client
        query: Research question or topic
        poll_interval: Seconds between status checks (default 10)

    Returns:
        Generated research report as string
    """
    print(f"Starting research: {query[:50]}...")

    # Create research interaction
    interaction = client.interactions.create(
        input=query,
        agent='deep-research-pro-preview-12-2025',
        background=True
    )

    print(f"Interaction ID: {interaction.id}")

    # Poll until completion
    while True:
        interaction = client.interactions.get(interaction.id)
        status = interaction.status

        if status == "completed":
            print("Research completed!")
            return interaction.outputs[-1].text

        elif status == "failed":
            error_msg = getattr(interaction, 'error', 'Unknown error')
            raise Exception(f"Research failed: {error_msg}")

        else:
            print(f"Status: {status} - waiting {poll_interval}s...")
            time.sleep(poll_interval)


def main():
    """Example usage of basic research agent."""
    client = create_research_agent()

    # Example research queries
    queries = [
        "What are the latest breakthroughs in quantum computing hardware in 2025?",
        "Compare the environmental impact of electric vehicles vs hydrogen fuel cells",
        "Analyze the current state of AI regulation in the European Union",
    ]

    # Run first query as demonstration
    query = queries[0]
    print(f"\n{'='*60}")
    print(f"Research Query: {query}")
    print('='*60 + "\n")

    report = research(client, query)

    print("\n" + "="*60)
    print("RESEARCH REPORT")
    print("="*60)
    print(report)


if __name__ == "__main__":
    main()
