#!/usr/bin/env python3
"""
Streaming Gemini Deep Research Agent

Implementation with real-time streaming and reconnection support.
Shows intermediate thinking and progress updates.

Install: pip install google-genai
Set: export GEMINI_API_KEY="your-api-key"
"""

import os
import time
import sys
from dataclasses import dataclass, field
from typing import Generator
from google import genai


@dataclass
class StreamState:
    """Track streaming state for reconnection."""
    last_event_id: str | None = None
    interaction_id: str | None = None
    is_complete: bool = False
    collected_text: list = field(default_factory=list)
    thinking_chunks: list = field(default_factory=list)


def create_client(api_key: str | None = None) -> genai.Client:
    """Create authenticated Gemini client."""
    return genai.Client(api_key=api_key or os.getenv("GEMINI_API_KEY"))


def process_stream_events(event_stream, state: StreamState, show_thinking: bool = True):
    """
    Process events from a stream, updating state and yielding content.

    Args:
        event_stream: Stream of events from Gemini API
        state: StreamState object to track progress
        show_thinking: Whether to print thinking summaries
    """
    for event in event_stream:
        # Track interaction ID from start event
        if event.event_type == "interaction.start":
            state.interaction_id = event.interaction.id
            print(f"[Started: {state.interaction_id[:20]}...]", file=sys.stderr)

        # Update last event ID for reconnection
        if hasattr(event, 'event_id') and event.event_id:
            state.last_event_id = event.event_id

        # Handle content deltas
        if event.event_type == "content.delta":
            if hasattr(event, 'delta') and hasattr(event.delta, 'text'):
                text = event.delta.text
                delta_type = getattr(event.delta, 'type', 'text')

                if delta_type == "thought_summary" and show_thinking:
                    state.thinking_chunks.append(text)
                    print(f"\n[Thinking: {text[:100]}...]", file=sys.stderr)
                else:
                    state.collected_text.append(text)
                    yield text

        # Handle completion
        if event.event_type == "interaction.complete":
            state.is_complete = True
            print("\n[Research complete]", file=sys.stderr)
            break

        # Handle errors
        if event.event_type == "error":
            state.is_complete = True
            error_info = getattr(event, 'error', event)
            raise Exception(f"Stream error: {error_info}")


def research_streaming(
    client: genai.Client,
    query: str,
    show_thinking: bool = True,
    max_reconnects: int = 5,
    reconnect_delay: int = 2
) -> Generator[str, None, None]:
    """
    Stream research results with automatic reconnection.

    Args:
        client: Gemini client
        query: Research query
        show_thinking: Show intermediate thinking summaries
        max_reconnects: Maximum reconnection attempts
        reconnect_delay: Seconds between reconnection attempts

    Yields:
        Text chunks as they arrive
    """
    state = StreamState()
    reconnect_count = 0

    # Initial stream creation
    print(f"[Starting research: {query[:40]}...]", file=sys.stderr)

    try:
        stream = client.interactions.create(
            input=query,
            agent='deep-research-pro-preview-12-2025',
            background=True,
            stream=True,
            agent_config={
                "type": "deep-research",
                "thinking_summaries": "auto" if show_thinking else "none"
            }
        )

        yield from process_stream_events(stream, state, show_thinking)

    except Exception as e:
        print(f"[Initial stream error: {e}]", file=sys.stderr)

    # Reconnection loop
    while not state.is_complete and state.interaction_id and reconnect_count < max_reconnects:
        reconnect_count += 1
        print(f"[Reconnecting ({reconnect_count}/{max_reconnects})...]", file=sys.stderr)
        time.sleep(reconnect_delay)

        try:
            resume_params = {"id": state.interaction_id, "stream": True}
            if state.last_event_id:
                resume_params["last_event_id"] = state.last_event_id

            resume_stream = client.interactions.get(**resume_params)
            yield from process_stream_events(resume_stream, state, show_thinking)

        except Exception as e:
            print(f"[Reconnection error: {e}]", file=sys.stderr)
            continue

    if not state.is_complete:
        raise Exception("Research did not complete after maximum reconnection attempts")


def research_to_string(client: genai.Client, query: str, show_thinking: bool = True) -> str:
    """Convenience function to get full research as string."""
    chunks = list(research_streaming(client, query, show_thinking))
    return "".join(chunks)


def main():
    """Demonstration of streaming research agent."""
    client = create_client()

    query = "What are the most significant AI safety research developments in 2025?"

    print("\n" + "="*70)
    print("GEMINI DEEP RESEARCH - STREAMING MODE")
    print("="*70)
    print(f"\nQuery: {query}\n")
    print("-"*70)
    print("LIVE RESEARCH OUTPUT:")
    print("-"*70 + "\n")

    # Stream and print in real-time
    for chunk in research_streaming(client, query, show_thinking=True):
        print(chunk, end="", flush=True)

    print("\n\n" + "="*70)
    print("Research complete!")
    print("="*70)


if __name__ == "__main__":
    main()
