#!/usr/bin/env dotnet
#:package Google.Apis.Gmail.v1@1.69.0.3742
#:package Google.Apis.Auth@1.69.0
#:package Spectre.Console@0.49.1

using System.Diagnostics;
using Google.Apis.Auth.OAuth2;
using Google.Apis.Auth.OAuth2.Flows;
using Google.Apis.Gmail.v1;
using Google.Apis.Util.Store;
using Spectre.Console;

// OAuth 2.0 credentials - REPLACE WITH YOUR OWN (must match gmail.cs)
//
// To get your credentials:
// 1. Go to https://console.cloud.google.com/
// 2. Create a new project (or select an existing one)
// 3. Enable the Gmail API for your project
// 4. Go to "Credentials" > "Create Credentials" > "OAuth client ID"
// 5. Choose "Desktop app" as the application type
// 6. Copy the Client ID and Client Secret and paste them below AND in gmail.cs
const string ClientId = "YOUR_CLIENT_ID.apps.googleusercontent.com";
const string ClientSecret = "YOUR_CLIENT_SECRET";

// Get email from command line args
if (args.Length == 0)
{
    AnsiConsole.MarkupLine("[red]Usage: get-gmail-token.cs <email-address>[/]");
    AnsiConsole.MarkupLine("[yellow]Example: ./get-gmail-token.cs lucas@lucasmeijer.com[/]");
    return 1;
}

var emailAddress = args[0];

var clientId = ClientId;
var clientSecret = ClientSecret;

AnsiConsole.MarkupLine($"[green]Starting Gmail OAuth flow for {Markup.Escape(emailAddress)}...[/]");
AnsiConsole.WriteLine();

try
{
    var clientSecrets = new ClientSecrets
    {
        ClientId = clientId,
        ClientSecret = clientSecret
    };

    // Use a temporary directory for token storage during this flow
    var tempTokenPath = Path.Combine(Path.GetTempPath(), $"gmail-token-{Guid.NewGuid()}");

    AnsiConsole.MarkupLine("[yellow]A browser window will open for you to authorize the application.[/]");
    AnsiConsole.MarkupLine($"[yellow]Please sign in with: {Markup.Escape(emailAddress)}[/]");
    AnsiConsole.WriteLine();

    var credential = await GoogleWebAuthorizationBroker.AuthorizeAsync(
        clientSecrets,
        new[] { GmailService.Scope.GmailModify, GmailService.Scope.GmailCompose },
        "user",
        CancellationToken.None,
        new FileDataStore(tempTokenPath, true));

    // Get the token
    var token = await credential.GetAccessTokenForRequestAsync();
    var refreshToken = credential.Token.RefreshToken;

    AnsiConsole.WriteLine();
    AnsiConsole.MarkupLine("[green]✓ Authorization successful![/]");
    AnsiConsole.WriteLine();

    // Display the credentials
    var table = new Table();
    table.AddColumn("Field");
    table.AddColumn("Value");
    table.Border(TableBorder.Rounded);

    table.AddRow("[cyan]client_id[/]", "[dim](embedded in code)[/]");
    table.AddRow("[cyan]client_secret[/]", "[dim](embedded in code)[/]");
    table.AddRow("[cyan]refresh_token[/]", refreshToken ?? "[red]ERROR: No refresh token received[/]");

    AnsiConsole.Write(table);
    AnsiConsole.WriteLine();

    if (string.IsNullOrEmpty(refreshToken))
    {
        AnsiConsole.MarkupLine("[red]ERROR: No refresh token was received.[/]");
        AnsiConsole.MarkupLine("[yellow]This can happen if you've already authorized this app before.[/]");
        AnsiConsole.MarkupLine("[yellow]Try revoking access at: https://myaccount.google.com/permissions[/]");
        return 1;
    }

    // Save to 1Password automatically
    AnsiConsole.WriteLine();
    await SaveTo1Password(emailAddress, refreshToken);

    // Cleanup temp directory
    try
    {
        Directory.Delete(tempTokenPath, true);
    }
    catch { }

    return 0;
}
catch (Exception ex)
{
    AnsiConsole.WriteException(ex);
    return 1;
}

async Task SaveTo1Password(string email, string refreshToken)
{
    var itemName = $"Gmail - {email}";
    try
    {
        AnsiConsole.Status()
            .Start("Saving to 1Password...", ctx =>
            {
                // First try to edit existing item
                var editProcess = new Process
                {
                    StartInfo = new ProcessStartInfo
                    {
                        FileName = "op",
                        Arguments = $"item edit \"{itemName}\" " +
                                  $"\"refresh_token[password]={refreshToken}\"",
                        RedirectStandardOutput = true,
                        RedirectStandardError = true,
                        UseShellExecute = false,
                        CreateNoWindow = true
                    }
                };

                editProcess.Start();
                editProcess.WaitForExit();

                if (editProcess.ExitCode == 0)
                {
                    AnsiConsole.MarkupLine("[green]✓ Refresh token updated in 1Password successfully![/]");
                    return;
                }

                // Item doesn't exist, create it
                var createProcess = new Process
                {
                    StartInfo = new ProcessStartInfo
                    {
                        FileName = "op",
                        Arguments = $"item create --category login --title \"{itemName}\" " +
                                  $"\"refresh_token[password]={refreshToken}\"",
                        RedirectStandardOutput = true,
                        RedirectStandardError = true,
                        UseShellExecute = false,
                        CreateNoWindow = true
                    }
                };

                createProcess.Start();
                createProcess.WaitForExit();

                if (createProcess.ExitCode == 0)
                {
                    AnsiConsole.MarkupLine("[green]✓ Refresh token saved to new 1Password item successfully![/]");
                }
                else
                {
                    var error = createProcess.StandardError.ReadToEnd();
                    AnsiConsole.MarkupLine($"[red]Failed to save to 1Password: {Markup.Escape(error)}[/]");
                }
            });
    }
    catch (Exception ex)
    {
        AnsiConsole.MarkupLine($"[red]Error saving to 1Password: {Markup.Escape(ex.Message)}[/]");
    }
}
