#!/usr/bin/env dotnet
#:package Google.Apis.Gmail.v1@1.69.0.3742
#:package Google.Apis.Auth@1.69.0
#:package Spectre.Console@0.49.1
#:package Spectre.Console.Cli@0.49.1
#:package MimeKit@4.9.0
#:package CliWrap@3.6.6

#pragma warning disable IL2026
#pragma warning disable IL3050

using System.ComponentModel;
using System.Diagnostics;
using System.Diagnostics.CodeAnalysis;
using System.Text;
using System.Text.Json;
using System.Text.Json.Serialization;
using CliWrap;
using CliWrap.Buffered;
using Google.Apis.Auth.OAuth2;
using Google.Apis.Auth.OAuth2.Flows;
using Google.Apis.Auth.OAuth2.Responses;
using Google.Apis.Gmail.v1;
using Google.Apis.Gmail.v1.Data;
using Google.Apis.Services;
using MimeKit;
using Spectre.Console;
using Spectre.Console.Cli;

// Setup command app
var app = new CommandApp();
app.Configure(config =>
{
    config.SetApplicationName("gmail");

    config.AddCommand<SearchCommand>("search")
        .WithDescription("Search emails");

    config.AddCommand<DraftCommand>("draft")
        .WithDescription("Create a draft email");

    config.AddCommand<ReplyCommand>("reply")
        .WithDescription("Create a reply draft to an existing email");

    config.AddCommand<DownloadAttachmentsCommand>("download-attachments")
        .WithDescription("Download attachments from an email");
});

return app.Run(args);

// ============================================================================
// Command Handlers
// ============================================================================

class SearchCommand : AsyncCommand<SearchCommand.Settings>
{
    public class Settings : CommandSettings
    {
        [Description("Email account to operate on")]
        [CommandOption("--email")]
        public required string Account { get; init; }

        [Description("Gmail search query")]
        [CommandOption("-q|--query")]
        public string? Query { get; init; }

        [Description("Maximum number of results")]
        [CommandOption("-m|--max-results")]
        [DefaultValue(20)]
        public int MaxResults { get; init; } = 20;
    }

    public override async Task<int> ExecuteAsync(CommandContext context, Settings settings)
    {
        try
        {
            var service = await GmailHelper.GetGmailServiceAsync(settings.Account);
            // Use "me" to refer to the authenticated user (the account we got the token for)
            var request = service.Users.Messages.List("me");
            request.Q = settings.Query;
            request.MaxResults = settings.MaxResults;

            var response = await request.ExecuteAsync();
            var messages = new List<EmailMessage>();

            if (response.Messages != null)
            {
                foreach (var message in response.Messages)
                {
                    var fullMessage = await service.Users.Messages.Get("me", message.Id).ExecuteAsync();
                    messages.Add(GmailHelper.ParseMessage(fullMessage));
                }
            }

            var result = new { success = true, count = messages.Count, messages };
            Console.WriteLine(JsonSerializer.Serialize(result, GmailHelper.JsonOptions));
            return 0;
        }
        catch (Exception ex)
        {
            GmailHelper.OutputError(ex.Message);
            return 1;
        }
    }
}

class DraftCommand : AsyncCommand<DraftCommand.Settings>
{
    public class Settings : CommandSettings
    {
        [Description("Email account to operate on")]
        [CommandOption("--email")]
        public required string Account { get; init; }

        [Description("Recipient email address")]
        [CommandOption("-t|--to")]
        public required string To { get; init; }

        [Description("Email subject")]
        [CommandOption("-s|--subject")]
        public string? Subject { get; init; }

        [Description("Email body content")]
        [CommandOption("-b|--body")]
        public required string Body { get; init; }
    }

    public override async Task<int> ExecuteAsync(CommandContext context, Settings settings)
    {
        try
        {
            var service = await GmailHelper.GetGmailServiceAsync(settings.Account);
            var message = GmailHelper.CreateMimeMessage(
                settings.Account,
                settings.To,
                settings.Subject ?? "(No Subject)",
                settings.Body);

            var draft = new Draft
            {
                Message = new Message
                {
                    Raw = GmailHelper.Base64UrlEncode(message.ToString())
                }
            };

            var createdDraft = await service.Users.Drafts.Create(draft, "me").ExecuteAsync();

            var result = new
            {
                success = true,
                draftId = createdDraft.Id,
                messageId = createdDraft.Message.Id,
                message = "Draft created successfully"
            };
            Console.WriteLine(JsonSerializer.Serialize(result, GmailHelper.JsonOptions));
            return 0;
        }
        catch (Exception ex)
        {
            GmailHelper.OutputError(ex.Message);
            return 1;
        }
    }
}

class ReplyCommand : AsyncCommand<ReplyCommand.Settings>
{
    public class Settings : CommandSettings
    {
        [Description("Email account to operate on")]
        [CommandOption("--email")]
        public required string Account { get; init; }

        [Description("Gmail message ID to reply to")]
        [CommandOption("-m|--message-id")]
        public required string MessageId { get; init; }

        [Description("Reply body content")]
        [CommandOption("-b|--body")]
        public required string Body { get; init; }
    }

    public override async Task<int> ExecuteAsync(CommandContext context, Settings settings)
    {
        try
        {
            var service = await GmailHelper.GetGmailServiceAsync(settings.Account);

            // Get the original message (use "me" to refer to the authenticated user)
            var originalMessage = await service.Users.Messages.Get("me", settings.MessageId).ExecuteAsync();
            var parsed = GmailHelper.ParseMessage(originalMessage);

            // Create reply
            var message = GmailHelper.CreateReplyMimeMessage(settings.Account, parsed, settings.Body);

            var draft = new Draft
            {
                Message = new Message
                {
                    Raw = GmailHelper.Base64UrlEncode(message.ToString()),
                    ThreadId = originalMessage.ThreadId
                }
            };

            var createdDraft = await service.Users.Drafts.Create(draft, "me").ExecuteAsync();

            var result = new
            {
                success = true,
                draftId = createdDraft.Id,
                messageId = createdDraft.Message.Id,
                threadId = createdDraft.Message.ThreadId,
                inReplyTo = settings.MessageId,
                message = "Reply draft created successfully"
            };
            Console.WriteLine(JsonSerializer.Serialize(result, GmailHelper.JsonOptions));
            return 0;
        }
        catch (Exception ex)
        {
            GmailHelper.OutputError(ex.Message);
            return 1;
        }
    }
}

class DownloadAttachmentsCommand : AsyncCommand<DownloadAttachmentsCommand.Settings>
{
    public class Settings : CommandSettings
    {
        [Description("Email account to operate on")]
        [CommandOption("--email")]
        public required string Account { get; init; }

        [Description("Gmail message ID")]
        [CommandOption("-m|--message-id")]
        public required string MessageId { get; init; }

        [Description("Output directory for attachments")]
        [CommandOption("-o|--output-dir")]
        public required string OutputDir { get; init; }
    }

    public override async Task<int> ExecuteAsync(CommandContext context, Settings settings)
    {
        try
        {
            var service = await GmailHelper.GetGmailServiceAsync(settings.Account);
            // Use "me" to refer to the authenticated user
            var message = await service.Users.Messages.Get("me", settings.MessageId).ExecuteAsync();

            var attachments = new List<AttachmentInfo>();

            if (message.Payload.Parts != null)
            {
                await ProcessPartsAsync(service, settings.MessageId, message.Payload.Parts, settings.OutputDir, attachments);
            }

            var result = new
            {
                success = true,
                messageId = settings.MessageId,
                attachmentCount = attachments.Count,
                attachments
            };
            Console.WriteLine(JsonSerializer.Serialize(result, GmailHelper.JsonOptions));
            return 0;
        }
        catch (Exception ex)
        {
            GmailHelper.OutputError(ex.Message);
            return 1;
        }
    }

    private async Task ProcessPartsAsync(GmailService service, string messageId, IList<Google.Apis.Gmail.v1.Data.MessagePart> parts, string outputDir, List<AttachmentInfo> attachments)
    {
        foreach (var part in parts)
        {
            // Check if this part has an attachment
            if (!string.IsNullOrEmpty(part.Filename) && part.Body?.AttachmentId != null)
            {
                // Download the attachment (use "me" to refer to the authenticated user)
                var attachment = await service.Users.Messages.Attachments.Get("me", messageId, part.Body.AttachmentId).ExecuteAsync();
                var data = GmailHelper.Base64UrlDecodeBytes(attachment.Data);

                // Create output directory if it doesn't exist
                Directory.CreateDirectory(outputDir);

                // Save the file
                var filePath = Path.Combine(outputDir, part.Filename);
                await File.WriteAllBytesAsync(filePath, data);

                attachments.Add(new AttachmentInfo
                {
                    Filename = part.Filename,
                    FilePath = filePath,
                    Size = data.Length,
                    MimeType = part.MimeType
                });
            }

            // Recursively process nested parts
            if (part.Parts != null)
            {
                await ProcessPartsAsync(service, messageId, part.Parts, outputDir, attachments);
            }
        }
    }
}

// ============================================================================
// Gmail API Service
// ============================================================================

// IMPORTANT FOR AI AGENTS:
// When outputting JSON, ALWAYS use Console.WriteLine, NEVER use AnsiConsole.WriteLine!
// AnsiConsole.WriteLine wraps long lines at terminal width (~80 chars), inserting literal
// newlines into JSON strings, which breaks JSON parsing. This is a bug in Spectre.Console.
// Use AnsiConsole only for human-readable formatted output, not for machine-readable JSON.

static class GmailHelper
{
    // OAuth 2.0 credentials - REPLACE WITH YOUR OWN
    //
    // To get your credentials:
    // 1. Go to https://console.cloud.google.com/
    // 2. Create a new project (or select an existing one)
    // 3. Enable the Gmail API for your project
    // 4. Go to "Credentials" > "Create Credentials" > "OAuth client ID"
    // 5. Choose "Desktop app" as the application type
    // 6. Copy the Client ID and Client Secret and paste them below
    //
    // Note: For desktop/CLI apps, client secrets cannot be fully protected per OAuth 2.0 spec,
    // so it's safe to commit these to your private repository (but not a public one).
    private const string ClientId = "YOUR_CLIENT_ID.apps.googleusercontent.com";
    private const string ClientSecret = "YOUR_CLIENT_SECRET";

    public static readonly JsonSerializerOptions JsonOptions = new()
    {
        WriteIndented = true,
        DefaultIgnoreCondition = JsonIgnoreCondition.WhenWritingNull,
        TypeInfoResolver = new System.Text.Json.Serialization.Metadata.DefaultJsonTypeInfoResolver()
    };

    public static async Task<GmailService> GetGmailServiceAsync(string account)
    {
        // Try to get a valid access token from cache first
        var cachedToken = await ReadCachedTokenAsync(account);
        if (cachedToken != null && cachedToken.ExpiresAt > DateTime.UtcNow.AddMinutes(5))
        {
            // Use cached token (with 5 minute buffer before expiry)
            var credential = GoogleCredential.FromAccessToken(cachedToken.AccessToken);
            return new GmailService(new BaseClientService.Initializer
            {
                HttpClientInitializer = credential,
                ApplicationName = "Gmail AI Agent"
            });
        }

        // Cache miss or expired - need to refresh from 1Password
        // Use file lock to prevent multiple processes from hitting 1Password simultaneously
        var lockFilePath = GetLockFilePath(account);
        Directory.CreateDirectory(GetCacheDirectory());

        const int maxWaitMs = 30000; // 30 seconds max wait
        const int pollIntervalMs = 100; // Poll every 100ms
        var stopwatch = Stopwatch.StartNew();

        while (stopwatch.ElapsedMilliseconds < maxWaitMs)
        {
            FileStream? lockStream = null;
            try
            {
                // Try to acquire exclusive lock
                lockStream = new FileStream(lockFilePath, FileMode.OpenOrCreate, FileAccess.ReadWrite, FileShare.None);

                // We got the lock! But first check if another process already refreshed while we waited
                cachedToken = await ReadCachedTokenAsync(account);
                if (cachedToken != null && cachedToken.ExpiresAt > DateTime.UtcNow.AddMinutes(5))
                {
                    lockStream.Dispose();
                    var credential = GoogleCredential.FromAccessToken(cachedToken.AccessToken);
                    return new GmailService(new BaseClientService.Initializer
                    {
                        HttpClientInitializer = credential,
                        ApplicationName = "Gmail AI Agent"
                    });
                }

                // Still need to refresh - we're the winner, do the actual refresh
                var accessToken = await RefreshTokenFromOnePasswordAsync(account);

                lockStream.Dispose();

                var credential2 = GoogleCredential.FromAccessToken(accessToken);
                return new GmailService(new BaseClientService.Initializer
                {
                    HttpClientInitializer = credential2,
                    ApplicationName = "Gmail AI Agent"
                });
            }
            catch (IOException)
            {
                // Lock held by another process - wait and check cache
                lockStream?.Dispose();

                await Task.Delay(pollIntervalMs);

                // Check if the other process has written a valid token
                cachedToken = await ReadCachedTokenAsync(account);
                if (cachedToken != null && cachedToken.ExpiresAt > DateTime.UtcNow.AddMinutes(5))
                {
                    var credential = GoogleCredential.FromAccessToken(cachedToken.AccessToken);
                    return new GmailService(new BaseClientService.Initializer
                    {
                        HttpClientInitializer = credential,
                        ApplicationName = "Gmail AI Agent"
                    });
                }
                // Continue waiting...
            }
        }

        throw new Exception($"Timeout waiting for Gmail token refresh (waited {maxWaitMs}ms)");
    }

    static async Task<string> RefreshTokenFromOnePasswordAsync(string account)
    {
        var credentials = await GetCredentialsFromOnePasswordAsync(account);

        // Manually refresh the token using HTTP
        var httpClient = new HttpClient();
        var tokenRequest = new Dictionary<string, string>
        {
            ["client_id"] = credentials.ClientId,
            ["client_secret"] = credentials.ClientSecret,
            ["refresh_token"] = credentials.RefreshToken,
            ["grant_type"] = "refresh_token"
        };

        var response = await httpClient.PostAsync(
            "https://oauth2.googleapis.com/token",
            new FormUrlEncodedContent(tokenRequest));

        var responseContent = await response.Content.ReadAsStringAsync();

        if (!response.IsSuccessStatusCode)
        {
            throw new Exception($"Token refresh failed: {responseContent}");
        }

        var tokenData = JsonSerializer.Deserialize<JsonElement>(responseContent, JsonOptions);

        if (!tokenData.TryGetProperty("access_token", out var accessTokenElement))
        {
            throw new Exception($"No access_token in response: {responseContent}");
        }

        var accessToken = accessTokenElement.GetString();

        // Get expiry time (default to 1 hour if not provided)
        var expiresIn = tokenData.TryGetProperty("expires_in", out var expiresInElement)
            ? expiresInElement.GetInt32()
            : 3600;

        // Cache the new token
        await WriteCachedTokenAsync(account, new CachedToken
        {
            AccessToken = accessToken ?? throw new Exception("Access token is null"),
            ExpiresAt = DateTime.UtcNow.AddSeconds(expiresIn)
        });

        return accessToken;
    }

    // ============================================================================
    // Token Caching
    // ============================================================================

    static string GetCacheDirectory()
    {
        var home = Environment.GetFolderPath(Environment.SpecialFolder.UserProfile);
        return Path.Combine(home, ".cache", "gmail-tokens");
    }

    static string GetCacheFilePath(string account)
    {
        return Path.Combine(GetCacheDirectory(), $"{account}.json");
    }

    static string GetLockFilePath(string account)
    {
        return Path.Combine(GetCacheDirectory(), $"{account}.lock");
    }

    static async Task<CachedToken?> ReadCachedTokenAsync(string account)
    {
        try
        {
            var cachePath = GetCacheFilePath(account);
            if (!File.Exists(cachePath))
                return null;

            var json = await File.ReadAllTextAsync(cachePath);
            return JsonSerializer.Deserialize<CachedToken>(json, JsonOptions);
        }
        catch
        {
            // If cache read fails for any reason, just return null
            return null;
        }
    }

    static async Task WriteCachedTokenAsync(string account, CachedToken token)
    {
        try
        {
            var cacheDir = GetCacheDirectory();
            Directory.CreateDirectory(cacheDir);

            var cachePath = GetCacheFilePath(account);
            var json = JsonSerializer.Serialize(token, JsonOptions);
            await File.WriteAllTextAsync(cachePath, json);

            // Set restrictive permissions (Unix only)
            if (!OperatingSystem.IsWindows())
            {
                File.SetUnixFileMode(cachePath, UnixFileMode.UserRead | UnixFileMode.UserWrite);
            }
        }
        catch
        {
            // If cache write fails, just continue - we can still function without cache
        }
    }

    // ============================================================================
    // 1Password Integration
    // ============================================================================

    static async Task<GmailCredentials> GetCredentialsFromOnePasswordAsync(string account)
    {
        // Expected 1Password item structure:
        // Item name: "Gmail - {account}"
        // Fields:
        //   - refresh_token (password)
        //
        // Note: client_id and client_secret are embedded in the source code

        var itemName = $"Gmail - {account}";
        var refreshToken = await RunOnePasswordCommandAsync($"item get \"{itemName}\" --fields refresh_token --reveal");

        return new GmailCredentials
        {
            ClientId = ClientId,
            ClientSecret = ClientSecret,
            RefreshToken = refreshToken.Trim()
        };
    }

    static async Task<string> RunOnePasswordCommandAsync(string arguments)
    {
        var result = await Cli.Wrap("op")
            .WithArguments(arguments)
            .WithValidation(CommandResultValidation.None)
            .ExecuteBufferedAsync();

        if (result.ExitCode != 0)
        {
            throw new Exception($"1Password CLI error: {result.StandardError}");
        }

        return result.StandardOutput;
    }

    // ============================================================================
    // Message Parsing and Creation
    // ============================================================================

    public static EmailMessage ParseMessage(Message message)
    {
        var headers = message.Payload.Headers;
        var result = new EmailMessage
        {
            Id = message.Id,
            ThreadId = message.ThreadId,
            From = GetHeader(headers, "From"),
            To = GetHeader(headers, "To"),
            Subject = GetHeader(headers, "Subject"),
            Date = GetHeader(headers, "Date"),
            Snippet = message.Snippet
        };

        // Get body
        if (message.Payload.Body?.Data != null)
        {
            result.Body = Base64UrlDecode(message.Payload.Body.Data);
        }
        else if (message.Payload.Parts != null)
        {
            foreach (var part in message.Payload.Parts)
            {
                if (part.MimeType == "text/plain" && part.Body?.Data != null)
                {
                    result.Body = Base64UrlDecode(part.Body.Data);
                    break;
                }
            }

            // Fallback to HTML if no plain text
            if (string.IsNullOrEmpty(result.Body))
            {
                foreach (var part in message.Payload.Parts)
                {
                    if (part.MimeType == "text/html" && part.Body?.Data != null)
                    {
                        result.BodyHtml = Base64UrlDecode(part.Body.Data);
                        break;
                    }
                }
            }
        }

        return result;
    }

    public static MimeMessage CreateMimeMessage(string from, string to, string subject, string body)
    {
        var message = new MimeMessage();
        message.From.Add(MailboxAddress.Parse(from));
        message.To.Add(MailboxAddress.Parse(to));
        message.Subject = subject;
        message.Body = new TextPart("plain") { Text = body };
        return message;
    }

    public static MimeMessage CreateReplyMimeMessage(string from, EmailMessage originalMessage, string body)
    {
        var message = new MimeMessage();
        message.From.Add(MailboxAddress.Parse(from));
        message.To.Add(MailboxAddress.Parse(originalMessage.From ?? ""));

        // Add Re: prefix if not already there
        var subject = originalMessage.Subject ?? "";
        if (!subject.StartsWith("Re:", StringComparison.OrdinalIgnoreCase))
        {
            subject = $"Re: {subject}";
        }
        message.Subject = subject;

        // Set reply headers
        message.InReplyTo = originalMessage.Id;
        message.References.Add(originalMessage.Id);

        message.Body = new TextPart("plain") { Text = body };
        return message;
    }

    public static string GetHeader(IList<MessagePartHeader> headers, string name)
    {
        return headers.FirstOrDefault(h => h.Name.Equals(name, StringComparison.OrdinalIgnoreCase))?.Value ?? "";
    }

    public static string Base64UrlEncode(string input)
    {
        var bytes = Encoding.UTF8.GetBytes(input);
        return Convert.ToBase64String(bytes)
            .Replace('+', '-')
            .Replace('/', '_')
            .Replace("=", "");
    }

    static string Base64UrlDecode(string input)
    {
        var output = input.Replace('-', '+').Replace('_', '/');
        switch (output.Length % 4)
        {
            case 2: output += "=="; break;
            case 3: output += "="; break;
        }
        var bytes = Convert.FromBase64String(output);
        return Encoding.UTF8.GetString(bytes);
    }

    public static byte[] Base64UrlDecodeBytes(string input)
    {
        var output = input.Replace('-', '+').Replace('_', '/');
        switch (output.Length % 4)
        {
            case 2: output += "=="; break;
            case 3: output += "="; break;
        }
        return Convert.FromBase64String(output);
    }

    public static void OutputError(string message)
    {
        var error = new { success = false, error = message };
        Console.WriteLine(JsonSerializer.Serialize(error, JsonOptions));
        Environment.Exit(1);
    }
}

// ============================================================================
// Data Models
// ============================================================================

class CachedToken
{
    [JsonPropertyName("access_token")]
    public required string AccessToken { get; set; }

    [JsonPropertyName("expires_at")]
    public required DateTime ExpiresAt { get; set; }
}

class GmailCredentials
{
    public required string ClientId { get; set; }
    public required string ClientSecret { get; set; }
    public required string RefreshToken { get; set; }
}

class EmailMessage
{
    [JsonPropertyName("id")]
    public string? Id { get; set; }

    [JsonPropertyName("threadId")]
    public string? ThreadId { get; set; }

    [JsonPropertyName("from")]
    public string? From { get; set; }

    [JsonPropertyName("to")]
    public string? To { get; set; }

    [JsonPropertyName("subject")]
    public string? Subject { get; set; }

    [JsonPropertyName("date")]
    public string? Date { get; set; }

    [JsonPropertyName("snippet")]
    public string? Snippet { get; set; }

    [JsonPropertyName("body")]
    public string? Body { get; set; }

    [JsonPropertyName("bodyHtml")]
    public string? BodyHtml { get; set; }
}

class AttachmentInfo
{
    [JsonPropertyName("filename")]
    public string? Filename { get; set; }

    [JsonPropertyName("filePath")]
    public string? FilePath { get; set; }

    [JsonPropertyName("size")]
    public long Size { get; set; }

    [JsonPropertyName("mimeType")]
    public string? MimeType { get; set; }
}
