# Stage Entry Queries Reference

HubSpot stage entry property patterns for accurate goal tracking.

## Stage Entry Property Names (VERIFIED)

| Stage | Metric Name | HubSpot Property |
|-------|-------------|------------------|
| 1 | new_leads | `createdate` (Contacts API) |
| 2 | discovery_scheduled | `hs_v2_date_entered_1090865183` |
| 3 | discovery_complete | `hs_v2_date_entered_d2a08d6f_cc04_4423_9215_594fe682e538_1563988860` |
| 4 | rate_creation | `hs_v2_date_entered_e1c4321e_afb6_4b29_97d4_2b2425488535_710201006` |
| 5 | proposals_sent | `hs_v2_date_entered_d607df25_2c6d_4a5d_9835_6ed1e4f4020a_234478410` |
| 6 | setup_docs_sent | `hs_v2_date_entered_4e549d01_674b_4b31_8a90_91ec03122715_516305102` |
| 7 | implementation | `hs_v2_date_entered_08d9c411_5e1b_487b_8732_9c2bcbbd0307_1395957378` |
| 8 | started_shipping | `hs_v2_date_entered_3fd46d94_78b4_452b_8704_62a338a210fb_628848678` |

## Query Templates

### Stage 1: New Leads (Contacts API)

```python
# CORRECT: Use Contacts API
url = "https://api.hubapi.com/crm/v3/objects/contacts/search"
payload = {
    "filterGroups": [{
        "filters": [
            {
                "propertyName": "createdate",
                "operator": "BETWEEN",
                "highValue": str(end_ms),
                "value": str(start_ms)
            },
            {
                "propertyName": "hubspot_owner_id",
                "operator": "EQ",
                "value": "699257003"
            }
        ]
    }]
}
```

### Stages 2-8: Pipeline Stages (Deals API)

```python
# CORRECT: Use stage entry property
url = "https://api.hubapi.com/crm/v3/objects/deals/search"
payload = {
    "filterGroups": [{
        "filters": [
            {
                "propertyName": "hs_v2_date_entered_1090865183",  # discovery_scheduled
                "operator": "BETWEEN",
                "highValue": str(end_ms),
                "value": str(start_ms)
            },
            {
                "propertyName": "hubspot_owner_id",
                "operator": "EQ",
                "value": "699257003"
            },
            {
                "propertyName": "pipeline",
                "operator": "EQ",
                "value": "8bd9336b-4767-4e67-9fe2-35dfcad7c8be"
            }
        ]
    }]
}
```

## Date Filtering Rules

### Convert Dates to Unix Milliseconds

```python
from datetime import datetime

def to_unix_ms(dt: datetime) -> str:
    """Convert datetime to Unix ms STRING (HubSpot requirement)"""
    return str(int(dt.timestamp() * 1000))

# Example: December 16, 2025 00:00:00
start = datetime(2025, 12, 16, 0, 0, 0)
start_ms = to_unix_ms(start)  # "1734307200000"
```

### Week Boundaries

```python
def get_week_range():
    """Get current week Monday-Sunday"""
    today = datetime.now()
    monday = today - timedelta(days=today.weekday())
    monday = monday.replace(hour=0, minute=0, second=0, microsecond=0)
    sunday = monday + timedelta(days=6, hours=23, minutes=59, seconds=59)
    return to_unix_ms(monday), to_unix_ms(sunday)
```

## Common Mistakes to Avoid

### WRONG: Counting Current Stage

```python
# WRONG - This counts deals IN the stage NOW, not entries
{
    "propertyName": "dealstage",
    "operator": "EQ",
    "value": "1090865183"
}
```

### WRONG: Using Deals API for new_leads

```python
# WRONG - Deals don't exist until after discovery
# new_leads are CONTACTS, not deals
url = "https://api.hubapi.com/crm/v3/objects/deals/search"
```

### WRONG: Integer timestamps

```python
# WRONG - HubSpot requires STRING timestamps
"value": 1734307200000

# CORRECT
"value": "1734307200000"
```

## Pipeline and Owner IDs

```python
OWNER_ID = "699257003"  # Brett Walker
PIPELINE_ID = "8bd9336b-4767-4e67-9fe2-35dfcad7c8be"
```

## Full Working Example

```python
import requests
from datetime import datetime, timedelta

def count_stage_entries(stage_prop: str, start_ms: str, end_ms: str) -> int:
    """Count deals that entered a stage in date range"""

    url = "https://api.hubapi.com/crm/v3/objects/deals/search"
    headers = {
        "Authorization": f"Bearer {API_KEY}",
        "Content-Type": "application/json"
    }

    payload = {
        "filterGroups": [{
            "filters": [
                {"propertyName": stage_prop, "operator": "BETWEEN",
                 "value": start_ms, "highValue": end_ms},
                {"propertyName": "hubspot_owner_id", "operator": "EQ",
                 "value": "699257003"},
                {"propertyName": "pipeline", "operator": "EQ",
                 "value": "8bd9336b-4767-4e67-9fe2-35dfcad7c8be"}
            ]
        }],
        "limit": 100
    }

    response = requests.post(url, headers=headers, json=payload)
    return response.json().get("total", 0)
```
