# Goal Calculations Reference

Formulas and patterns for all goal tracking math.

## Time Period Conversions

| From | To | Formula |
|------|-----|---------|
| Weekly | Monthly | × 4.3 |
| Weekly | Quarterly | × 13 |
| Monthly | Annual | × 12 |
| Quarterly | Annual | × 4 |

## Quarterly Distribution

Annual goals are NOT evenly distributed. Use this pattern:

| Quarter | % of Annual | Rationale |
|---------|-------------|-----------|
| Q1 | 15% | Ramp-up, post-holiday |
| Q2 | 23% | Momentum building |
| Q3 | 28% | Pre-peak preparation |
| Q4 | 34% | Peak season, year-end push |

**Example**: $1M annual target
- Q1: $150K
- Q2: $230K
- Q3: $280K
- Q4: $340K

## Full Funnel Pattern

The 15/6/4/4/3/2/1/1 pattern represents weekly targets working backward from 1 close:

| Stage | Weekly Target | Conversion to Next |
|-------|---------------|-------------------|
| New Leads | 15 | 40% → Scheduled |
| Discovery Scheduled | 6 | 67% show rate |
| Discovery Complete | 4 | 100% → Rate |
| Rate Creation | 4 | 75% → Proposal |
| Proposals Sent | 3 | 67% → Setup |
| Setup Docs | 2 | 50% → Impl |
| Implementation | 1 | 100% → Close |
| Closed | 1 | - |

**Overall Conversion**: 15 leads → 1 close = 6.6%

## Archetype Weekly Patterns

| Archetype | Pattern | Description |
|-----------|---------|-------------|
| whale_hunter | 2/1/2/1/1/1/0/0 | Few large deals |
| volume_closer | 7/3/5/2/2/1/1/1 | High activity |
| farm_the_base | 3/2/4/1/1/1/1/0 | Expansion focus |
| hunter_mode | 8/2/2/1/1/0/0/0 | Heavy prospecting |
| executive_sponsor | 2/2/3/2/1/1/1/0 | Strategic deals |
| volume_closer_transition | 15/6/4/4/3/2/1/1 | Brett's current |

## Goal Status Thresholds

```python
def get_status(actual: int, target: int) -> str:
    if target == 0:
        return "no_target"

    pct = (actual / target) * 100

    if pct >= 90:
        return "on_track"      # Green
    elif pct >= 70:
        return "behind"        # Yellow
    elif pct >= 50:
        return "at_risk"       # Orange
    else:
        return "critical"      # Red
```

## Working Backward from Revenue

To calculate required activity from revenue target:

```python
def calculate_required_leads(revenue_target: float, avg_deal_size: float) -> dict:
    """Calculate full funnel requirements from revenue target"""

    deals_needed = revenue_target / avg_deal_size

    # Work backward through funnel
    implementation = deals_needed / 1.0      # 100% impl → close
    setup_docs = implementation / 0.5        # 50% setup → impl
    proposals = setup_docs / 0.67            # 67% proposal → setup
    rate_creation = proposals / 0.75         # 75% rate → proposal
    discovery_complete = rate_creation / 1.0 # 100% complete → rate
    discovery_scheduled = discovery_complete / 0.67  # 67% show rate
    new_leads = discovery_scheduled / 0.4    # 40% lead → scheduled

    return {
        "new_leads": int(new_leads),
        "discovery_scheduled": int(discovery_scheduled),
        "discovery_complete": int(discovery_complete),
        "rate_creation": int(rate_creation),
        "proposals_sent": int(proposals),
        "setup_docs_sent": int(setup_docs),
        "implementation": int(implementation),
        "deals_closed": int(deals_needed)
    }
```

## Example Calculations

### Monthly from Weekly

```python
weekly_targets = {
    "new_leads": 15,
    "discovery_scheduled": 6,
    "discovery_complete": 4,
    "rate_creation": 4,
    "proposals_sent": 3,
    "setup_docs_sent": 2,
    "implementation": 1,
    "deals_closed": 1
}

monthly_targets = {k: int(v * 4.3) for k, v in weekly_targets.items()}
# Result: 64/26/17/17/13/9/4/4
```

### Quarterly from Weekly

```python
quarterly_targets = {k: int(v * 13) for k, v in weekly_targets.items()}
# Result: 195/78/52/52/39/26/13/13
```

### Progress Percentage

```python
def progress_pct(actual: int, target: int) -> float:
    """Calculate progress percentage with division protection"""
    if target == 0:
        return 0.0
    return round((actual / target) * 100, 1)
```

## Dashboard Display Format

```
Stage                | Actual | Target | Progress | Status
---------------------|--------|--------|----------|--------
New Leads            |     12 |     15 |    80.0% | behind
Discovery Scheduled  |      4 |      6 |    66.7% | at_risk
Discovery Complete   |      3 |      4 |    75.0% | behind
Rate Creation        |      4 |      4 |   100.0% | on_track
Proposals Sent       |      2 |      3 |    66.7% | at_risk
Setup Docs Sent      |      1 |      2 |    50.0% | at_risk
Implementation       |      1 |      1 |   100.0% | on_track
Deals Closed         |      0 |      1 |     0.0% | critical
```

## Conversion Rate Monitoring

Track actual conversion rates vs expected:

| Conversion | Expected | Warning If |
|------------|----------|------------|
| Lead → Scheduled | 40% | < 30% |
| Scheduled → Complete | 67% | < 50% |
| Complete → Rate | 100% | < 80% |
| Rate → Proposal | 75% | < 60% |
| Proposal → Setup | 67% | < 50% |
| Setup → Impl | 50% | < 35% |
| Impl → Close | 100% | < 90% |
