# Known Failures Reference

Documented failures with prevention patterns. Check this BEFORE any goal calculation.

## F001: Entry vs Current Stage Counting

**Date Discovered**: 2025-12-16

**What Happened**:
Counted deals that are currently IN a stage instead of deals that ENTERED the stage during the measurement period.

**Why It's Wrong**:
A deal in Stage 3 today might have entered Stage 3 weeks ago. Counting current stage membership shows stale pipeline state, not actual activity/progress.

**Symptoms**:
- Numbers don't change even when deals move
- Same deals counted week after week
- Metrics seem "stuck"

**Prevention**:
```python
# WRONG - counts current state
{
    "propertyName": "dealstage",
    "operator": "EQ",
    "value": "1090865183"
}

# CORRECT - counts entries in date range
{
    "propertyName": "hs_v2_date_entered_1090865183",
    "operator": "BETWEEN",
    "value": str(start_ms),
    "highValue": str(end_ms)
}
```

**Verification**: After any stage query, ask: "Am I counting WHEN deals entered, or WHERE deals are now?"

---

## F002: Contacts vs Deals for New Leads

**Date Discovered**: 2025-12-16

**What Happened**:
Queried Deals API for "new_leads" metric, got 0 results.

**Why It's Wrong**:
In the FirstMile sales process, deals are NOT created until after discovery is scheduled. New leads exist only as Contacts (and Companies) in HubSpot. There is no "Stage 0" in the deals pipeline.

**Symptoms**:
- new_leads always shows 0
- Brand Scout work not reflected in goals
- Contact creation not tracked

**Prevention**:
```python
# WRONG - deals don't exist yet
url = "https://api.hubapi.com/crm/v3/objects/deals/search"

# CORRECT - leads are contacts
url = "https://api.hubapi.com/crm/v3/objects/contacts/search"
payload = {
    "filterGroups": [{
        "filters": [
            {"propertyName": "createdate", "operator": "BETWEEN", ...},
            {"propertyName": "hubspot_owner_id", "operator": "EQ", "value": "699257003"}
        ]
    }]
}
```

**Verification**: Before querying for leads, confirm: "Am I using Contacts API, not Deals API?"

---

## F003: Archetype Goal Mismatch

**Date Discovered**: 2025-12-16

**What Happened**:
Dashboard showed incorrect weekly targets because it used default archetype instead of rep's actual archetype.

**Why It's Wrong**:
Each rep has a specific archetype in their profile that determines their weekly goal pattern. Using wrong archetype = wrong targets = misleading dashboard.

**Symptoms**:
- Goals don't match expectations
- "On track" when actually behind (or vice versa)
- Weekly pattern doesn't fit rep's actual work style

**Prevention**:
```python
# ALWAYS load rep profile first
rep_config = load_yaml(".claude/config/reps/brett_walker.yaml")
archetype = rep_config.get("archetype", "volume_closer")  # Default if missing

# Then load archetype pattern
team_config = load_yaml(".claude/config/cvm_goals_2026.yaml")
weekly_pattern = team_config["archetypes"][archetype]["weekly"]
```

**Verification**: Before displaying goals, confirm: "Which archetype am I using? Is it from the rep profile?"

---

## F004: Integer Timestamps (Not String)

**Date Discovered**: 2025-12-16

**What Happened**:
HubSpot API returned unexpected results when timestamps were passed as integers instead of strings.

**Why It's Wrong**:
HubSpot expects Unix millisecond timestamps as STRINGS in JSON payloads.

**Symptoms**:
- Empty results when data exists
- Inconsistent filtering behavior
- API errors in some cases

**Prevention**:
```python
# WRONG
"value": 1734307200000

# CORRECT
"value": "1734307200000"

# Helper function
def to_unix_ms(dt: datetime) -> str:
    return str(int(dt.timestamp() * 1000))
```

**Verification**: Before any HubSpot query with dates, confirm: "Are my timestamps strings?"

---

## F005: Division by Zero in Progress Calculation

**Date Discovered**: 2025-12-16

**What Happened**:
Dashboard crashed when calculating progress percentage for a stage with target = 0.

**Why It's Wrong**:
Some archetypes have 0 targets for certain stages (e.g., whale_hunter has 0 for Stage 7-8). Division by zero crashes the calculation.

**Symptoms**:
- ZeroDivisionError
- Dashboard fails to render
- NaN or Infinity values

**Prevention**:
```python
# WRONG
progress = actual / target * 100

# CORRECT
progress = (actual / target * 100) if target > 0 else 0
```

**Verification**: Every division involving targets must have zero-protection.

---

## Pre-Calculation Checklist

Before running any goal calculation:

1. [ ] Am I using hs_v2_date_entered for stage metrics? (F001)
2. [ ] Am I using Contacts API for new_leads? (F002)
3. [ ] Did I load the rep's archetype from their profile? (F003)
4. [ ] Are all timestamps strings? (F004)
5. [ ] Do all divisions have zero-protection? (F005)

---

## Adding New Failures

When a new failure is discovered:

1. Add entry here with date, description, prevention
2. Update expertise.yaml learned_behaviors.failures
3. Increment expertise.yaml version (patch level)
4. Add to pre-calculation checklist if applicable
