# Improvement Patterns

Patterns for fixing common issues found during grading.

## Contents

- [Size Reduction](#size-reduction)
- [Adding TOC](#adding-toc)
- [Extracting to References](#extracting-to-references)
- [Fixing Anti-Patterns](#fixing-anti-patterns)
- [Monorepo Splitting](#monorepo-splitting)
- [Cross-Platform Support](#cross-platform-support)

---

## Size Reduction

### When to Apply
- File > 300 lines
- Token count > 1,200
- AGENTS.md > 15KB

### Pattern: Extract by Topic

**Before:**
```markdown
# Project

## Commands
...20 lines...

## API Patterns
...150 lines of API documentation...

## Database
...100 lines of schema docs...

## Testing
...80 lines of test patterns...
```

**After:**
```markdown
# Project

## Commands
npm run build | npm test | npm lint

## Documentation
- API Patterns: @docs/api-patterns.md
- Database: @docs/database.md
- Testing: @docs/testing.md
```

### Pattern: Extract Code Examples

**Before:**
```markdown
## Example API Handler

```typescript
// 50+ lines of code example
import express from 'express';
import { validateInput } from './validation';
...
```
```

**After:**
```markdown
## Example API Handler
See @examples/api-handler.ts for complete example.

Key points:
- Validate inputs with Zod
- Use async/await
- Return consistent error format
```

---

## Adding TOC

### When to Apply
- File > 100 lines
- No existing `## Contents` section

### Pattern: Simple TOC

Insert after title, before first section:

```markdown
# Project Name

Brief description.

## Contents

- [Commands](#commands)
- [Structure](#structure)
- [Conventions](#conventions)
- [Documentation](#documentation)

---

## Commands
...
```

### Pattern: Detailed TOC (>300 lines)

```markdown
## Contents

### Getting Started
- [Commands](#commands)
- [Setup](#setup)

### Development
- [Code Style](#code-style)
- [Testing](#testing)
- [Debugging](#debugging)

### Reference
- [API](#api)
- [Database](#database)
```

### Anchor Generation Rules

Convert header to anchor:
1. Lowercase
2. Remove punctuation (except hyphens)
3. Replace spaces with hyphens
4. Remove duplicate hyphens

Examples:
- `## Quick Start` → `#quick-start`
- `## API v2.0` → `#api-v20`
- `## What's New?` → `#whats-new`

---

## Extracting to References

### When to Apply
- Section > 50 lines
- Content not used in every session
- Detailed documentation inline

### Pattern: @import (CLAUDE.md)

**Step 1:** Create reference file
```markdown
# docs/api-patterns.md

## REST Conventions
...detailed content...

## Error Handling
...detailed content...
```

**Step 2:** Replace inline content
```markdown
## API Development
Quick reference: POST/GET/PUT/DELETE standard patterns.
Full guide: @docs/api-patterns.md
```

### Pattern: Link (AGENTS.md)

```markdown
## API Development
See [API Patterns](docs/api-patterns.md) for detailed conventions.
```

### What to Extract

| Content Type | Extract? | Reason |
|--------------|----------|--------|
| Detailed API docs | ✅ Yes | Rarely needed in full |
| Full schema definitions | ✅ Yes | Load on demand |
| Code examples >20 lines | ✅ Yes | Reference when needed |
| Build/test commands | ❌ No | Used frequently |
| File structure overview | ❌ No | Orientation is key |
| Commit conventions | ❌ No | Short, always relevant |

---

## Fixing Anti-Patterns

### Style Rules → Linters

**Before:**
```markdown
## Code Style
- Use 2-space indentation
- Always use semicolons
- Prefer single quotes
- Max line length 80
```

**After:**
```markdown
## Code Style
Enforced by ESLint and Prettier. Run `npm run lint` to check.
Config: `.eslintrc.json` and `.prettierrc`
```

### Negative-Only → Alternatives

**Before:**
```markdown
- Never use var
- Don't use console.log
- Avoid any type
```

**After:**
```markdown
- Use `const` or `let` instead of `var`
- Use `logger.debug()` instead of `console.log`
- Use specific types; `unknown` if truly dynamic
```

### Secrets → Environment Variables

**Before:**
```markdown
## API Config
API_KEY=sk-abc123...
DATABASE_URL=postgres://user:pass@...
```

**After:**
```markdown
## API Config
Credentials in environment variables (see `.env.example`).
Never commit secrets to this file.
```

### Explaining Obvious → Assume Knowledge

**Before:**
```markdown
## JSON
JSON (JavaScript Object Notation) is a lightweight
data-interchange format that is easy for humans to
read and write...
```

**After:**
```markdown
## Data Format
API uses JSON. See @docs/api-schema.md for response shapes.
```

### Vague → Specific

**Before:**
```markdown
- Be careful with the database
- Make sure tests pass
- Keep code clean
```

**After:**
```markdown
- Use transactions for multi-table operations
- Run `npm test` before committing; CI blocks failures
- Run `npm run lint:fix` before committing
```

---

## Monorepo Splitting

### When to Apply
- Project has packages/ or apps/ directory
- Root CLAUDE.md > 300 lines
- Different packages have different conventions

### Pattern: Root + Package Configs

**Step 1:** Create lean root
```markdown
# Monorepo

## Workspace Commands
pnpm build    # Build all
pnpm test     # Test all
pnpm --filter <pkg> <cmd>  # Single package

## Packages
- web/: React frontend
- api/: Express backend
- shared/: Common utilities

Each package has its own CLAUDE.md.
```

**Step 2:** Create package configs
```markdown
# packages/web/CLAUDE.md

## Commands (local to this package)
pnpm dev      # Start dev server
pnpm build    # Production build

## Patterns
- Components in src/components/
- Use React hooks, no class components
- Tailwind for styling
```

**Step 3:** Move shared patterns
```markdown
# packages/shared/CLAUDE.md

## Usage
Shared utilities for all packages.
Import: `import { x } from '@project/shared'`

## Adding New Utils
1. Create in src/
2. Export from src/index.ts
3. Add tests
```

---

## Cross-Platform Support

### When to Apply
- Team uses both Claude Code and Codex/OpenCode
- Want consistent guidance across tools

### Pattern: Shared + Platform-Specific

```
project/
├── CLAUDE.md           # Claude Code specific
├── AGENTS.md           # Codex/OpenCode specific
└── docs/
    └── shared-rules.md # Common content
```

**CLAUDE.md:**
```markdown
# Claude Code Config
@docs/shared-rules.md

## Claude-Specific
Use /compact when context is large.
Use # to add session notes.
```

**AGENTS.md:**
```markdown
# Codex Config
See docs/shared-rules.md for common guidelines.

## Codex-Specific
Prefer explicit file paths in responses.
```

**docs/shared-rules.md:**
```markdown
# Shared Development Rules

## Commands
- Build: `npm run build`
- Test: `npm test`

## Conventions
- TypeScript strict mode
- Prettier for formatting
```

---

## Implementation Checklist

When implementing improvements:

```
Pre-Implementation:
- [ ] Backup original: cp CLAUDE.md CLAUDE.md.backup

Structure Changes:
- [ ] Create docs/ directory if extracting content
- [ ] Create package CLAUDE.md files if splitting monorepo
- [ ] Move extracted content to new files

Content Updates:
- [ ] Add TOC if file > 100 lines
- [ ] Replace inline content with @imports
- [ ] Fix all identified anti-patterns
- [ ] Add alternatives to negative rules

Validation:
- [ ] All @imports resolve to existing files
- [ ] File size reduced to target
- [ ] Commands still accurate
- [ ] Structure is navigable

Post-Implementation:
- [ ] Re-run grading to verify improvement
- [ ] Show before/after comparison
- [ ] Commit changes with descriptive message
```
