# Decision Rules Reference

IF-THEN rules for all HubSpot CRM operations. Check before every API call.

---

## Record Creation Rules

### Contact Creation
```
IF creating new contact:
  1. Search for existing by email AND name first
  2. Include lead_type field ("Outbound" or "Inbound")
  3. Set hubspot_owner_id to 699257003
  4. Associate to company immediately after creation
  5. Use v3 PUT endpoint for association (NOT v4)
```

### Company Creation
```
IF creating new company:
  1. Search for existing by domain/name first
  2. Set hubspot_owner_id to 699257003
  3. ALWAYS create contact(s) too - goal tracker counts contacts
```

### Lead Creation (Dec 2025+)
```
IF creating Lead object:
  1. MUST include PRIMARY associations inline (578 + 580)
  2. Use hs_lead_type = "NEW_BUSINESS" or "UPSELL"
  3. Use hs_lead_label = "COLD", "WARM", or "HOT" (uppercase)
  4. Use attempting-stage-id for new outreach leads
  5. Cannot create then associate - will fail validation
```

### Deal Creation
```
IF creating deal:
  1. Only create AFTER discovery call scheduled
  2. Set pipeline to 8bd9336b-4767-4e67-9fe2-35dfcad7c8be
  3. Set dealstage to appropriate stage ID
  4. Set hubspot_owner_id to 699257003
  5. Associate to contact and company
```

### Task Creation
```
IF creating task:
  1. Use hs_timestamp for due date (NOT hs_task_due_date)
  2. Set hs_task_status = "NOT_STARTED"
  3. Set hs_task_type = "TODO"
  4. Associate to deal using type 216
  5. Skip weekends when calculating due dates
```

### Note Creation
```
IF creating note:
  1. Use hs_timestamp for creation time (Unix ms)
  2. Use hs_note_body for content (HTML allowed)
  3. Associate to deal using type 214
  4. Include context: subject, recipients, summary
```

---

## API Call Rules

### Fetching Deals
```
IF fetching deals:
  ALWAYS include filters:
    - hubspot_owner_id = "699257003"
    - pipeline = "8bd9336b-4767-4e67-9fe2-35dfcad7c8be"
  NEVER fetch without filters (pulls 3,000+ deals)
```

### Setting Dates
```
IF setting any date field:
  1. Convert to Unix milliseconds
  2. Convert to STRING (not integer)
  3. Example: str(int(datetime.timestamp() * 1000))

  WRONG: "2025-01-31" (ISO string)
  WRONG: 1738281600 (Unix seconds)
  WRONG: 1738281600000 (integer)
  CORRECT: "1738281600000" (string)
```

### Batch Operations
```
IF doing bulk create:
  Use POST /crm/v3/objects/{type}/batch/create
  Max 100 records per batch

IF doing bulk archive:
  Use POST /crm/v3/objects/{type}/batch/archive
  Max 100 records per batch
```

### Rate Limiting
```
IF receiving 429 response:
  1. Check Retry-After header
  2. Wait specified seconds
  3. Retry with exponential backoff (2^attempt)
  4. Max 3 retry attempts

IF approaching limits:
  Burst: 100 requests per 10 seconds
  Daily: 150,000 requests per 24 hours
```

---

## Field Update Rules

### Read-Only Fields
```
IF need to set notes_next_activity_date:
  CANNOT set directly - create task instead
  HubSpot auto-calculates from task due dates
```

### Lead Type Fields
```
IF setting hs_lead_type:
  Use: "NEW_BUSINESS" or "UPSELL"
  NOT: "Outbound" or "Inbound"

IF setting hs_lead_label:
  Use: "COLD", "WARM", or "HOT"
  MUST be uppercase

IF setting lead_type (text field on Contact):
  Can use any string: "Apollo.io", "Web Research", etc.
  Appears in Account Summary

IF setting lead_type__c (enum on Contact):
  Use: "Inbound", "Outbound", or "Existing Client"
  Appears as "Lead Type"
```

### Task Date Fields
```
IF setting task due date:
  Use: hs_timestamp
  NOT: hs_task_due_date (doesn't exist)

IF counting completed tasks:
  Use: hs_task_completion_date
  NOT: hs_timestamp (that's creation date)
```

---

## Association Rules

### Standard Associations
```
IF associating Contact to Company:
  Type ID: 279
  Endpoint: PUT /crm/v3/objects/contacts/{id}/associations/companies/{id}/contact_to_company

IF associating Deal to Company:
  Type ID: 341

IF associating Deal to Contact:
  Type ID: 3

IF associating Note to Deal:
  Type ID: 214

IF associating Task to Deal:
  Type ID: 216
```

### Lead Associations
```
IF associating Lead to Contact (PRIMARY):
  Type ID: 578
  MUST be inline at creation

IF associating Lead to Company (PRIMARY):
  Type ID: 580
  MUST be inline at creation
```

---

## Stage Transition Rules

### Moving Deals
```
IF updating dealstage:
  Use stage ID (not stage name)
  Log note explaining why (especially for Closed Lost)

IF moving to Closed Lost:
  ALWAYS add explanatory note
  Set closed lost reason if available
```

### Stage Entry Tracking
```
IF counting deals that ENTERED stage:
  Use: hs_v2_date_entered_{stageid}_{pipelineid}
  NOT: current dealstage (shows stale data)
```

---

## Error Prevention Rules

### Windows Scripts
```
IF script outputs emoji/unicode:
  Add at top: sys.stdout = io.TextIOWrapper(sys.stdout.buffer, encoding='utf-8')

IF using subprocess:
  Use: encoding='utf-8'
  NOT: text=True (uses cp1252 on Windows)
```

### Division Safety
```
IF calculating percentage:
  ALWAYS: pct = (x / y * 100) if y > 0 else 0
  NEVER: pct = x / y * 100 (crashes on zero)
```

### Credential Safety
```
IF using API key:
  ALWAYS: load from .env file
  NEVER: hardcode in script

IF in heredoc Python:
  Read .env file directly (load_dotenv() fails in heredoc)
```

---

## Validation Checklist

### Before Any API Call
- [ ] Credentials loaded from .env?
- [ ] Rate limiter in place?
- [ ] Filters include Owner ID + Pipeline ID?
- [ ] Timestamps are Unix ms as STRING?
- [ ] Division denominators checked?

### Before Creating Records
- [ ] Searched for existing first?
- [ ] Owner ID set?
- [ ] Required associations included?
- [ ] For Leads: PRIMARY associations inline?

### Before Updating Deals
- [ ] Deal ID verified (fetched, not assumed)?
- [ ] Not setting read-only fields directly?
- [ ] Note added if moving to Closed Lost?
