# API Patterns Reference

Core HubSpot API patterns for rate limiting, timestamps, headers, and batch operations.

---

## Authentication

### Standard Headers
```python
import os
from dotenv import load_dotenv

load_dotenv()
API_KEY = os.environ.get('HUBSPOT_API_KEY')

headers = {
    'Authorization': f'Bearer {API_KEY}',
    'Content-Type': 'application/json'
}
```

### Direct .env Read (for heredoc scripts)
```python
# load_dotenv() fails in heredoc - read directly
env_path = Path.home() / '.env'
with open(env_path, 'r', encoding='utf-8', errors='ignore') as f:
    for line in f:
        if line.startswith('HUBSPOT_API_KEY='):
            API_KEY = line.split('=', 1)[1].strip().strip('"').strip("'")
            break
```

---

## Rate Limiting

### Limits
| Type | Limit | Window |
|------|-------|--------|
| Burst | 100 requests | 10 seconds |
| Daily | 150,000 requests | 24 hours |

### Token Bucket Pattern
```python
from hubspot_sync_core import HubSpotSyncManager

# Initialize with built-in rate limiting
sync = HubSpotSyncManager()

# Rate limiter handles waiting automatically
deals = sync.fetch_deals()

# Check current utilization
stats = sync.get_rate_limit_stats()
print(f"Burst: {stats['burst_utilization']:.1%}")
print(f"Daily: {stats['daily_utilization']:.1%}")
```

### Manual Rate Limiting
```python
import time

class SimpleRateLimiter:
    def __init__(self, requests_per_second=10):
        self.min_interval = 1.0 / requests_per_second
        self.last_request = 0

    def wait(self):
        elapsed = time.time() - self.last_request
        if elapsed < self.min_interval:
            time.sleep(self.min_interval - elapsed)
        self.last_request = time.time()
```

### 429 Error Handling
```python
import time

def api_call_with_retry(url, headers, payload, max_retries=3):
    for attempt in range(max_retries):
        response = requests.post(url, headers=headers, json=payload)

        if response.status_code == 429:
            retry_after = int(response.headers.get('Retry-After', 10))
            wait_time = retry_after * (2 ** attempt)  # Exponential backoff
            time.sleep(wait_time)
            continue

        return response

    raise Exception("Max retries exceeded")
```

---

## Timestamp Formatting

### Converting to HubSpot Format
```python
from datetime import datetime

# Current time
now_ms = str(int(datetime.now().timestamp() * 1000))

# Specific date
target = datetime(2025, 1, 31)
target_ms = str(int(target.timestamp() * 1000))

# With timezone
from datetime import timezone
utc_now = datetime.now(timezone.utc)
utc_ms = str(int(utc_now.timestamp() * 1000))
```

### Skip Weekends for Due Dates
```python
from datetime import datetime, timedelta

def business_date(days_out):
    """Calculate due date, skipping weekends"""
    target = datetime.now() + timedelta(days=days_out)
    while target.weekday() >= 5:  # 5=Saturday, 6=Sunday
        target += timedelta(days=1)
    return str(int(target.timestamp() * 1000))
```

### Common Timestamps
```python
# Today midnight
today = datetime.now().replace(hour=0, minute=0, second=0, microsecond=0)
today_ms = str(int(today.timestamp() * 1000))

# 7 days from now
week_out = datetime.now() + timedelta(days=7)
week_ms = str(int(week_out.timestamp() * 1000))

# Start of current week (Monday)
monday = datetime.now() - timedelta(days=datetime.now().weekday())
monday = monday.replace(hour=0, minute=0, second=0, microsecond=0)
monday_ms = str(int(monday.timestamp() * 1000))
```

---

## Search Endpoint Pattern

### Basic Search
```python
url = "https://api.hubapi.com/crm/v3/objects/deals/search"

payload = {
    "filterGroups": [
        {
            "filters": [
                {
                    "propertyName": "hubspot_owner_id",
                    "operator": "EQ",
                    "value": "699257003"
                },
                {
                    "propertyName": "pipeline",
                    "operator": "EQ",
                    "value": "8bd9336b-4767-4e67-9fe2-35dfcad7c8be"
                }
            ]
        }
    ],
    "properties": ["dealname", "amount", "dealstage", "closedate"],
    "limit": 100
}

response = requests.post(url, headers=headers, json=payload)
deals = response.json().get('results', [])
```

### With Date Filtering
```python
# Deals created in last 7 days
week_ago_ms = str(int((datetime.now() - timedelta(days=7)).timestamp() * 1000))

payload = {
    "filterGroups": [
        {
            "filters": [
                {"propertyName": "hubspot_owner_id", "operator": "EQ", "value": "699257003"},
                {"propertyName": "pipeline", "operator": "EQ", "value": "8bd9336b-4767-4e67-9fe2-35dfcad7c8be"},
                {"propertyName": "createdate", "operator": "GTE", "value": week_ago_ms}
            ]
        }
    ],
    "properties": ["dealname", "createdate"],
    "limit": 100
}
```

### Pagination
```python
def fetch_all_deals():
    all_deals = []
    after = None

    while True:
        payload = {
            "filterGroups": [...],
            "properties": [...],
            "limit": 100
        }
        if after:
            payload["after"] = after

        response = requests.post(url, headers=headers, json=payload)
        data = response.json()

        all_deals.extend(data.get('results', []))

        paging = data.get('paging', {})
        if 'next' in paging:
            after = paging['next']['after']
        else:
            break

    return all_deals
```

---

## Batch Operations

### Batch Create
```python
url = "https://api.hubapi.com/crm/v3/objects/contacts/batch/create"

payload = {
    "inputs": [
        {"properties": {"email": "john@acme.com", "firstname": "John"}},
        {"properties": {"email": "jane@acme.com", "firstname": "Jane"}},
        # Max 100 per batch
    ]
}

response = requests.post(url, headers=headers, json=payload)
```

### Batch Archive
```python
url = "https://api.hubapi.com/crm/v3/objects/tasks/batch/archive"

payload = {
    "inputs": [
        {"id": "12345"},
        {"id": "12346"},
        # Max 100 per batch
    ]
}

response = requests.post(url, headers=headers, json=payload)
```

### Batch Update
```python
url = "https://api.hubapi.com/crm/v3/objects/deals/batch/update"

payload = {
    "inputs": [
        {"id": "12345", "properties": {"amount": "50000"}},
        {"id": "12346", "properties": {"amount": "75000"}},
    ]
}

response = requests.post(url, headers=headers, json=payload)
```

---

## Task Audit Trail Pattern

**Purpose**: Create full audit trail in HubSpot timeline by creating task then marking complete.

**Flow**: Agenda Item → Complete work → Create task (NOT_STARTED) → Mark COMPLETED → Both events in timeline

**Why This Matters**:
- HubSpot timeline shows: "Task created" + "Task completed"
- Full audit trail for compliance and visibility
- Proves work was done with timestamps

### Create and Complete Task for Audit Trail
```python
import time
import requests

def create_completed_task(deal_id, subject, body):
    """
    Create task and immediately mark complete for audit trail.
    Results in TWO timeline events: created + completed.
    """

    now_ms = str(int(time.time() * 1000))

    # Step 1: Create task with NOT_STARTED status
    create_payload = {
        "properties": {
            "hs_task_subject": subject,
            "hs_task_body": body,
            "hs_task_status": "NOT_STARTED",
            "hs_task_priority": "MEDIUM",
            "hs_task_type": "TODO",
            "hs_timestamp": now_ms,
            "hubspot_owner_id": "699257003"
        },
        "associations": [
            {
                "to": {"id": deal_id},
                "types": [{"associationCategory": "HUBSPOT_DEFINED", "associationTypeId": 216}]
            }
        ]
    }

    response = requests.post(
        "https://api.hubapi.com/crm/v3/objects/tasks",
        headers=headers,
        json=create_payload
    )

    if response.status_code != 201:
        return None

    task_id = response.json()['id']

    # Step 2: Mark task COMPLETED
    complete_payload = {
        "properties": {
            "hs_task_status": "COMPLETED",
            "hs_task_completion_date": now_ms
        }
    }

    requests.patch(
        f"https://api.hubapi.com/crm/v3/objects/tasks/{task_id}",
        headers=headers,
        json=complete_payload
    )

    return task_id

# Usage
task_id = create_completed_task(
    deal_id="12345678",
    subject="Discovery call completed",
    body="Covered all 38 Questions. Scored 4/5 on Volume Reality."
)
```

### What Appears in HubSpot Timeline
1. **"Task created"** - Shows task was assigned
2. **"Task completed"** - Shows task was finished

Both events have timestamps for full audit compliance.

---

## Single Record Operations

### Get Single Deal
```python
deal_id = "12345678"
url = f"https://api.hubapi.com/crm/v3/objects/deals/{deal_id}"
params = {"properties": "dealname,amount,dealstage,closedate"}

response = requests.get(url, headers=headers, params=params)
deal = response.json()
```

### Update Single Deal
```python
deal_id = "12345678"
url = f"https://api.hubapi.com/crm/v3/objects/deals/{deal_id}"

payload = {
    "properties": {
        "amount": "100000",
        "hs_next_step": "Schedule implementation call"
    }
}

response = requests.patch(url, headers=headers, json=payload)
```

### Create with Associations
```python
url = "https://api.hubapi.com/crm/v3/objects/deals"

payload = {
    "properties": {
        "dealname": "Acme Corp - Xparcel",
        "amount": "50000",
        "dealstage": "1090865183",
        "pipeline": "8bd9336b-4767-4e67-9fe2-35dfcad7c8be",
        "hubspot_owner_id": "699257003"
    },
    "associations": [
        {
            "to": {"id": company_id},
            "types": [{"associationCategory": "HUBSPOT_DEFINED", "associationTypeId": 341}]
        },
        {
            "to": {"id": contact_id},
            "types": [{"associationCategory": "HUBSPOT_DEFINED", "associationTypeId": 3}]
        }
    ]
}

response = requests.post(url, headers=headers, json=payload)
```

---

## Custom Properties

### Get All Properties
```python
url = "https://api.hubapi.com/crm/v3/properties/deals"
response = requests.get(url, headers=headers)

for prop in response.json().get('results', []):
    print(f"{prop['name']} | {prop['label']} | {prop['type']}")
```

### Common Custom Fields
| API Name | Label | Type |
|----------|-------|------|
| `monthly_volume__c` | Monthly Volume | number |
| `returns_per_month_volume__c` | Returns per Month | number |
| Custom fields end with `__c` | | |

---

## Windows Console Fix

**Required for any script with emoji/unicode:**
```python
import sys
import io

# Add at very top of script
sys.stdout = io.TextIOWrapper(sys.stdout.buffer, encoding='utf-8')
```

**For subprocess calls:**
```python
result = subprocess.run(
    ['python', 'script.py'],
    capture_output=True,
    encoding='utf-8',  # NOT text=True
    timeout=30
)
```
