# Association Rules Reference

Complete reference for HubSpot association type IDs and payloads.

---

## Association Type IDs

### Standard Object Associations

| From | To | Type ID | Direction | Usage |
|------|-----|---------|-----------|-------|
| Contact | Company | 279 | Contact → Company | Creating contacts in companies |
| Company | Contact | 280 | Company → Contact | Reverse lookup |
| Deal | Company | 341 | Deal → Company | Linking deals to companies |
| Company | Deal | 342 | Company → Deal | Reverse lookup |
| Deal | Contact | 3 | Deal → Contact | Linking deals to contacts |
| Contact | Deal | 4 | Contact → Deal | Reverse lookup |

### Activity Associations

| From | To | Type ID | Usage |
|------|-----|---------|-------|
| Note | Deal | 214 | Activity logging |
| Task | Deal | 216 | Follow-up tasks |
| Note | Contact | 10 | Contact-level notes |
| Task | Contact | 204 | Contact-level tasks |

### Lead Object Associations (Dec 2025+)

| From | To | Type ID | Requirement |
|------|-----|---------|-------------|
| Lead | Contact | **578** | PRIMARY - **Required at creation** |
| Lead | Company | **580** | PRIMARY - **Required at creation** |
| Lead | Contact | 608 | Secondary - Optional |
| Lead | Company | 610 | Secondary - Optional |

---

## Association Payload Structure

### Standard Association (Inline at Creation)
```python
{
    "properties": {...},
    "associations": [
        {
            "to": {"id": "12345"},
            "types": [
                {
                    "associationCategory": "HUBSPOT_DEFINED",
                    "associationTypeId": 341  # Type ID here
                }
            ]
        }
    ]
}
```

### Multiple Associations
```python
{
    "properties": {...},
    "associations": [
        {
            "to": {"id": company_id},
            "types": [{"associationCategory": "HUBSPOT_DEFINED", "associationTypeId": 341}]
        },
        {
            "to": {"id": contact_id},
            "types": [{"associationCategory": "HUBSPOT_DEFINED", "associationTypeId": 3}]
        }
    ]
}
```

---

## Contact → Company Association

### Using v3 PUT Endpoint (Recommended)
```python
contact_id = "12345"
company_id = "67890"

url = f"https://api.hubapi.com/crm/v3/objects/contacts/{contact_id}/associations/companies/{company_id}/contact_to_company"

response = requests.put(url, headers=headers)
# Response: 200 OK with empty body
```

### Association at Creation
```python
url = "https://api.hubapi.com/crm/v3/objects/contacts"

payload = {
    "properties": {
        "email": "john@acme.com",
        "firstname": "John",
        "lastname": "Smith"
    },
    "associations": [
        {
            "to": {"id": company_id},
            "types": [{"associationCategory": "HUBSPOT_DEFINED", "associationTypeId": 279}]
        }
    ]
}
```

---

## Deal Associations

### Deal → Company + Contact at Creation
```python
url = "https://api.hubapi.com/crm/v3/objects/deals"

payload = {
    "properties": {
        "dealname": "Acme Corp - Xparcel Ground",
        "amount": "100000",
        "dealstage": "1090865183",
        "pipeline": "8bd9336b-4767-4e67-9fe2-35dfcad7c8be",
        "hubspot_owner_id": "699257003"
    },
    "associations": [
        {
            "to": {"id": company_id},
            "types": [{"associationCategory": "HUBSPOT_DEFINED", "associationTypeId": 341}]
        },
        {
            "to": {"id": contact_id},
            "types": [{"associationCategory": "HUBSPOT_DEFINED", "associationTypeId": 3}]
        }
    ]
}
```

### Add Association to Existing Deal
```python
deal_id = "12345"
contact_id = "67890"

url = f"https://api.hubapi.com/crm/v3/objects/deals/{deal_id}/associations/contacts/{contact_id}/deal_to_contact"

response = requests.put(url, headers=headers)
```

---

## Note → Deal Association

### Create Note with Deal Association
```python
import time

url = "https://api.hubapi.com/crm/v3/objects/notes"

payload = {
    "properties": {
        "hs_timestamp": str(int(time.time() * 1000)),
        "hs_note_body": "<p>Meeting scheduled for Friday to discuss proposal.</p>",
        "hubspot_owner_id": "699257003"
    },
    "associations": [
        {
            "to": {"id": deal_id},
            "types": [{"associationCategory": "HUBSPOT_DEFINED", "associationTypeId": 214}]
        }
    ]
}

response = requests.post(url, headers=headers, json=payload)
```

---

## Task → Deal Association

### Create Task with Deal Association
```python
from datetime import datetime, timedelta

# Due date 3 business days out
due_date = datetime.now() + timedelta(days=3)
while due_date.weekday() >= 5:
    due_date += timedelta(days=1)
due_ms = str(int(due_date.timestamp() * 1000))

url = "https://api.hubapi.com/crm/v3/objects/tasks"

payload = {
    "properties": {
        "hs_task_subject": "Follow up on proposal",
        "hs_task_body": "Check if they reviewed the proposal",
        "hs_task_status": "NOT_STARTED",
        "hs_task_priority": "MEDIUM",
        "hs_task_type": "TODO",
        "hs_timestamp": due_ms,  # Due date - NOT hs_task_due_date
        "hubspot_owner_id": "699257003"
    },
    "associations": [
        {
            "to": {"id": deal_id},
            "types": [{"associationCategory": "HUBSPOT_DEFINED", "associationTypeId": 216}]
        }
    ]
}

response = requests.post(url, headers=headers, json=payload)
```

---

## Lead Object Associations (CRITICAL)

### Lead Creation with PRIMARY Associations
```python
# Lead REQUIRES inline PRIMARY associations - CANNOT create then associate

url = "https://api.hubapi.com/crm/v3/objects/leads"

payload = {
    "properties": {
        "hs_lead_type": "NEW_BUSINESS",  # or "UPSELL"
        "hs_lead_label": "COLD",          # "COLD", "WARM", "HOT" (uppercase)
        "hs_lead_status": "NEW",
        "hubspot_owner_id": "699257003"
    },
    "associations": [
        {
            "to": {"id": contact_id},
            "types": [
                {
                    "associationCategory": "HUBSPOT_DEFINED",
                    "associationTypeId": 578  # PRIMARY - Required
                }
            ]
        },
        {
            "to": {"id": company_id},
            "types": [
                {
                    "associationCategory": "HUBSPOT_DEFINED",
                    "associationTypeId": 580  # PRIMARY - Required
                }
            ]
        }
    ]
}

response = requests.post(url, headers=headers, json=payload)
```

### Common Lead Creation Error
```python
# WRONG - This will FAIL
lead = create_lead(properties)
associate_lead_to_contact(lead_id, contact_id)  # TOO LATE!

# CORRECT - Associations inline
lead = create_lead(properties, associations=[contact_578, company_580])
```

---

## Batch Associations

### Create Multiple Associations
```python
url = "https://api.hubapi.com/crm/v4/associations/deals/contacts/batch/create"

payload = {
    "inputs": [
        {
            "from": {"id": deal_id_1},
            "to": {"id": contact_id_1},
            "types": [{"associationCategory": "HUBSPOT_DEFINED", "associationTypeId": 3}]
        },
        {
            "from": {"id": deal_id_2},
            "to": {"id": contact_id_2},
            "types": [{"associationCategory": "HUBSPOT_DEFINED", "associationTypeId": 3}]
        }
    ]
}

response = requests.post(url, headers=headers, json=payload)
```

---

## Get Existing Associations

### Get Deal's Contacts
```python
deal_id = "12345"
url = f"https://api.hubapi.com/crm/v3/objects/deals/{deal_id}/associations/contacts"

response = requests.get(url, headers=headers)
contacts = response.json().get('results', [])
```

### Get Deal's Company
```python
deal_id = "12345"
url = f"https://api.hubapi.com/crm/v3/objects/deals/{deal_id}/associations/companies"

response = requests.get(url, headers=headers)
companies = response.json().get('results', [])
```

---

## Quick Reference Table

| Operation | Type ID | Endpoint Pattern |
|-----------|---------|------------------|
| Link Contact to Company | 279 | PUT /contacts/{id}/associations/companies/{id}/contact_to_company |
| Link Deal to Company | 341 | PUT /deals/{id}/associations/companies/{id}/deal_to_company |
| Link Deal to Contact | 3 | PUT /deals/{id}/associations/contacts/{id}/deal_to_contact |
| Link Note to Deal | 214 | Include in note creation payload |
| Link Task to Deal | 216 | Include in task creation payload |
| Link Lead to Contact | 578 | Must be inline at Lead creation |
| Link Lead to Company | 580 | Must be inline at Lead creation |
