# Stage Properties Reference

Pipeline stage IDs, entry properties, and stage mapping.

---

## FM Pipeline Stages

### Stage ID Reference

| Stage | Name | Stage ID | Win Probability |
|-------|------|----------|-----------------|
| [01] | Discovery Scheduled | `1090865183` | 20% |
| [02] | Discovery Complete | `d2a08d6f-cc04-4423-9215-594fe682e538` | 20% |
| [03] | Rate Creation | `e1c4321e-afb6-4b29-97d4-2b2425488535` | 40% |
| [04] | Proposal Sent | `d607df25-2c6d-4a5d-9835-6ed1e4f4020a` | 60% |
| [05] | Setup Docs Sent | `4e549d01-674b-4b31-8a90-91ec03122715` | 80% |
| [06] | Implementation | `08d9c411-5e1b-487b-8732-9c2bcbbd0307` | 90% |
| [07] | Started Shipping | `3fd46d94-78b4-452b-8704-62a338a210fb` | 100% |
| [08] | Closed Lost | `02d8a1d7-d0b3-41d9-adc6-44ab768a61b8` | 0% |

### Pipeline ID
```
8bd9336b-4767-4e67-9fe2-35dfcad7c8be
```

---

## Stage Entry Properties

### Property Pattern
```
hs_v2_date_entered_{stageid}_{pipelineid}
```

### Full Property Names

| Stage | Entry Property Name |
|-------|---------------------|
| [01] Discovery Scheduled | `hs_v2_date_entered_1090865183_8bd9336b-4767-4e67-9fe2-35dfcad7c8be` |
| [02] Discovery Complete | `hs_v2_date_entered_d2a08d6f_cc04_4423_9215_594fe682e538_8bd9336b-4767-4e67-9fe2-35dfcad7c8be` |
| [03] Rate Creation | `hs_v2_date_entered_e1c4321e_afb6_4b29_97d4_2b2425488535_8bd9336b-4767-4e67-9fe2-35dfcad7c8be` |
| [04] Proposal Sent | `hs_v2_date_entered_d607df25_2c6d_4a5d_9835_6ed1e4f4020a_8bd9336b-4767-4e67-9fe2-35dfcad7c8be` |
| [05] Setup Docs Sent | `hs_v2_date_entered_4e549d01_674b_4b31_8a90_91ec03122715_8bd9336b-4767-4e67-9fe2-35dfcad7c8be` |
| [06] Implementation | `hs_v2_date_entered_08d9c411_5e1b_487b_8732_9c2bcbbd0307_8bd9336b-4767-4e67-9fe2-35dfcad7c8be` |
| [07] Started Shipping | `hs_v2_date_entered_3fd46d94_78b4_452b_8704_62a338a210fb_8bd9336b-4767-4e67-9fe2-35dfcad7c8be` |
| [08] Closed Lost | `hs_v2_date_entered_02d8a1d7_d0b3_41d9_adc6_44ab768a61b8_8bd9336b-4767-4e67-9fe2-35dfcad7c8be` |

---

## Stage Mapping Dictionaries

### Python: Stage ID → Folder Name
```python
STAGE_MAP = {
    '1090865183': '[01-DISCOVERY-SCHEDULED]',
    'd2a08d6f-cc04-4423-9215-594fe682e538': '[02-DISCOVERY-COMPLETE]',
    'e1c4321e-afb6-4b29-97d4-2b2425488535': '[03-RATE-CREATION]',
    'd607df25-2c6d-4a5d-9835-6ed1e4f4020a': '[04-PROPOSAL-SENT]',
    '4e549d01-674b-4b31-8a90-91ec03122715': '[05-SETUP-DOCS-SENT]',
    '08d9c411-5e1b-487b-8732-9c2bcbbd0307': '[06-IMPLEMENTATION]',
    '3fd46d94-78b4-452b-8704-62a338a210fb': '[07-STARTED-SHIPPING]',
    '02d8a1d7-d0b3-41d9-adc6-44ab768a61b8': '[08-CLOSED-LOST]'
}
```

### Python: Folder Name → Stage ID
```python
FOLDER_TO_STAGE_ID = {
    '[01-DISCOVERY-SCHEDULED]': '1090865183',
    '[02-DISCOVERY-COMPLETE]': 'd2a08d6f-cc04-4423-9215-594fe682e538',
    '[03-RATE-CREATION]': 'e1c4321e-afb6-4b29-97d4-2b2425488535',
    '[04-PROPOSAL-SENT]': 'd607df25-2c6d-4a5d-9835-6ed1e4f4020a',
    '[05-SETUP-DOCS-SENT]': '4e549d01-674b-4b31-8a90-91ec03122715',
    '[06-IMPLEMENTATION]': '08d9c411-5e1b-487b-8732-9c2bcbbd0307',
    '[07-STARTED-SHIPPING]': '3fd46d94-78b4-452b-8704-62a338a210fb',
    '[08-CLOSED-LOST]': '02d8a1d7-d0b3-41d9-adc6-44ab768a61b8'
}
```

---

## Counting Deals by Stage Entry

### IMPORTANT: Entry vs Current Stage

**WRONG** - Counting deals IN a stage (shows stale data):
```python
# This counts current stage, NOT when they entered
payload = {
    "filterGroups": [{
        "filters": [{"propertyName": "dealstage", "operator": "EQ", "value": stage_id}]
    }]
}
```

**CORRECT** - Counting deals that ENTERED a stage in date range:
```python
from datetime import datetime, timedelta

# Start of week
week_start = datetime.now() - timedelta(days=datetime.now().weekday())
week_start = week_start.replace(hour=0, minute=0, second=0, microsecond=0)
week_start_ms = str(int(week_start.timestamp() * 1000))

# Entry property for stage 3
entry_prop = "hs_v2_date_entered_e1c4321e_afb6_4b29_97d4_2b2425488535_8bd9336b-4767-4e67-9fe2-35dfcad7c8be"

payload = {
    "filterGroups": [{
        "filters": [
            {"propertyName": "hubspot_owner_id", "operator": "EQ", "value": "699257003"},
            {"propertyName": "pipeline", "operator": "EQ", "value": "8bd9336b-4767-4e67-9fe2-35dfcad7c8be"},
            {"propertyName": entry_prop, "operator": "GTE", "value": week_start_ms}
        ]
    }],
    "properties": ["dealname", entry_prop]
}
```

---

## Lead Pipeline Stages

### Lead Stage IDs (String Format)
```yaml
new-stage-id          # New lead
attempting-stage-id   # Outreach in progress (default for new outreach)
connected-stage-id    # Made contact
qualified-stage-id    # Qualified opportunity
unqualified-stage-id  # Not a fit
```

### Setting Lead Stage
```python
payload = {
    "properties": {
        "hs_lead_status": "NEW",
        "hs_pipeline_stage": "attempting-stage-id"  # Use string ID
    }
}
```

---

## Stage Transition Patterns

### Move Deal to Next Stage
```python
def move_deal_to_stage(deal_id, new_stage_id, note=None):
    """Move deal to new stage with optional note"""

    # Update deal stage
    response = requests.patch(
        f"https://api.hubapi.com/crm/v3/objects/deals/{deal_id}",
        headers=headers,
        json={"properties": {"dealstage": new_stage_id}}
    )

    # Add explanatory note (especially important for Closed Lost)
    if note:
        requests.post(
            "https://api.hubapi.com/crm/v3/objects/notes",
            headers=headers,
            json={
                "properties": {
                    "hs_timestamp": str(int(time.time() * 1000)),
                    "hs_note_body": note,
                    "hubspot_owner_id": "699257003"
                },
                "associations": [
                    {"to": {"id": deal_id}, "types": [{"associationCategory": "HUBSPOT_DEFINED", "associationTypeId": 214}]}
                ]
            }
        )

    return response.status_code == 200
```

### Move to Closed Lost (ALWAYS add note)
```python
def close_deal_lost(deal_id, reason):
    """Close deal as lost with required note"""

    # ALWAYS add explanatory note when closing lost
    move_deal_to_stage(
        deal_id,
        "02d8a1d7-d0b3-41d9-adc6-44ab768a61b8",  # Closed Lost
        note=f"Closed Lost: {reason}"
    )
```

---

## Days in Stage Calculation

### Calculate Time in Current Stage
```python
from datetime import datetime

def days_in_stage(deal):
    """Calculate days deal has been in current stage"""
    props = deal.get('properties', {})
    stage_id = props.get('dealstage')

    # Get entry property for current stage
    entry_prop = f"hs_v2_date_entered_{stage_id.replace('-', '_')}_8bd9336b-4767-4e67-9fe2-35dfcad7c8be"
    entry_ms = props.get(entry_prop)

    if not entry_ms:
        return None

    entry_date = datetime.fromtimestamp(int(entry_ms) / 1000)
    days = (datetime.now() - entry_date).days
    return days
```

### Flag Stale Deals
```python
STALE_THRESHOLDS = {
    '1090865183': 7,   # Discovery Scheduled: 7 days
    'd2a08d6f-cc04-4423-9215-594fe682e538': 5,   # Discovery Complete: 5 days
    'e1c4321e-afb6-4b29-97d4-2b2425488535': 5,   # Rate Creation: 5 days
    'd607df25-2c6d-4a5d-9835-6ed1e4f4020a': 14,  # Proposal Sent: 14 days
}

def is_stale(deal):
    """Check if deal is stale based on stage thresholds"""
    stage_id = deal['properties'].get('dealstage')
    threshold = STALE_THRESHOLDS.get(stage_id)

    if not threshold:
        return False

    days = days_in_stage(deal)
    return days and days > threshold
```

---

## WIP Limits by Stage

| Stage | WIP Limit | Reason |
|-------|-----------|--------|
| [01] Discovery Scheduled | 20 | Plenty of room for outreach |
| [02] Discovery Complete | 15 | Need to move to rate creation |
| [03] Rate Creation | 6 | Active work required |
| [04] Proposal Sent | 4 | Follow-up intensive |
| [05] Setup Docs Sent | 3 | Close to finish line |
| [06] Implementation | 3 | White glove attention |
| [07] Started Shipping | 2 | Account management |

### Check WIP Limit
```python
WIP_LIMITS = {
    '1090865183': 20,
    'd2a08d6f-cc04-4423-9215-594fe682e538': 15,
    'e1c4321e-afb6-4b29-97d4-2b2425488535': 6,
    'd607df25-2c6d-4a5d-9835-6ed1e4f4020a': 4,
    '4e549d01-674b-4b31-8a90-91ec03122715': 3,
    '08d9c411-5e1b-487b-8732-9c2bcbbd0307': 3,
    '3fd46d94-78b4-452b-8704-62a338a210fb': 2,
}

def check_wip(stage_id, current_count):
    """Check if stage is at or over WIP limit"""
    limit = WIP_LIMITS.get(stage_id, 10)
    return current_count >= limit
```
