# Known Failures Reference

Documented HubSpot API failures with prevention patterns. Check before every API call.

---

## F001: Wrong Timestamp Format

**Date Discovered**: 2025-12-16

**What Happened**:
Tried to set closedate using ISO string `"2025-01-31"`. HubSpot rejected the value silently or threw validation error.

**Why It's Wrong**:
HubSpot requires Unix milliseconds as STRING, not ISO dates or Unix seconds.

**Symptoms**:
- Date fields not updating
- Validation errors on date fields
- Silent failures where data doesn't save

**Prevention**:
```python
# WRONG
properties["closedate"] = "2025-01-31"      # ISO string
properties["closedate"] = 1738281600        # Unix seconds
properties["closedate"] = 1738281600000     # Integer

# CORRECT
from datetime import datetime
timestamp = datetime(2025, 1, 31)
properties["closedate"] = str(int(timestamp.timestamp() * 1000))  # "1738281600000"
```

---

## F002: Setting Read-Only Field

**Date Discovered**: 2025-12-16

**What Happened**:
Tried to set `notes_next_activity_date` directly on a deal. Got error: "notes_next_activity_date is a read only property; its value cannot be set."

**Why It's Wrong**:
Some fields are calculated by HubSpot based on related objects. You can't set them directly.

**Symptoms**:
- API returns 400 error
- Error message mentions "read only property"

**Prevention**:
```python
# WRONG - Direct field setting
properties["notes_next_activity_date"] = "1738281600000"

# CORRECT - Create a task, HubSpot auto-calculates
task_payload = {
    "properties": {
        "hs_task_subject": "Follow up",
        "hs_timestamp": "1738281600000",  # Due date
        "hs_task_status": "NOT_STARTED"
    },
    "associations": [...]
}
requests.post(".../tasks", json=task_payload)
```

---

## F003: Wrong Task Due Date Property

**Date Discovered**: 2025-12-16

**What Happened**:
Used `hs_task_due_date` property for task due dates. Tasks failed to create because property doesn't exist.

**Why It's Wrong**:
The property is named `hs_timestamp`, not `hs_task_due_date`.

**Symptoms**:
- Task creation fails
- "Property does not exist" error

**Prevention**:
```python
# WRONG
properties["hs_task_due_date"] = "1738281600000"

# CORRECT
properties["hs_timestamp"] = "1738281600000"
```

---

## F004: Counting Tasks by Wrong Date

**Date Discovered**: 2025-12-16

**What Happened**:
Counted productivity by `hs_timestamp` (task due/creation date) instead of `hs_task_completion_date`. Metrics showed tasks created, not completed.

**Why It's Wrong**:
`hs_timestamp` is when task is DUE or created, not when it was completed.

**Symptoms**:
- Productivity metrics don't match actual work done
- Counting future tasks as completed

**Prevention**:
```python
# WRONG - Counts by creation/due date
payload = {"filterGroups": [{
    "filters": [{"propertyName": "hs_timestamp", "operator": "GTE", "value": week_start}]
}]}

# CORRECT - Counts by completion date
payload = {"filterGroups": [{
    "filters": [
        {"propertyName": "hs_task_completion_date", "operator": "GTE", "value": week_start},
        {"propertyName": "hs_task_status", "operator": "EQ", "value": "COMPLETED"}
    ]
}]}
```

---

## F005: Lead Creation Without Inline Associations

**Date Discovered**: 2025-12-16

**What Happened**:
Created Lead object, then tried to associate to Contact/Company in separate API calls. Lead creation failed with validation error.

**Why It's Wrong**:
Lead objects REQUIRE PRIMARY associations (578, 580) inline at creation time. Cannot create then associate.

**Symptoms**:
- Lead creation returns 400 error
- "Association required" validation error

**Prevention**:
```python
# WRONG - Separate association
lead = create_lead(properties)
associate_lead_to_contact(lead_id, contact_id)  # FAILS

# CORRECT - Inline associations
lead_payload = {
    "properties": {...},
    "associations": [
        {"to": {"id": contact_id}, "types": [{"associationCategory": "HUBSPOT_DEFINED", "associationTypeId": 578}]},
        {"to": {"id": company_id}, "types": [{"associationCategory": "HUBSPOT_DEFINED", "associationTypeId": 580}]}
    ]
}
requests.post(".../leads", json=lead_payload)
```

---

## F006: Wrong Lead Type Values

**Date Discovered**: 2025-12-16

**What Happened**:
Used "Outbound" for `hs_lead_type` field. HubSpot rejected the value.

**Why It's Wrong**:
`hs_lead_type` accepts "NEW_BUSINESS" or "UPSELL", not "Outbound" or "Inbound".

**Symptoms**:
- Lead creation fails
- Validation error on hs_lead_type

**Prevention**:
```python
# WRONG
properties["hs_lead_type"] = "Outbound"

# CORRECT
properties["hs_lead_type"] = "NEW_BUSINESS"  # or "UPSELL"

# For lead_type text field on Contact (different field!)
properties["lead_type"] = "Apollo.io"  # Any string OK
```

**Lead Label**:
```python
# Must be uppercase
properties["hs_lead_label"] = "COLD"   # or "WARM" or "HOT"
```

---

## F007: Division by Zero

**Date Discovered**: 2025-12-16

**What Happened**:
Calculated SLA percentage when denominator was 0. Script crashed with ZeroDivisionError.

**Why It's Wrong**:
Zero denominators are common when no deals in a stage or no completed tasks.

**Symptoms**:
- ZeroDivisionError crashes script
- Report generation fails

**Prevention**:
```python
# WRONG
percentage = completed / total * 100

# CORRECT
percentage = (completed / total * 100) if total > 0 else 0
```

---

## F008: Windows Console Encoding

**Date Discovered**: 2025-12-16

**What Happened**:
Script with emoji output (status indicators, summaries) crashed with UnicodeEncodeError on Windows.

**Why It's Wrong**:
Windows console uses cp1252 by default, which can't handle emoji/unicode.

**Symptoms**:
- UnicodeEncodeError on sys.stdout.write
- Crash when printing status emojis

**Prevention**:
```python
# Add at TOP of every script with emoji output
import sys
import io

sys.stdout = io.TextIOWrapper(sys.stdout.buffer, encoding='utf-8')
```

**For subprocess**:
```python
# WRONG
result = subprocess.run(cmd, text=True)  # Uses cp1252

# CORRECT
result = subprocess.run(cmd, encoding='utf-8')
```

---

## F009: Missing Owner/Pipeline Filter

**Date Discovered**: 2025-12-16

**What Happened**:
Fetched deals without filtering by Owner ID and Pipeline ID. Pulled 3,000+ deals from entire CRM.

**Why It's Wrong**:
Without filters, you get all deals across all pipelines and owners.

**Symptoms**:
- API call takes forever
- Returns way more deals than expected
- Memory issues from large response

**Prevention**:
```python
# WRONG - No filters
payload = {"properties": ["dealname"]}

# CORRECT - Always filter
payload = {
    "filterGroups": [{
        "filters": [
            {"propertyName": "hubspot_owner_id", "operator": "EQ", "value": "699257003"},
            {"propertyName": "pipeline", "operator": "EQ", "value": "8bd9336b-4767-4e67-9fe2-35dfcad7c8be"}
        ]
    }],
    "properties": ["dealname"]
}
```

---

## F010: Assuming Deal ID Format

**Date Discovered**: 2025-12-16

**What Happened**:
Used example deal IDs or assumed format. Update failed because deal didn't exist.

**Why It's Wrong**:
Deal IDs are assigned by HubSpot. You must fetch them first.

**Symptoms**:
- 404 Not Found on deal operations
- "Object not found" error

**Prevention**:
```python
# WRONG - Guessing ID
deal_id = "12345678"  # Where did this come from?

# CORRECT - Fetch first
deals = sync.fetch_deals()
deal_id = deals[0]['id']  # Use actual ID from API
```

---

## Pre-API-Call Checklist

### Before Any API Call
- [ ] API key loaded from .env (not hardcoded)?
- [ ] Using rate limiter or HubSpotSyncManager?
- [ ] Filters include Owner ID + Pipeline ID?
- [ ] Timestamps are Unix ms as STRING?
- [ ] Division denominators checked?
- [ ] Windows encoding fix at top of script?

### Before Creating Records
- [ ] Searched for existing first?
- [ ] Owner ID set?
- [ ] Required associations included?
- [ ] For Leads: PRIMARY associations (578, 580) inline?
- [ ] For Lead: hs_lead_type is NEW_BUSINESS or UPSELL?

### Before Updating Records
- [ ] Deal ID verified (fetched, not assumed)?
- [ ] Not setting read-only fields?
- [ ] Using hs_timestamp for task due dates?
- [ ] Note added if moving to Closed Lost?

---

## Adding New Failures

When a new HubSpot API failure is discovered:

1. Add entry here with date, description, prevention
2. Update expertise.yaml `learned_behaviors.failures`
3. Increment expertise.yaml version (patch level)
4. Add to relevant checklist above
5. Consider if RULES.md also needs update
