// 基本的なInngest関数の実装例
// Next.js App Router + TypeScript

// ===========================================
// 1. クライアント設定 (src/inngest/client.ts)
// ===========================================

import { EventSchemas, Inngest } from "inngest";

// イベント型定義
type Events = {
  "app/user.signup": {
    data: {
      userId: string;
      email: string;
      name: string;
    };
  };
  "app/order.completed": {
    data: {
      orderId: string;
      userId: string;
      total: number;
      items: Array<{ productId: string; quantity: number }>;
    };
  };
};

export const inngest = new Inngest({
  id: "my-app",
  schemas: new EventSchemas().fromRecord<Events>(),
});

// ===========================================
// 2. 関数定義 (src/inngest/functions/welcome.ts)
// ===========================================

export const sendWelcomeEmail = inngest.createFunction(
  {
    id: "send-welcome-email",
    retries: 3,
  },
  { event: "app/user.signup" },
  async ({ event, step, logger }) => {
    const { userId, email, name } = event.data;

    logger.info("Processing welcome email", { userId, email });

    // ステップ1: ユーザー情報を取得
    const user = await step.run("fetch-user-details", async () => {
      // 実際にはDBから取得
      return {
        id: userId,
        email,
        name,
        createdAt: new Date().toISOString(),
      };
    });

    // ステップ2: 1分待機（ユーザーが他のアクションを取る時間を与える）
    await step.sleep("wait-before-email", "1m");

    // ステップ3: ウェルカムメールを送信
    const emailResult = await step.run("send-email", async () => {
      // 実際にはメールサービスを呼び出す
      console.log("Sending welcome email to " + user.email);
      return {
        messageId: "msg_" + Date.now(),
        sentAt: new Date().toISOString(),
      };
    });

    // ステップ4: 分析イベントを送信
    await step.sendEvent("track-signup", {
      name: "analytics/user.welcomed" as any,
      data: {
        userId,
        emailMessageId: emailResult.messageId,
      },
    });

    return {
      success: true,
      emailMessageId: emailResult.messageId,
    };
  }
);

// ===========================================
// 3. APIルート (src/app/api/inngest/route.ts)
// ===========================================

import { serve } from "inngest/next";
// import { inngest } from "@/inngest/client";
// import { sendWelcomeEmail } from "@/inngest/functions/welcome";

export const { GET, POST, PUT } = serve({
  client: inngest,
  functions: [sendWelcomeEmail],
});

// ===========================================
// 4. イベント送信 (src/app/api/signup/route.ts)
// ===========================================

import { NextResponse } from "next/server";

export async function POST(request: Request) {
  const body = await request.json();

  // ユーザー作成ロジック（省略）
  const userId = "user_" + Date.now();

  // Inngestイベントを送信
  await inngest.send({
    name: "app/user.signup",
    data: {
      userId,
      email: body.email,
      name: body.name,
    },
  });

  return NextResponse.json({
    success: true,
    userId,
    message: "User created, welcome email will be sent shortly",
  });
}

// ===========================================
// 5. Cronジョブの例
// ===========================================

export const dailyCleanup = inngest.createFunction(
  { id: "daily-cleanup" },
  { cron: "0 0 * * *" }, // 毎日UTC 0:00
  async ({ step, logger }) => {
    logger.info("Starting daily cleanup");

    // 古いセッションを削除
    const deletedSessions = await step.run("delete-old-sessions", async () => {
      // 実際にはDBから削除
      return { count: 42 };
    });

    // 古い通知を削除
    const deletedNotifications = await step.run(
      "delete-old-notifications",
      async () => {
        return { count: 128 };
      }
    );

    return {
      deletedSessions: deletedSessions.count,
      deletedNotifications: deletedNotifications.count,
    };
  }
);
