// エラーハンドリングの実装例
// 堅牢なエラー処理とリトライ戦略

import { EventSchemas, Inngest, NonRetriableError, RetryAfterError } from "inngest";

// ===========================================
// 1. クライアント設定
// ===========================================

type Events = {
  "order/payment.process": {
    data: {
      orderId: string;
      userId: string;
      amount: number;
      paymentMethod: string;
    };
  };
  "user/data.sync": {
    data: {
      userId: string;
      source: string;
    };
  };
};

const inngest = new Inngest({
  id: "payment-service",
  schemas: new EventSchemas().fromRecord<Events>(),
});

// ===========================================
// 2. NonRetriableError - 再試行不要なエラー
// ===========================================

export const processPayment = inngest.createFunction(
  {
    id: "process-payment",
    retries: 5,
  },
  { event: "order/payment.process" },
  async ({ event, step, logger }) => {
    const { orderId, userId, amount, paymentMethod } = event.data;

    // ステップ1: 注文を検証
    const order = await step.run("validate-order", async () => {
      // 実際にはDBから取得
      const orderData = {
        id: orderId,
        status: "pending",
        userId,
        amount,
      };

      // 注文が見つからない場合は再試行しない
      if (!orderData) {
        throw new NonRetriableError("Order " + orderId + " not found", {
          cause: new Error("NOT_FOUND"),
        });
      }

      // すでに処理済みの場合は再試行しない
      if (orderData.status === "completed") {
        throw new NonRetriableError("Order " + orderId + " already processed", {
          cause: new Error("ALREADY_PROCESSED"),
        });
      }

      return orderData;
    });

    // ステップ2: ユーザーを検証
    const user = await step.run("validate-user", async () => {
      // 実際にはDBから取得
      const userData = {
        id: userId,
        email: "user@example.com",
        status: "active",
      };

      // ユーザーがBANされている場合は再試行しない
      if (userData.status === "banned") {
        throw new NonRetriableError("User " + userId + " is banned", {
          cause: new Error("USER_BANNED"),
        });
      }

      return userData;
    });

    // ステップ3: 支払いを処理
    const payment = await step.run("charge-payment", async () => {
      // 支払いゲートウェイを呼び出す
      const result = await processPaymentGateway(amount, paymentMethod);

      // カードが無効な場合は再試行しない
      if (result.error === "CARD_DECLINED") {
        throw new NonRetriableError("Card was declined", {
          cause: new Error(result.error),
        });
      }

      // カードの有効期限切れは再試行しない
      if (result.error === "CARD_EXPIRED") {
        throw new NonRetriableError("Card has expired", {
          cause: new Error(result.error),
        });
      }

      return result;
    });

    logger.info("Payment processed successfully", {
      orderId,
      paymentId: payment.id,
    });

    return { success: true, paymentId: payment.id };
  }
);

// ===========================================
// 3. RetryAfterError - 制御された再試行
// ===========================================

export const syncUserData = inngest.createFunction(
  {
    id: "sync-user-data",
    retries: 10,
  },
  { event: "user/data.sync" },
  async ({ event, step, logger }) => {
    const { userId, source } = event.data;

    // 外部APIを呼び出し（レート制限あり）
    const externalData = await step.run("fetch-external-data", async () => {
      const response = await fetchExternalAPI(userId, source);

      // レート制限に達した場合
      if (response.status === 429) {
        const retryAfter = response.headers["retry-after"] || "60";
        throw new RetryAfterError(
          "Rate limited by " + source,
          new Date(Date.now() + parseInt(retryAfter) * 1000)
        );
      }

      // サービス一時停止の場合
      if (response.status === 503) {
        throw new RetryAfterError(
          source + " is temporarily unavailable",
          "5m" // 5分後に再試行
        );
      }

      return response.data;
    });

    // データを保存
    await step.run("save-data", async () => {
      // 実際にはDBに保存
      console.log("Saving data for user " + userId, externalData);
    });

    return { success: true, userId };
  }
);

// ===========================================
// 4. onFailure - 失敗時の通知
// ===========================================

export const criticalJob = inngest.createFunction(
  {
    id: "critical-job",
    retries: 3,
    onFailure: async ({ error, event, step }) => {
      // Slackに通知
      await step.run("notify-slack", async () => {
        await sendSlackAlert({
          channel: "#alerts",
          text: `Critical job failed!`,
          blocks: [
            {
              type: "section",
              text: {
                type: "mrkdwn",
                text: "*Critical Job Failed*\n\nError: " + error.message + "\nEvent: " + event.name + "\nData: " + JSON.stringify(event.data),
              },
            },
          ],
        });
      });

      // PagerDutyにインシデントを作成
      await step.run("create-pagerduty-incident", async () => {
        await createPagerDutyIncident({
          title: "Critical job failed: " + error.message,
          severity: "high",
          details: {
            eventName: event.name,
            eventData: event.data,
            errorStack: error.stack,
          },
        });
      });

      // DBにエラーログを保存
      await step.run("log-error", async () => {
        // await db.errorLogs.create({ ... });
        console.error("Logged error to database", error);
      });
    },
  },
  { event: "order/payment.process" },
  async ({ event, step }) => {
    // 重要な処理
    await step.run("critical-operation", async () => {
      // 失敗する可能性のある処理
      throw new Error("Something went wrong!");
    });
  }
);

// ===========================================
// 5. try-catch パターン（ステップ内）
// ===========================================

export const robustDataProcessor = inngest.createFunction(
  { id: "robust-data-processor" },
  { event: "user/data.sync" },
  async ({ event, step, logger }) => {
    const { userId } = event.data;

    // オプショナルな処理（失敗しても続行）
    const enrichedData = await step.run("enrich-data", async () => {
      try {
        const extra = await fetchOptionalEnrichment(userId);
        return { success: true, data: extra };
      } catch (error) {
        // エラーをログに記録するが、処理は続行
        logger.warn("Enrichment failed, continuing without it", {
          error: (error as Error).message,
        });
        return { success: false, data: null };
      }
    });

    // 必須の処理
    const coreData = await step.run("fetch-core-data", async () => {
      return { userId, timestamp: Date.now() };
    });

    // 結果を結合
    await step.run("save-result", async () => {
      const finalData = {
        ...coreData,
        enriched: enrichedData.success ? enrichedData.data : null,
      };
      // await db.userData.upsert(finalData);
      console.log("Saved data", finalData);
    });

    return {
      success: true,
      enrichmentIncluded: enrichedData.success,
    };
  }
);

// ===========================================
// 6. 条件付きリトライ
// ===========================================

export const conditionalRetry = inngest.createFunction(
  {
    id: "conditional-retry",
    retries: 5,
  },
  { event: "user/data.sync" },
  async ({ event, step, attempt }) => {
    // 試行回数に基づいてロジックを変更
    const strategy = attempt < 3 ? "fast" : "slow";

    await step.run("process-with-strategy", async () => {
      if (strategy === "fast") {
        // 初期の試行では高速な方法を試す
        return await fastMethod();
      } else {
        // 後の試行ではより堅牢な方法を使用
        return await robustMethod();
      }
    });

    return { success: true, strategy, attempt };
  }
);

// ===========================================
// ヘルパー関数（モック）
// ===========================================

async function processPaymentGateway(
  amount: number,
  method: string
): Promise<{ id: string; error?: string }> {
  return { id: "pay_" + Date.now() };
}

async function fetchExternalAPI(
  userId: string,
  source: string
): Promise<{ status: number; headers: Record<string, string>; data: any }> {
  return { status: 200, headers: {}, data: { userId } };
}

async function sendSlackAlert(params: any): Promise<void> {
  console.log("Slack alert:", params);
}

async function createPagerDutyIncident(params: any): Promise<void> {
  console.log("PagerDuty incident:", params);
}

async function fetchOptionalEnrichment(userId: string): Promise<any> {
  return { extra: "data" };
}

async function fastMethod(): Promise<any> {
  return { method: "fast" };
}

async function robustMethod(): Promise<any> {
  return { method: "robust" };
}

// ===========================================
// エクスポート
// ===========================================

export const functions = [
  processPayment,
  syncUserData,
  criticalJob,
  robustDataProcessor,
  conditionalRetry,
];
