// Fan-outパターンの実装例
// 大量のユーザーにメールを送信するシナリオ

import { EventSchemas, Inngest } from "inngest";

// ===========================================
// 1. イベント型定義
// ===========================================

type Events = {
  // ローダーがトリガーするバッチジョブ開始イベント
  "email/weekly-digest.requested": {
    data: {
      campaignId: string;
      templateId: string;
    };
  };
  // Fan-outで各ユーザーに送信されるイベント
  "email/weekly-digest.send": {
    data: {
      campaignId: string;
      templateId: string;
      userId: string;
      email: string;
    };
  };
};

const inngest = new Inngest({
  id: "email-service",
  schemas: new EventSchemas().fromRecord<Events>(),
});

// ===========================================
// 2. ローダー関数（Fan-outの起点）
// ===========================================

// 方法1: Cronトリガー
export const weeklyDigestLoader = inngest.createFunction(
  { id: "weekly-digest-loader" },
  { cron: "0 9 * * 1" }, // 毎週月曜9時UTC
  async ({ step, logger }) => {
    logger.info("Starting weekly digest campaign");

    // アクティブなユーザーを取得
    const users = await step.run("fetch-active-users", async () => {
      // 実際にはDBから取得
      return [
        { id: "user_1", email: "alice@example.com" },
        { id: "user_2", email: "bob@example.com" },
        { id: "user_3", email: "charlie@example.com" },
        // ... 数千〜数万ユーザー
      ];
    });

    // キャンペーンIDを生成
    const campaignId = "campaign_" + Date.now();

    // Fan-out: 各ユーザーにイベントを送信
    const events = users.map((user) => ({
      name: "email/weekly-digest.send" as const,
      data: {
        campaignId,
        templateId: "weekly-digest-v2",
        userId: user.id,
        email: user.email,
      },
    }));

    // 一括でイベントを送信
    await step.sendEvent("fan-out-emails", events);

    logger.info("Fan-out complete: " + users.length + " emails queued", {
      campaignId,
    });

    return {
      campaignId,
      usersQueued: users.length,
    };
  }
);

// 方法2: イベントトリガー（手動でバッチを開始する場合）
export const manualDigestLoader = inngest.createFunction(
  { id: "manual-digest-loader" },
  { event: "email/weekly-digest.requested" },
  async ({ event, step, logger }) => {
    const { campaignId, templateId } = event.data;

    logger.info("Starting manual digest campaign", { campaignId });

    // ユーザーをチャンクで取得（大量データ対応）
    let offset = 0;
    const batchSize = 1000;
    let totalQueued = 0;

    while (true) {
      const users = await step.run("fetch-users-batch-" + offset, async () => {
        // 実際にはDBからバッチで取得
        // return db.users.findMany({ skip: offset, take: batchSize });
        if (offset === 0) {
          return [
            { id: "user_1", email: "alice@example.com" },
            { id: "user_2", email: "bob@example.com" },
          ];
        }
        return []; // 2回目以降は空
      });

      if (users.length === 0) break;

      const events = users.map((user) => ({
        name: "email/weekly-digest.send" as const,
        data: {
          campaignId,
          templateId,
          userId: user.id,
          email: user.email,
        },
      }));

      await step.sendEvent("fan-out-batch-" + offset, events);

      totalQueued += users.length;
      offset += batchSize;
    }

    return { campaignId, totalQueued };
  }
);

// ===========================================
// 3. ワーカー関数（各ユーザーを処理）
// ===========================================

export const sendWeeklyDigest = inngest.createFunction(
  {
    id: "send-weekly-digest",
    retries: 5,
    // メールサービスのレート制限に対応
    throttle: {
      limit: 100, // 1分あたり100通
      period: "1m",
    },
    // 並行実行を制限
    concurrency: {
      limit: 50, // 最大50並行
    },
  },
  { event: "email/weekly-digest.send" },
  async ({ event, step, logger }) => {
    const { campaignId, templateId, userId, email } = event.data;

    logger.info("Processing weekly digest", { campaignId, userId });

    // ユーザーのダイジェストコンテンツを生成
    const content = await step.run("generate-content", async () => {
      // 実際にはユーザーのアクティビティからコンテンツを生成
      return {
        newFollowers: 5,
        newLikes: 42,
        topPost: "Amazing post content...",
        recommendations: ["post_1", "post_2", "post_3"],
      };
    });

    // メールを送信
    const result = await step.run("send-email", async () => {
      // 実際にはメールサービスを呼び出す
      // return await emailService.send({ to: email, template: templateId, data: content });
      return {
        messageId: "msg_" + Date.now(),
        status: "sent",
      };
    });

    // 送信結果を記録
    await step.run("record-delivery", async () => {
      // 実際にはDBに記録
      // await db.emailLogs.create({ campaignId, userId, messageId: result.messageId });
      console.log("Email sent to " + email, result);
    });

    return {
      success: true,
      messageId: result.messageId,
    };
  }
);

// ===========================================
// 4. 関数内並列ステップ（Fan-in）
// ===========================================

export const generateReport = inngest.createFunction(
  { id: "generate-monthly-report" },
  { cron: "0 0 1 * *" }, // 毎月1日
  async ({ step, logger }) => {
    logger.info("Generating monthly report");

    // 複数のデータソースから並列でデータを取得
    // awaitせずにPromiseを作成
    const userStats = step.run("fetch-user-stats", async () => {
      return { totalUsers: 10000, activeUsers: 8500, newUsers: 500 };
    });

    const revenueStats = step.run("fetch-revenue-stats", async () => {
      return { mrr: 50000, churn: 2.5, growth: 15 };
    });

    const engagementStats = step.run("fetch-engagement-stats", async () => {
      return { dau: 5000, sessions: 25000, avgDuration: 8.5 };
    });

    // 並列実行し、すべての結果を待つ
    const [users, revenue, engagement] = await Promise.all([
      userStats,
      revenueStats,
      engagementStats,
    ]);

    // レポートを生成
    const report = await step.run("compile-report", async () => {
      return {
        generatedAt: new Date().toISOString(),
        users,
        revenue,
        engagement,
        summary: "Monthly report: " + users.activeUsers + " active users, $" + revenue.mrr + " MRR",
      };
    });

    // レポートを配信
    await step.sendEvent("distribute-report", {
      name: "report/monthly.generated" as any,
      data: { reportId: "report_" + Date.now(), summary: report.summary },
    });

    return report;
  }
);

// ===========================================
// 5. エクスポート
// ===========================================

export const functions = [
  weeklyDigestLoader,
  manualDigestLoader,
  sendWeeklyDigest,
  generateReport,
];
