# 関数設定オプション完全リファレンス

Inngest関数の設定オプションの詳細リファレンス。

## 設定オプション一覧

| オプション    | 型            | デフォルト | 説明                     |
| ------------- | ------------- | ---------- | ------------------------ |
| `id`          | string        | **必須**   | 関数の一意識別子         |
| `name`        | string        | -          | UI表示名                 |
| `retries`     | number        | 4          | 再試行回数（0-20）       |
| `concurrency` | number/object | -          | 並行実行制限             |
| `throttle`    | object        | -          | スループット制限         |
| `rateLimit`   | object        | -          | レート制限（超過は破棄） |
| `debounce`    | object        | -          | デバウンス設定           |
| `idempotency` | string        | -          | 冪等性キー（24時間）     |
| `batchEvents` | object        | -          | バッチ処理設定           |
| `cancelOn`    | array         | -          | キャンセルイベント       |
| `timeouts`    | object        | -          | タイムアウト設定         |
| `onFailure`   | function      | -          | 失敗時コールバック       |

## id（必須）

関数の一意識別子。ケバブケースを推奨。

```typescript
inngest.createFunction(
  { id: "process-user-signup" },
  { event: "app/user.signup" },
  async ({ event, step }) => { /* ... */ }
);
```

## name

UI表示名。指定しない場合はidが使用される。

```typescript
inngest.createFunction(
  {
    id: "process-user-signup",
    name: "Process User Signup"
  },
  { event: "app/user.signup" },
  async ({ event, step }) => { /* ... */ }
);
```

## retries

再試行回数。0-20の範囲で指定。デフォルトは4回。

```typescript
inngest.createFunction(
  {
    id: "handle-form",
    retries: 10
  },
  { event: "api/form.submitted" },
  async ({ event, step, attempt }) => {
    // attemptは0から始まる（0=初回、1=1回目の再試行）
    console.log("現在の試行: " + attempt);
  }
);
```

### 再試行のバックオフ

デフォルトのバックオフスケジュール:
- 1回目: 〜10秒後
- 2回目: 〜30秒後
- 3回目: 〜90秒後
- 4回目: 〜270秒後

各再試行には0-30秒のジッターが追加される。

## concurrency

並行実行制限。数値または詳細設定オブジェクト。

### 基本的な並行制限

```typescript
inngest.createFunction(
  {
    id: "generate-ai-summary",
    concurrency: 10  // 最大10同時実行
  },
  { event: "ai/summary.requested" },
  async ({ event, step }) => { /* ... */ }
);
```

### キーベースの並行制限

```typescript
inngest.createFunction(
  {
    id: "sync-contacts",
    concurrency: {
      limit: 10,
      key: "event.data.userId",  // ユーザーごとに別々の制限
      scope: "fn"  // "fn" | "env" | "account"
    }
  },
  { event: "app/contacts.sync" },
  async ({ event, step }) => { /* ... */ }
);
```

### 複数の並行制限（最大2つ）

```typescript
inngest.createFunction(
  {
    id: "import-data",
    concurrency: [
      { limit: 100 },                        // 全体制限
      { limit: 5, key: "event.data.userId" } // ユーザーごと
    ]
  },
  { event: "data/import.requested" },
  async ({ event, step }) => { /* ... */ }
);
```

### scopeオプション

| scope     | 説明                           |
| --------- | ------------------------------ |
| `fn`      | この関数のみ（デフォルト）     |
| `env`     | この環境のすべての関数         |
| `account` | アカウント全体のすべての関数   |

## throttle

スループット制御。制限を超えた実行はキューに入る。

```typescript
inngest.createFunction(
  {
    id: "ai-enrichment",
    throttle: {
      limit: 60,               // 60回/分
      period: "1m",
      burst: 5,                // バースト許容
      key: "event.data.user_id"
    }
  },
  { event: "ai/summary.requested" },
  async ({ event, step }) => { /* ... */ }
);
```

### パラメータ

| パラメータ | 必須 | 説明 |
| ---------- | ---- | ---- |
| `limit`    | Yes  | 期間あたりの最大実行回数 |
| `period`   | Yes  | 期間（"1m", "1h", "1d"等） |
| `burst`    | No   | バースト許容数 |
| `key`      | No   | キーごとの制限（CEL式） |

## rateLimit

厳格なレート制限。制限を超えた実行は**破棄される**。

```typescript
inngest.createFunction(
  {
    id: "synchronize-data",
    rateLimit: {
      limit: 1,
      period: "4h",
      key: "event.data.company_id"
    }
  },
  { event: "intercom/company.updated" },
  async ({ event, step }) => {
    // company_idごとに4時間に1回のみ実行
  }
);
```

### throttle vs rateLimit

| 特性     | throttle         | rateLimit        |
| -------- | ---------------- | ---------------- |
| 超過時   | キューに入る     | **破棄される**   |
| 使用例   | API制限対応      | 重複防止         |

## debounce

デバウンス設定。連続したイベントを1つにまとめる。

```typescript
inngest.createFunction(
  {
    id: "sync-user-data",
    debounce: {
      period: "5m",
      key: "event.data.userId"
    }
  },
  { event: "user/data.changed" },
  async ({ event, step }) => {
    // 5分以内の同じuserIdのイベントは1回にまとめられる
  }
);
```

### パラメータ

| パラメータ | 必須 | 説明 |
| ---------- | ---- | ---- |
| `period`   | Yes  | デバウンス期間 |
| `key`      | No   | キーごとのデバウンス（CEL式） |
| `timeout`  | No   | 最大待機時間 |

## idempotency

冪等性キー。24時間以内の重複実行を防止。

```typescript
inngest.createFunction(
  {
    id: "send-checkout-email",
    idempotency: "event.data.cartId"  // CEL式
  },
  { event: "cart/checkout.completed" },
  async ({ event, step }) => { /* ... */ }
);
```

**注意**: 冪等性キーは**24時間のみ**有効。

## batchEvents

複数のイベントを1つの関数実行でまとめて処理。

```typescript
inngest.createFunction(
  {
    id: "record-api-calls",
    batchEvents: {
      maxSize: 100,                    // 最大100イベント/バッチ
      timeout: "5s",                   // 5秒待機してバッチを埋める
      key: "event.data.user_id"        // ユーザーごとにグループ化
    }
  },
  { event: "log/api.call" },
  async ({ events, step }) => {        // 'event'ではなく'events'
    const attrs = events.map(evt => ({
      user_id: evt.data.user_id,
      timestamp: evt.ts,
    }));

    await step.run("record-to-db", () => db.bulkWrite(attrs));
    return { recorded: attrs.length };
  }
);
```

### 制限事項

- 最大100イベント/バッチ
- 最大10MB/バッチ
- `idempotency`/`rateLimit`/`cancelOn`と併用不可

## cancelOn

特定のイベントで実行をキャンセル。

```typescript
inngest.createFunction(
  {
    id: "schedule-reminder",
    cancelOn: [
      {
        event: "tasks/reminder.deleted",
        if: "async.data.reminderId == event.data.reminderId",
        timeout: "7d"
      }
    ]
  },
  { event: "tasks/reminder.created" },
  async ({ event, step }) => {
    await step.sleepUntil("wait", event.data.remindAt);
    await step.run("send-push", () => pushNotification(event.data));
  }
);
```

### パラメータ

| パラメータ | 必須 | 説明 |
| ---------- | ---- | ---- |
| `event`    | Yes  | キャンセルをトリガーするイベント名 |
| `if`       | No   | マッチング条件（CEL式） |
| `timeout`  | No   | キャンセル監視期間 |
| `match`    | No   | フィールドマッチング |

**注意**: キャンセルはステップ間でのみ発生する。ステップ実行中はキャンセルされない。

## timeouts

タイムアウト設定。

```typescript
inngest.createFunction(
  {
    id: "time-sensitive",
    timeouts: {
      start: "10s",   // 10秒以内に開始しなければキャンセル
      finish: "30s"   // 開始後30秒以内に完了しなければキャンセル
    }
  },
  { event: "urgent/task" },
  async ({ event, step }) => { /* ... */ }
);
```

### パラメータ

| パラメータ | 説明 |
| ---------- | ---- |
| `start`    | 実行開始までの最大待機時間 |
| `finish`   | 実行完了までの最大時間 |

## onFailure

すべての再試行が失敗した後に呼び出されるコールバック。

```typescript
inngest.createFunction(
  {
    id: "sync-products",
    retries: 5,
    onFailure: async ({ error, event, step }) => {
      await step.run("notify-slack", async () => {
        await slack.postMessage({
          channel: "alerts",
          text: "Sync failed: " + error.message
        });
      });
    }
  },
  { event: "shop/sync.requested" },
  async ({ event, step }) => { /* ... */ }
);
```

### onFailureハンドラーのパラメータ

| パラメータ | 説明 |
| ---------- | ---- |
| `error`    | 発生したエラー |
| `event`    | トリガーイベント |
| `step`     | ステップツール（通常のステップが使用可能） |

## トリガー設定

### イベントトリガー

```typescript
inngest.createFunction(
  { id: "my-function" },
  { event: "app/user.created" },  // イベントトリガー
  async ({ event, step }) => { /* ... */ }
);
```

### Cronトリガー

```typescript
inngest.createFunction(
  { id: "daily-cleanup" },
  { cron: "0 0 * * *" },  // 毎日UTC 0:00
  async ({ step }) => { /* ... */ }
);
```

### タイムゾーン指定

```typescript
inngest.createFunction(
  { id: "weekly-report" },
  { cron: "TZ=Asia/Tokyo 0 9 * * 1" },  // 毎週月曜9時（日本時間）
  async ({ step }) => { /* ... */ }
);
```

### 複数トリガー

```typescript
inngest.createFunction(
  { id: "handle-changes" },
  [
    { event: "db/user.updated" },
    { event: "db/user.deleted" }
  ],
  async ({ event, step }) => { /* ... */ }
);
```

## 型安全なハンドラーパラメータ

```typescript
inngest.createFunction(
  { id: "my-function" },
  { event: "app/user.created" },
  async ({
    event,    // イベントデータ（型付き）
    step,     // ステップツール
    attempt,  // 現在の試行回数（0から開始）
    logger    // ロギングユーティリティ
  }) => {
    logger.info("Processing user", { userId: event.data.userId });
    // ...
  }
);
```
