# 高度なパターン

Inngestの高度な使用パターンのリファレンス。

## Fan-out/Fan-in パターン

大量のアイテムを並列処理するパターン。

### Fan-out with step.sendEvent()

ローダー関数がイベントを発行し、ワーカー関数が個別に処理する。

```typescript
// ローダー関数（Cronトリガー）
export const loadCron = inngest.createFunction(
  { id: "weekly-email-loader" },
  { cron: "0 12 * * 5" },
  async ({ step }) => {
    const users = await step.run("fetch-users", () => fetchUsers());

    // Fan-out: 各ユーザーにイベント送信
    const events = users.map(user => ({
      name: "app/weekly-email.send",
      data: { userId: user.id, email: user.email },
    }));

    await step.sendEvent("fan-out-emails", events);
  }
);

// ワーカー関数（ユーザーごとに並列実行）
export const emailSender = inngest.createFunction(
  { id: "weekly-email-sender" },
  { event: "app/weekly-email.send" },
  async ({ event, step }) => {
    await step.run("send-email", () => sendEmail(event.data.email));
  }
);
```

### メリット

- 各ユーザーの処理が独立
- 1つの失敗が他に影響しない
- 自動的に並列実行される
- 個別に再試行される

### 使用シナリオ

- 大量メール送信
- データ同期
- バッチ処理
- レポート生成

## 関数内の並列ステップ

単一の関数内で複数のステップを並列実行する。

```typescript
inngest.createFunction(
  { id: "post-payment-flow" },
  { event: "stripe/charge.created" },
  async ({ event, step }) => {
    // awaitせずにPromiseを作成
    const sendEmail = step.run("confirmation-email", () =>
      sendEmail(event.data.email)
    );
    const updateUser = step.run("update-user", () =>
      db.updateUserWithCharge(event)
    );

    // 並列実行、結果を集約
    const [emailID, updates] = await Promise.all([sendEmail, updateUser]);
    return { emailID, updates };
  }
);
```

### 注意点

- `await`を付けずにステップを開始
- `Promise.all`で結果を待つ
- 各ステップは独立して再試行される

## Human-in-the-Loop パターン

人間の承認を待つワークフロー。

```typescript
inngest.createFunction(
  { id: "invoice-approval" },
  { event: "invoice/submitted" },
  async ({ event, step }) => {
    // 承認リクエストを送信
    await step.run("send-approval-request", () =>
      sendSlackApprovalRequest(event.data)
    );

    // 承認を待機（最大7日）
    const approval = await step.waitForEvent("wait-for-approval", {
      event: "invoice/approved",
      timeout: "7d",
      match: "data.invoiceId",
    });

    if (approval) {
      await step.run("process-payment", () =>
        processPayment(event.data.invoiceId)
      );
      return { status: "approved" };
    } else {
      await step.run("send-reminder", () =>
        sendReminder(event.data.invoiceId)
      );
      return { status: "timeout" };
    }
  }
);
```

### 承認イベントの送信

```typescript
// 別のAPIエンドポイントから
await inngest.send({
  name: "invoice/approved",
  data: { invoiceId: "inv_123", approvedBy: "user_456" },
});
```

## キャンセル可能な長時間タスク

外部イベントでキャンセル可能なタスク。

```typescript
inngest.createFunction(
  {
    id: "schedule-reminder",
    cancelOn: [
      {
        event: "tasks/reminder.deleted",
        if: "async.data.reminderId == event.data.reminderId",
        timeout: "7d",
      },
    ],
  },
  { event: "tasks/reminder.created" },
  async ({ event, step }) => {
    // リマインダー時刻まで待機
    await step.sleepUntil("wait", event.data.remindAt);

    // 通知を送信
    await step.run("send-push", () => pushNotification(event.data));
  }
);
```

### キャンセルの注意点

- キャンセルは**ステップ間**でのみ発生
- ステップ実行中はキャンセルされない
- キャンセル後は`onFailure`が呼ばれない

## サガパターン

複数サービスにまたがるトランザクション的な処理。

```typescript
inngest.createFunction(
  { id: "order-saga" },
  { event: "order/created" },
  async ({ event, step }) => {
    // 在庫を予約
    const reservation = await step.run("reserve-inventory", () =>
      inventoryService.reserve(event.data.items)
    );

    try {
      // 支払いを処理
      const payment = await step.run("process-payment", () =>
        paymentService.charge(event.data.paymentMethod, event.data.total)
      );

      // 出荷を開始
      await step.run("start-shipping", () =>
        shippingService.ship(event.data.address, reservation.id)
      );

      return { status: "completed", paymentId: payment.id };
    } catch (error) {
      // 補償トランザクション：在庫予約をキャンセル
      await step.run("cancel-reservation", () =>
        inventoryService.cancelReservation(reservation.id)
      );
      throw error;
    }
  }
);
```

## 定期的なヘルスチェック

Cronとイベント待機を組み合わせたパターン。

```typescript
inngest.createFunction(
  { id: "health-monitor" },
  { cron: "*/5 * * * *" },  // 5分ごと
  async ({ step }) => {
    const services = ["api", "database", "cache"];

    for (const service of services) {
      const isHealthy = await step.run("check-" + service, () =>
        checkServiceHealth(service)
      );

      if (!isHealthy) {
        await step.sendEvent("alert", {
          name: "alert/service-unhealthy",
          data: { service, timestamp: Date.now() },
        });
      }
    }
  }
);
```

## リトライ戦略のカスタマイズ

### 即座に失敗させる

```typescript
import { NonRetriableError } from "inngest";

await step.run("validate-input", async () => {
  if (!isValid(data)) {
    throw new NonRetriableError("Invalid input", { cause: validationErrors });
  }
  return data;
});
```

### 特定時間後に再試行

```typescript
import { RetryAfterError } from "inngest";

await step.run("call-api", async () => {
  const response = await api.call();
  if (response.status === 429) {
    throw new RetryAfterError(
      "Rate limited",
      response.headers.get("Retry-After")
    );
  }
  return response.data;
});
```

## バッチ処理パターン

大量データを効率的に処理。

### batchEventsを使用

```typescript
inngest.createFunction(
  {
    id: "bulk-insert",
    batchEvents: {
      maxSize: 100,
      timeout: "5s",
    },
  },
  { event: "data/record.created" },
  async ({ events, step }) => {
    const records = events.map(e => e.data.record);
    await step.run("bulk-insert", () => db.bulkInsert(records));
    return { inserted: records.length };
  }
);
```

### チャンク処理

```typescript
inngest.createFunction(
  { id: "process-large-dataset" },
  { event: "data/import.requested" },
  async ({ event, step }) => {
    const totalItems = event.data.items;
    const chunkSize = 100;

    for (let i = 0; i < totalItems.length; i += chunkSize) {
      const chunk = totalItems.slice(i, i + chunkSize);
      await step.run("process-chunk-" + i, () => processChunk(chunk));
    }
  }
);
```

## TypeScript型定義パターン

### fromRecord

```typescript
import { EventSchemas, Inngest } from "inngest";

type Events = {
  "app/user.created": {
    data: { userId: string; email: string };
  };
  "app/order.placed": {
    data: { orderId: string; amount: number };
  };
};

export const inngest = new Inngest({
  id: "my-app",
  schemas: new EventSchemas().fromRecord<Events>(),
});
```

### fromZod

```typescript
import { EventSchemas, Inngest } from "inngest";
import { z } from "zod";

export const inngest = new Inngest({
  id: "my-app",
  schemas: new EventSchemas().fromZod({
    "app/account.created": {
      data: z.object({
        userId: z.string(),
        email: z.string().email(),
      }),
    },
  }),
});
```

### 型ヘルパー

```typescript
import {
  type GetEvents,
  type GetFunctionInput,
  type GetStepTools,
} from "inngest";

// イベント型を抽出
type Events = GetEvents<typeof inngest>;

// 関数入力型を抽出
type InputArg = GetFunctionInput<typeof inngest, "app/user.created">;

// ステップツール型を抽出
type StepTools = GetStepTools<typeof inngest>;
```

## ミドルウェアパターン

### バリデーションミドルウェア

```typescript
import { validationMiddleware } from "@inngest/middleware-validation";

const inngest = new Inngest({
  id: "my-app",
  middleware: [validationMiddleware()],
  schemas: new EventSchemas().fromZod({
    "example/event": {
      data: z.object({ message: z.string() }),
    },
  }),
});
```

### ロギングミドルウェア

```typescript
inngest.createFunction(
  { id: "my-function" },
  { event: "app/event" },
  async ({ event, step, logger }) => {
    logger.info("Starting function", { eventId: event.id });

    await step.run("my-step", () => {
      logger.warn("Processing step");
    });

    logger.info("Function completed");
  }
);
```

## デプロイメントパターン

### 環境変数

```bash
# 本番環境
INNGEST_EVENT_KEY=your-event-key
INNGEST_SIGNING_KEY=your-signing-key
```

### Vercel統合

1. Vercel Marketplaceでinngest統合をインストール
2. 環境変数が自動設定される
3. デプロイごとに自動同期

### デプロイチェックリスト

- [ ] 全関数に一意の`id`
- [ ] ステップで全副作用をラップ
- [ ] `NonRetriableError`で適切なエラー処理
- [ ] 環境変数の設定
- [ ] `/api/inngest`エンドポイントがアクセス可能

## テストパターン

### @inngest/testを使用

```typescript
import { InngestTestEngine } from "@inngest/test";
import { myFunction } from "./functions";

describe("myFunction", () => {
  const t = new InngestTestEngine({ function: myFunction });

  test("executes successfully", async () => {
    const { result, ctx } = await t.execute({
      events: [
        {
          name: "user.created",
          data: { userId: "123" },
        },
      ],
    });

    expect(result).toEqual("Hello World!");
    expect(ctx.step.run).toHaveBeenCalledWith("my-step", expect.any(Function));
  });
});
```

### モック戦略

```typescript
// 外部サービスをモック
jest.mock("./services/email", () => ({
  sendEmail: jest.fn().mockResolvedValue({ id: "email-123" }),
}));

// ステップの結果を検証
expect(ctx.step.run).toHaveBeenCalledWith(
  "send-email",
  expect.any(Function)
);
```
