# Steps API 完全リファレンス

Steps APIはInngestの耐久実行の核心。各メソッドは**独立して再試行可能**で、結果はメモ化される。

## step.run() - コード実行

再試行可能なコードブロックを実行する。

### 基本使用

```typescript
const result = await step.run("step-id", async () => {
  return fetch("https://api.example.com/data").then(r => r.json());
});
```

### 並列実行

```typescript
// awaitせずにPromiseを作成し、Promise.allで並列実行
const [users, orders] = await Promise.all([
  step.run("fetch-users", () => fetchUsers()),
  step.run("fetch-orders", () => fetchOrders()),
]);
```

### 重要な原則：1ステップ1副作用

各ステップは**単一の副作用**にすべき。複数の副作用を1つのステップに入れると、再試行時に問題が発生する。

```typescript
// ❌ 悪い例：複数の副作用
await step.run("create-alert", async () => {
  const alertId = await createAlert();     // 成功
  await sendAlertToSlack(alertId);         // 失敗→再試行でalertが重複
});

// ✅ 良い例：1ステップ1副作用
const alertId = await step.run("create-alert", () => createAlert());
await step.run("send-to-slack", () => sendAlertToSlack(alertId));
```

## step.sleep() - 遅延

指定期間だけ実行を一時停止する。コンピューティングリソースを消費しない。

### 使用例

```typescript
await step.sleep("wait-30-min", "30m");
await step.sleep("wait-2-hours", "2 hours");
await step.sleep("wait-1-day", "1d");
await step.sleep("wait-5-sec", 5000);  // ミリ秒
```

### 対応フォーマット

- `"30s"` - 秒
- `"30m"` - 分
- `"2h"` / `"2 hours"` - 時間
- `"1d"` - 日
- ミリ秒（数値）

### 制限

- **最大スリープ時間**: 1年間（無料プランは7日間）

## step.sleepUntil() - 特定時刻まで待機

特定の日時まで実行を一時停止する。

```typescript
await step.sleepUntil("wait-new-year", "2025-01-01");
await step.sleepUntil("wait-specific-time", new Date(event.data.remind_at));
```

### パラメータ

- **step-id**: ステップの一意識別子
- **date**: ISO 8601文字列またはDateオブジェクト

## step.waitForEvent() - 外部イベント待機

マッチするイベントが到着するか、タイムアウトまで待機する。

### 基本使用

```typescript
const approval = await step.waitForEvent("wait-for-approval", {
  event: "app/invoice.approved",
  timeout: "7d",
});

if (approval) {
  // イベントが到着した
  await step.run("process-approval", () => processApproval(approval));
} else {
  // タイムアウト
  await step.run("send-reminder", () => sendReminder());
}
```

### イベントマッチング

`match`パラメータでイベント間のフィールドをマッチングする。

```typescript
const approval = await step.waitForEvent("wait-for-approval", {
  event: "app/invoice.approved",
  timeout: "7d",
  match: "data.invoiceId",  // event.data.invoiceId == async.data.invoiceId
});
```

### 条件式マッチング

`if`パラメータでCEL式による条件マッチングを行う。

```typescript
const approval = await step.waitForEvent("wait-for-approval", {
  event: "app/invoice.approved",
  timeout: "7d",
  if: "async.data.amount > 1000",
});
```

### パラメータ

| パラメータ | 必須 | 説明 |
| ---------- | ---- | ---- |
| `event`    | Yes  | 待機するイベント名 |
| `timeout`  | Yes  | タイムアウト期間（最大1年） |
| `match`    | No   | フィールドマッチング |
| `if`       | No   | CEL式による条件 |

### 戻り値

- イベントが到着した場合: イベントオブジェクト
- タイムアウトした場合: `null`

## step.sendEvent() - イベント送信

関数内から安全にイベントを送信する。ステップとしてメモ化されるため、再試行時に重複送信されない。

### 単一イベント

```typescript
await step.sendEvent("trigger-activation", {
  name: "app/user.activated",
  data: { userId: event.data.userId },
});
```

### 複数イベント（Fan-outパターン）

```typescript
await step.sendEvent("trigger-notifications", [
  { name: "notification/email", data: { userId: "123" } },
  { name: "notification/sms", data: { userId: "123" } },
  { name: "notification/push", data: { userId: "123" } },
]);
```

### 使用例：Fan-out

```typescript
// ローダー関数
export const loadCron = inngest.createFunction(
  { id: "weekly-email-loader" },
  { cron: "0 12 * * 5" },
  async ({ step }) => {
    const users = await step.run("fetch-users", () => fetchUsers());

    // Fan-out: 各ユーザーにイベント送信
    const events = users.map(user => ({
      name: "app/weekly-email.send",
      data: { userId: user.id, email: user.email },
    }));

    await step.sendEvent("fan-out-emails", events);
  }
);
```

## step.invoke() - 他の関数を呼び出し

別のInngest関数を呼び出して結果を待つ。

### 基本使用

```typescript
const computeSquare = inngest.createFunction(
  { id: "compute-square" },
  { event: "calculate/square" },
  async ({ event }) => ({ result: event.data.number ** 2 })
);

// 別の関数から呼び出し
const square = await step.invoke("compute-square", {
  function: computeSquare,
  data: { number: 4 },
  timeout: "1h",
});
// square.result === 16
```

### パラメータ

| パラメータ | 必須 | 説明 |
| ---------- | ---- | ---- |
| `function` | Yes  | 呼び出すInngest関数 |
| `data`     | Yes  | 関数に渡すデータ |
| `timeout`  | No   | タイムアウト期間 |

### 使用シナリオ

- 共通処理の再利用
- 複雑なワークフローの分割
- マイクロサービス的な設計

## ステップIDの重要性

各ステップには一意のIDが必要。このIDはメモ化のキーとして使用される。

### ベストプラクティス

```typescript
// ✅ 良い例：説明的なID
await step.run("fetch-user-profile", () => fetchProfile(userId));
await step.run("send-welcome-email", () => sendEmail(email));

// ❌ 悪い例：曖昧なID
await step.run("step1", () => fetchProfile(userId));
await step.run("step2", () => sendEmail(email));
```

### 動的IDの注意点

動的な値をIDに含める場合、その値が変わるとステップが再実行される。

```typescript
// 注意：userIdが変わると新しいステップとして認識される
await step.run("fetch-user-" + userId, () => fetchUser(userId));
```

## ステップ制限

- **最大ステップ数**: 1関数あたり1,000ステップ
- 大量のアイテムを処理する場合はFan-outパターンを使用する

## メモ化の動作

1. ステップが初めて実行される場合:
   - コードが実行される
   - 結果がInngestに保存される

2. ステップが再実行される場合（関数の再試行など）:
   - 保存された結果が即座に返される
   - コードは実行されない

この動作により:
- 成功したステップは再試行時にスキップされる
- 副作用は1回だけ実行される
- 長時間実行ワークフローでも効率的
