# Code Review Sample: Socratic Style

This example demonstrates the question-based code review approach.

## User's Code (React)

```typescript
function UserProfile({ userId }) {
  const [user, setUser] = useState(null);
  const [posts, setPosts] = useState([]);

  useEffect(() => {
    fetch(`/api/users/${userId}`)
      .then(res => res.json())
      .then(data => setUser(data));

    fetch(`/api/users/${userId}/posts`)
      .then(res => res.json())
      .then(data => setPosts(data));
  }, []);

  const fullName = user.firstName + ' ' + user.lastName;

  return (
    <div>
      <h1>{fullName}</h1>
      <PostList posts={posts} />
    </div>
  );
}
```

## BAD Review (Direct Corrections)

"You have several issues:
1. Add `userId` to the dependency array
2. Add null checking for `user`
3. Handle loading and error states
4. These fetches should run in parallel"

## GOOD Review (Socratic Questions)

---

I can see you're building a user profile component that fetches both user data and their posts. Nice job separating those concerns!

I noticed you're handling the async fetches cleanly. A few things I'm curious about:

1. **Line 6-13:** I see the `useEffect` has an empty dependency array `[]`. What would happen if `userId` changed while this component is mounted?

2. **Line 15:** This line assumes `user` has data. What happens on the very first render before the fetch completes?

3. **Lines 7-9 and 11-13:** Both fetches start at the same time, which is great! But what if one succeeds and the other fails? How would the user know something went wrong?

4. **Line 15:** The `fullName` calculation runs on every render. Is that intentional? When might you want to optimize this?

Take a moment to think through each of these. Which one feels most critical to address first?

---

## Key Differences

| Bad Approach | Good Approach |
|-------------|---------------|
| Lists problems directly | Asks questions about problems |
| Tells what to fix | Guides discovery of issues |
| Overwhelming (all at once) | Focused (3-5 questions) |
| No engagement | Invites response |
| No positive feedback | Acknowledges good decisions |
| Prescriptive | Exploratory |
